/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * stest.c
 *
 * Demonstrates:
 *  - the use of sound plugins ( including control() API )
 *
 * Uses:
 *  - libxmm		- LinuX MultiMedia Library
 *  - libm		- C Math Library
 *
 * Usage:
 *  stest
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

#define	PCM_BUFFER_SIZE		4096

#define	TONEFREQ		1000
#define	SAMPLERATE		48000

/*
 * Includes
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>

/*
 * Sound plugin, that should be used
 */

#define SOUND_PLUGIN LIBDIR "Plugins/Sound/oss.so"

/*
 * main()
 */

int main( int argc, char *argv[] )
{
  void			*xmm, *buffer;
  XMM_PluginSound	*pSound;
  XMM_PluginInput	*pInput;
  XMM_SoundFormat	sformat;
  int32_t		samples;
  int			ret, read;

  /* Initialize libxmm */  
  if(( xmm = xmm_Init()) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	return 1;
  }

  /* Open file */
  if(( pInput = xmm_InputOpen( xmm, argv[1], 0 )) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_Exit( xmm );
	return 1;
  }

  /* Open Sound Plugin */
  if(( pSound = xmm_SoundOpen( xmm, SOUND_PLUGIN )) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_InputClose( pInput );
	xmm_Exit( xmm );
	return 1;
  }

  sformat.format = XMM_SOUND_FMT_S16LE;
  sformat.samprate = SAMPLERATE;
  sformat.channels = 1;

  /* Get format */
  ret = xmm_InputControl( pInput, XMM_CTLGET_SFORMAT, 0, &sformat );
  if( ret != XMM_CTLRET_ARG )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_SoundClose( pSound );
	xmm_InputClose( pInput );
	xmm_Exit( xmm );
	return 1;
  }

  /* Check if format supported */
  ret = xmm_SoundControl( pSound, XMM_CTLQUERY_SFORMAT, 0, &sformat );
  if( ret == XMM_CTLRET_TRUE )	fprintf( stdout, "INFO: Sound format supported.\n" );
  else	if( ret == XMM_CTLRET_FALSE )	fprintf( stdout, "INFO: Sound format not supported.\n" );
  else	fprintf( stderr, "WARNING: Unable to query sound format ( %s ).\n", xmm_ErrorString( xmm ));

  /* Start sound output */
  if(( samples = xmm_SoundStart( pSound, &sformat, PCM_BUFFER_SIZE, -1 )) < XMM_RET_OK )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_SoundClose( pSound );
	xmm_InputClose( pInput );
	xmm_Exit( xmm );
	return 1;
  }

  /* output */
  if(( buffer = malloc( samples * 4 )) == NULL )
  {
	fprintf( stderr, "ERROR: Unable to allocate buffer.\n" );
	xmm_SoundClose( pSound );
	xmm_InputClose( pInput );
	xmm_Exit( xmm );
	return 1;
  }

  do
  {
	read = xmm_InputAudioRead( pInput, 0, buffer, samples );
	if( read > XMM_RET_OK )	xmm_SoundWrite( pSound, buffer, read );

  } while( read != XMM_RET_EOS );

  free( buffer );

  /* Close sound output */
  xmm_SoundClose( pSound );

  /* Close input */
  xmm_InputClose( pInput );

  /* Free libxmm */
  xmm_Exit( xmm );

  return 0;
}
