/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * gtest.c
 *
 * Demonstrates:
 *  - the use of graph plugins ( and scaling )
 *  - demonstrates how to create overlays ( packed and planer YCrCb )
 *  - handling of events from the graph plugin
 *  - writing directly into the graph surface
 *
 * Uses:
 *  - libxmm		- LinuX MultiMedia Library
 *  - libjpeg		- JPEG library from the Independet JPEG Group
 *
 * Usage:
 *  gtest <filename>	- filename has to be a JPEG image
 *
 *  - 'q' to quit
 *  - '1' for original size
 *  - '2' for double size
 *  - '3' for fullscreen
 *  - 't' set window title
 *  - 'r' restore window title to plugin default
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to output data directly in YCrCb ( packed ) */
#ifndef YUY2_OUTPUT
#undef	YUY2_OUTPUT
#endif

/* Change 'undef' in 'define' to output data directly in YCrCb ( planar ) */
#ifndef YV12_OUTPUT
#undef	YV12_OUTPUT
#endif

/* packed or planar */
#if defined( YUY2_OUTPUT ) && defined( YV12_OUTPUT )
#error "YUY2_OUTPUT or YV12_OUTPUT may be defined. never both..."
#endif

/* Change 'undef' in 'define' to resize to fullscreen after some seconds */
#ifndef AUTOMATIC_FULLSCREEN
#undef AUTOMATIC_FULLSCREEN
#endif

/* Change 'undef' in 'define' to write directly into the graph buffer */
#ifndef GRAPH_SURFACE
#define GRAPH_SURFACE
#endif

/* Bits per pixel in RGB mode, 15, 16, 24 or 32 */
#define	RGB_BITS	24

/*
 * Includes
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <setjmp.h>
#include <jpeglib.h>

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>

/*
 * Graph plugin, that should be used
 */

#define GRAPH_PLUGIN LIBDIR "Plugins/Graph/x11.so"

/*
 * Load JPEG image, return 0 on success
 * image size is stored in width and height, data[0] contains RGB, data[0,1,2] YCrCb data
 */

int LoadJPEG( char *filename, uint16_t *width, uint16_t *height, uint8_t *data[], uint32_t type, int flag, int stride[] );

/*
 * Event handler
 */

static int EventCB( void *priv, XMM_Event *event );

/*
 * main()
 */

int main( int argc, char *argv[] )
{
  void			*xmm;
  XMM_PluginGraph	*pGraph;
  XMM_ControlScale	scale;
  XMM_ControlRect	rect;
  XMM_Event		event;
  uint16_t		width, height;
  uint8_t		*image[3];
  int			ret, loop = 1, cnt;
#ifdef GRAPH_SURFACE
  XMM_ControlSurface	surface;
#endif

  /* Initialize libxmm */  
  if(( xmm = xmm_Init()) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	return 1;
  }

  /* Set event handler */
  xmm_SetEventCB( xmm, EventCB, xmm, 0 );

  /* Open Graph Plugin */
  if(( pGraph = xmm_GraphOpen( xmm, GRAPH_PLUGIN )) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_Exit( xmm );
	return 1;
  }

#ifdef GRAPH_SURFACE
  /* Load image - Get only width and height */
  if( LoadJPEG( argv[1], &width, &height, image, 0, 1, NULL ) != 0 )
  {
	return -1 * XMM_RET_ERROR;
  }
#else
  /* Load image ( LoadJPEG will allocate memory for the image )  */
  if( LoadJPEG( argv[1], &width, &height, image, 0, 0, NULL ) != 0 )
  {
	return -1 * XMM_RET_ERROR;
  }
#endif

  /* Init graph output */
  if( xmm_GraphStart( pGraph, width, height,
#ifdef YUY2_OUTPUT
	xmmFOURCC( 'Y','U','Y','2' ), 0 ) != XMM_RET_OK )
#elif defined( YV12_OUTPUT )
	xmmFOURCC( 'Y','V','1','2' ), 0 ) != XMM_RET_OK )
#else
	xmmFOURCC( 'R','G','B', RGB_BITS ), 0 ) != XMM_RET_OK )
#endif
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	return -1 * XMM_RET_ERROR;
  }

#ifdef GRAPH_SURFACE
  /* Get surface type and surface */
  if( xmm_GraphControl( pGraph, XMM_CTLGET_SURFACE, 0, &surface ) == XMM_CTLRET_ARG )
  {
	image[0] = surface.data[0];
	image[1] = surface.data[1];
	image[2] = surface.data[2];
  }
  else	surface.fourcc = 0;

  if( surface.fourcc == 0 )	fprintf( stdout, "ERROR: Unable to get surface or surface type\n" );
  else	fprintf( stdout, "INFO: Surface [%p,%p,%p]. Type %s (0x%x)\n", image[0], image[1], image[2], xmm_FOURCC_string( surface.fourcc ), surface.fourcc );

  /* Load image - into graph surface */
  if( LoadJPEG( argv[1], &width, &height, image, surface.fourcc, 0, surface.stride ) != 0 )
  {
	return -1 * XMM_RET_ERROR;
  }
#endif

  /* Set position */
  rect.x = 0;
  rect.y = 0;
  xmm_GraphControl( pGraph, XMM_CTLSET_POSITION, 0, &rect );

  /* Draw image */
  xmm_GraphDraw( pGraph, image, NULL, 0, 0, width, height, 0 );
  xmm_GraphBlit( pGraph );

  /* Event loop, to get events Blit() has to be called regularly */
  for( loop = 1, cnt = 0; loop && ( cnt < 20 ); cnt++ )
  {
    xmmSYS_usleep( 500000 );

#ifdef AUTOMATIC_FULLSCREEN
    if( cnt == 10 )
    {
	    event.key.type = XMM_EVENT_KEY;
	    event.key.state = XMM_KEY_RELEASED;
	    event.key.sym = '3';
	    event.key.mod = 0;
    }
    else
#endif
    /* Check for event */
    if( xmm_PollEvent( xmm, 0, &event ) <= 0 )	continue;

    /* We have an event */
    switch( event.type )
    {
	case XMM_EVENT_KEY:
		{
		    XMM_Event_Key *key = (XMM_Event_Key *) &event;

#ifdef DEBUG
		    printf( "Key event: scancode %i ASCII %i ( '%c' ) state %i mode = %x\n", key->scode, key->sym, key->sym, key->state, key->mod );
#endif

		    if(( key->state == XMM_KEY_RELEASED ) &&
			    (( key->sym >= '1' ) && ( key->sym <= '3' )))
		    {
			xmm_logging( 1, "gtest: Scaling...\n" );

			/* fill XMM_ControlScale structure */
#if 0
			scale.width = width;
			scale.height = height;
#else
			scale.width = 0;
			scale.height = 0;
#endif
			scale.flags = 0;

			if( key->sym == '1' )	scale.flags = XMM_GRAPH_RESIZE_ORIG;
			if( key->sym == '2' )	scale.flags = XMM_GRAPH_RESIZE_DOUBLE;
			if( key->sym == '3' )	scale.flags = XMM_GRAPH_RESIZE_FULLSCREEN;

			/* Resize using graph plugin */
			ret = xmm_GraphControl( pGraph, XMM_CTLSET_SCALE, 0, &scale );
			if( ret != XMM_CTLRET_TRUE )
			{
			    fprintf( stderr, "Error resizing output: %s\n", xmm_ErrorString( xmm ));
			}

			/* Re-Draw image, after scaling this has to be done */
			xmm_GraphDraw( pGraph, image, NULL, 0, 0, width, height, 0 );
			xmm_GraphBlit( pGraph );
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 'q' ))
		    {
			xmm_logging( 1, "gtest: Quitting...\n" );
			loop = 0;
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 't' ))
		    {
			xmm_logging( 1, "gtest: Setting window title...\n" );
			xmm_GraphControl( pGraph, XMM_CTLSET_TITLE, 0, "gtest" );
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 'r' ))
		    {
			xmm_logging( 1, "gtest: Restoring window title...\n" );
			xmm_GraphControl( pGraph, XMM_CTLSET_TITLE, 0, NULL );
		    }

		    break;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event.type );
		break;  
    }
  }

  /* Close graph output */
  xmm_GraphClose( pGraph );

  /* Free libxmm */
  xmm_Exit( xmm );

  return 0;
}

/*
 * Event handler
 * This probably never will be called. It only demonstrates the
 * TOC and Authorization events.
 * We poll for key events in main(), as we need the data defined
 * there for scaling...
 */

static int EventCB( void *priv, XMM_Event *event )
{

  switch( event->type )
  {
	/* Select from Table of Contents */
	case XMM_EVENT_TOC:
		{
		    int i;
		    XMM_Event_TOC *etoc = (XMM_Event_TOC *) event;

		    fprintf( stdout, "\nTable of Contents:\n" );
		    for( i = 0; i < etoc->entries; i++ )	fprintf( stdout, "\t%2i: %s\n", i, etoc->entry[i] );
		
		    fprintf( stdout, "Please make your selection: " );
		    scanf( "%i", &etoc->selected );
		    return 0;
		}

	/* Authorization requiered */
	case XMM_EVENT_AUTH:
		{
		    char *passwd;
		    XMM_Event_Auth *eauth = (XMM_Event_Auth *) event;

		    fprintf( stdout, "\nAuthorization required:\n" );
		    fprintf( stdout, "User: " );
		    scanf( "%s", eauth->user );
		    passwd = getpass( "Password:" );
		    if( passwd == NULL )	break;
		    strcpy( eauth->passwd, passwd );
		    return 0;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event->type );
		break;  
  }

  return XMM_RET_ERROR;
}

/*
 * JPEG stuff
 * Below is no interesting code, related to LinuX MultiMedia Project
 */

typedef struct
{
	struct jpeg_error_mgr pub;
	jmp_buf setjmp_buffer;
} JPEGerror;

void ErrorHandler( j_common_ptr cinfo )
{
  JPEGerror *error = (JPEGerror *) cinfo->err;

  (*cinfo->err->output_message)(cinfo);

  longjmp( error->setjmp_buffer, 1);
}

int LoadJPEG( char *filename, uint16_t *width, uint16_t *height, uint8_t *data[], uint32_t type, int flag, int stride[] )
{
  struct jpeg_decompress_struct cinfo;
  JPEGerror	jerr;
  JSAMPARRAY	buffer;
  uint8_t	*ptr;
  FILE		*stream;
  int		i, pixelsize;
#if defined( YUY2_OUTPUT ) || defined( YV12_OUTPUT )
  int		j;
#endif
#ifdef YV12_OUTPUT
  uint8_t	*uptr, *vptr;
#endif

  /* Open file */
  if(( stream = fopen( filename, "r" )) == NULL )
  {
	fprintf( stderr, "ERROR: Unable to open %s\n", filename );
	return -1;
  }

  cinfo.err = jpeg_std_error( &jerr.pub );
  jerr.pub.error_exit = ErrorHandler;

  /* JPEG library used setjmp for error handling */
  if( setjmp( jerr.setjmp_buffer ))
  {
	/* * * * Here we get when an error occurs * * */
	jpeg_destroy_decompress( &cinfo );
	fclose( stream );

	fprintf( stderr, "ERROR! Error sent by JPEG lib.\n" );
	return -1;
  }

  /* Initialize JPEG stuff */
  jpeg_create_decompress( &cinfo );
  jpeg_stdio_src( &cinfo, stream );
  jpeg_read_header( &cinfo, TRUE );

#if defined( YUY2_OUTPUT ) || defined( YV12_OUTPUT )
  cinfo.out_color_space = JCS_YCbCr;
#endif

  jpeg_start_decompress( &cinfo );

  /* Get width and height */
  *width = cinfo.output_width;
  *height = cinfo.output_height;

  /* Only width and height needed - don't decode data */
  if( flag == 1 )
  {
	/* De-Initialize JPEG stuff */
	jpeg_destroy_decompress( &cinfo );

	/* Close file */
	fclose( stream );

	return 0;
  }

  /* Allocate buffer for decoding */
  buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr) &cinfo, JPOOL_IMAGE, cinfo.output_width * cinfo.output_components, 2 );

  /* Only accept 24 bit images / no greyscale */
  if( cinfo.output_components != 3 )
  {
	fprintf( stderr, "ERROR! Only images with 3 components supported.\n");
	return -1;
  }

#ifdef GRAPH_SURFACE
#ifdef YUY2_OUTPUT
  if( type == XMM_GRAPH_FMT_YUY2 );
  else
#elif defined( YV12_OUTPUT )
  if( type == XMM_GRAPH_FMT_YV12 );
  else
#endif
  if( type == 0 )
#endif

  /* Allocate memory for picture */
  if(( data[0] = malloc( cinfo.output_width * cinfo.output_height * 4 )) == NULL )
  {
	fprintf( stderr, "ERROR! Cannot allocate image.\n");
	return -1;
  }

#if defined( YV12_OUTPUT )
#ifdef GRAPH_SURFACE
  if( type == 0 )
  {
#endif
	data[1] = data[0] + cinfo.output_width * cinfo.output_height;
	data[2] = data[1] + cinfo.output_width * cinfo.output_height / 4;
#ifdef GRAPH_SURFACE
  }
#endif

  uptr = data[1];
  vptr = data[2];
#endif

#if RGB_BITS == 15
  pixelsize = 2;
#elif RGB_BITS == 16
  pixelsize = 2;
#elif RGB_BITS == 24
  pixelsize = 3;
#else
  pixelsize = 4;
#endif

  /* Copy data to buffer */
  for( ptr = data[0]; cinfo.output_scanline < cinfo.output_height; ) 
  {
#ifdef YUY2_OUTPUT
	jpeg_read_scanlines( &cinfo, buffer, 1 );

	for( i = 0, j = 0; i < cinfo.output_width; i += 2, j += 6 )
	{
		*ptr++ = buffer[0][j + 0];
		*ptr++ = ( buffer[0][j + 1] + buffer[0][j + 4] ) >> 1;
		*ptr++ = buffer[0][j + 3];
		*ptr++ = ( buffer[0][j + 2] + buffer[0][j + 5] ) >> 1;
	}
#elif defined( YV12_OUTPUT )
	jpeg_read_scanlines( &cinfo, buffer, 2 );

	for( i = 0, j = 0; i < cinfo.output_width; i += 2, j += 6 )
	{
		*ptr++ = buffer[0][j + 0];
		*ptr++ = buffer[0][j + 3];
		*ptr++ = buffer[1][j + 0];
		*ptr++ = buffer[1][j + 3];

		/* Use only CrCb values from even lines */
		*uptr++ = ( buffer[0][j + 1] + buffer[0][j + 4] ) >> 1;
		*vptr++ = ( buffer[0][j + 2] + buffer[0][j + 5] ) >> 1;
	}
#else
	jpeg_read_scanlines( &cinfo, buffer, 1 );

	for( i = 0; i < *width; i++ )
	{
#if RGB_BITS == 15

#define	rgb2b15( red, green, blue ) \
  (uint16_t)((( (uint8_t)red & 0xF8 ) << 7 ) | (( (uint8_t)green & 0xF8 ) << 2 ) | (( (uint8_t)blue & 0xF8 ) >> 3 ))
	    *((short *)ptr)++ = rgb2b15( buffer[0][i * 3 + 2], buffer[0][i * 3 + 1], buffer[0][i * 3 + 0] );

#elif RGB_BITS == 16

#define rgb2b16( red, green, blue ) \
  (uint16_t)((( (uint8_t)red & 0xF8 ) << 8 ) | (( (uint8_t)green & 0xF8 ) << 3 ) | (( (uint8_t)blue & 0xF8 ) >> 3 ))
	    *((short *)ptr)++ = rgb2b16( buffer[0][i * 3 + 2], buffer[0][i * 3 + 1], buffer[0][i * 3 + 0] );
#else
	    *ptr++ = buffer[0][i * 3 + 0];
	    *ptr++ = buffer[0][i * 3 + 1];
	    *ptr++ = buffer[0][i * 3 + 2];
#if RGB_BITS == 32
	    *ptr++ = 0;
#endif /* RGB_BITS == 32 */

#endif /* RGB_BITS == 15 */

	}

	if( stride )	ptr += ( stride[0] - *width * pixelsize );
#endif
  }

  /* De-Initialize JPEG stuff */
  jpeg_finish_decompress( &cinfo );
  jpeg_destroy_decompress( &cinfo );

  /* Close file */
  fclose( stream );

  return 0;
}
