/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * stest.c
 *
 * Demonstrates:
 *  - the use of sound plugins ( including control() API )
 *
 * Uses:
 *  - libxmm		- LinuX MultiMedia Library
 *  - libm		- C Math Library
 *
 * Usage:
 *  stest
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

#define	PCM_BUFFER_SIZE		4096

#define	TONEFREQ		1000
#define	SAMPLERATE		48000

/*
 * Includes
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>

/*
 * Sound plugin, that should be used
 */

#define SOUND_PLUGIN LIBDIR "Plugins/Sound/wave.so"

/*
 * Prototypes
 */

void tone( uint8_t *data, int samples, int freq, int samplerate );

/*
 * main()
 */

int main( int argc, char *argv[] )
{
  void			*xmm, *buffer;
  XMM_PluginSound	*pSound;
  XMM_SoundFormat	sformat;
  int32_t		samples;
  int			ret, i;

  /* Initialize libxmm */  
  if(( xmm = xmm_Init()) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	return 1;
  }

  /* Open Sound Plugin */
  if(( pSound = xmm_SoundOpen( xmm, SOUND_PLUGIN )) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_Exit( xmm );
	return 1;
  }

  sformat.format = XMM_SOUND_FMT_S16LE;
  sformat.samprate = SAMPLERATE;
  sformat.channels = 1;

  /* Check if format supported */
  ret = xmm_SoundControl( pSound, XMM_CTLQUERY_SFORMAT, 0, &sformat );
  if( ret == XMM_CTLRET_TRUE )	fprintf( stdout, "INFO: Sound format supported.\n" );
  else	if( ret == XMM_CTLRET_FALSE )	fprintf( stdout, "INFO: Sound format not supported.\n" );
  else	fprintf( stderr, "WARNING: Unable to query sound format.\n" );

SOURCE_INFO
  /* Start sound output */
  if(( samples = xmm_SoundStart( pSound, &sformat, PCM_BUFFER_SIZE, -1 )) < XMM_RET_OK )
  {
SOURCE_INFO
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
SOURCE_INFO
	xmm_SoundClose( pSound );
SOURCE_INFO
	xmm_Exit( xmm );
SOURCE_INFO
	return 1;
  }

  /* output */
  if(( buffer = malloc( samples * 2 )) == NULL )
  {
	fprintf( stderr, "ERROR: Unable to allocate buffer.\n" );
	xmm_SoundClose( pSound );
	xmm_Exit( xmm );
	return 1;
  }

  for( i = 0; i < 40; i++ )
  {
	tone( buffer, samples, TONEFREQ, SAMPLERATE );
	xmm_SoundWrite( pSound, buffer, samples );

if( i == 0 )
{
  FILE *stream = fopen( "wave.xpm", "w" );
  char	xpm[256][samples/16+1];
  int	x;

  for( x = 0; x < 256; x++ )
  {
	memset( xpm[x], ' ', samples/16 );
	xpm[x][samples/16] = '\0';
  }

  for( x = 0; x < samples; x+=16 )
	xpm[(((int16_t *)buffer)[x/16] / 256 ) + 128][x/16] = '.';

  fprintf( stream, "/* XPM */\nstatic char * Jlock_xpm[] = {\n\"256 %i 2 1\",\n\" \ts black\tc black\",\n\".\tc white\",\n", samples/16 );

  for( x = 0; x < 256; x++ )
	fprintf( stream, "\"%s\",\n", xpm[x] );

  fclose( stream );
}

  }

  free( buffer );

  /* Close sound output */
  xmm_SoundClose( pSound );

  /* Free libxmm */
  xmm_Exit( xmm );

  return 0;
}

/*
 * Below is no interesting code, related to LinuX MultiMedia Project
 */

void tone( uint8_t *data, int samples, int freq, int samplerate )
{
  int		i;
  double	scale;
  static int	x = 0;

  scale = ( 2 * 3.14159265359 * freq ) / samplerate;

  for( i = 0; i < samples; i++ )
	((int16_t *)data)[i] = (int16_t) (( 1 << 15 ) - 1 ) * sin( scale * x++ );

  if( x > samplerate )	x -= samplerate;
}
