/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_config_h
#define INC_libxmm_config_h

/** @file config.h
 * Linux MultiMedia Project: library configuration / controllling
 */

#include <inttypes.h>
#include <libxmm/util/list.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_CFG_CODEC_ Codec types
 *
 * Codec types
 *
 * @{
 */

	/** ACM codec */
#define	XMM_CFG_CODEC_ACM		0x0010
	/** VFW codec */
#define	XMM_CFG_CODEC_VFW		0x0020
	/** DirectShow codec */
#define	XMM_CFG_CODEC_DSHOW		0x0040

/** @} */

/*
 * Types
 */

	/**
	 * Codec property.
	 * This struct contains on codecs.
	 *
	 * \note This information is stored in a special codec
	 * configuration file.
	 */
typedef struct XMM_AcCodecProperties_s
{
	/** Codec description. */
	char		info[XMM_MAXPATHLEN];
	/** Codec DLL. */
	char		file[XMM_MAXPATHLEN];
	/** Codec status. */
	char		stat[XMM_MAXPATHLEN];
	/** Codec flags. */
	uint32_t	flags;
	/** Codec output formats. ( Seperated by spaces in codec config file ) */
	char		ofmt[XMM_MAXPATHLEN];

} XMM_AcCodecProperties;

/*
 * Prototypes
 */

	/**
	 * Get codec properties.
	 * Get properties of a codec, that is able to handle the format.
	 *
	 * \param xmm XMM object
	 * \param format format ID of data
	 * \param type codec type. 'Or'ed values from the XMM_CFG_CODEC group.
	 * \param cp pointer to a #XMM_AcCodecProperties object.
	 *
	 * \return Upon successful completion the #XMM_RET_OK is returned.
	 *  Otherwise an error code ( negative value ).
	 */
int	xmm_AcCodecFile( void *xmm, uint32_t format, int type, XMM_AcCodecProperties *cp );

	/**
	 * Set/Get sound plugin name.
	 *
	 * \param xmm xmm object
	 * \param name Name of new sound plugin
	 *
	 * \return Upon successful completion a pointer to the current
	 * plugin name is returned. Otherwise NULL.
	 */
const char	*xmm_AcSoundPluginName( void *xmm, char *name );

	/**
	 * Set/Get graph plugin name.
	 *
	 * \param xmm xmm object
	 * \param name Name of new graph plugin
	 *
	 * \return Upon successful completion a pointer to the current
	 * plugin name is returned. Otherwise NULL.
	 */
const char	*xmm_AcGraphPluginName( void *xmm, char *name );

	/**
	 * Set/Get output plugin name.
	 *
	 * \param xmm xmm object
	 * \param name Name of new output plugin
	 *
	 * \return Upon successful completion a pointer to the current
	 * plugin name is returned. Otherwise NULL.
	 *
	 * \note Only if graph and sound plugin are both "",
	 * the complete output plugin will be used.
	 */
const char	*xmm_AcOutputPluginName( void *xmm, char *name );

	/**
	 * Set/Get graph conversion filter plugin name.
	 *
	 * \param xmm xmm object
	 * \param name Name of new graph conversion filter plugin
	 *
	 * \return Upon successful completion a pointer to the current
	 * plugin name is returned. Otherwise NULL.
	 */
const char	*xmm_AcGConvPluginName( void *xmm, char *name );

	/**
	 * Set/Get name of output file
	 *
	 * \param xmm xmm object
	 * \param name file name
	 * \param ext file extension
	 *
	 * \return Upon successful completion a pointer to the current
	 * output file name is returned. Otherwise NULL.
	 *
	 * \note name may be NULL. In this case extension will be appended
	 * to the filename set by a prior call of xmm_AcOutputFilename()
	 * or if the output file name was "" ( empty string ) the input
	 * file name will be used.
	 * If name is not NULL, the filename will be set.
	 * If both are NULL, the call will not modify the output filename.
	 */
const char	*xmm_AcOutputFilename( void *xmm, char *name, char *ext );

#ifdef __cplusplus
}
#endif

#endif
