/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_event_h
#define INC_libxmm_event_h

/** @file event.h
 * Linux MultiMedia Project: event subsystem
 */

#include <inttypes.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_EVENT_ Event types
 *
 * Event types
 *
 * @{
 */

	/** Key event. Key events are caused by graph output, on keyboard event */
#define	XMM_EVENT_KEY			0x0001
	/** Table Of Contents. User has to make a selection. */
#define	XMM_EVENT_TOC			0x0002
	/** Authorization. User and password required */
#define	XMM_EVENT_AUTH			0x0004

/** @} */

/*
 * Keyboard event
 */

/**
 * Keyboard Event.
 * XMM_Event_Key is a member of the #XMM_Event union.
 * \note It contains valid information if type is #XMM_EVENT_KEY
 */

typedef struct xmm_event_key_s
{
    uint16_t		type;		/**< XMM_EVENT_KEY */

    uint8_t		scode;		/**< hardware specific scan code */
    uint8_t		state;		/**< #MM_KEY_PRESSED or #XMM_KEY_RELEASED */

    uint16_t		sym;		/**< XMM virtual symbol code */
    uint16_t		mod;		/**< Key modifiers, XMM_KEYMOD_xxx */

} XMM_Event_Key;

/**
 * TOC event.
 * XMM_Event_TOC is a member of the #XMM_Event union.
 * Events of this type will never be queued, as you have to make choice.
 * \note It contains valid information if type is #XMM_EVENT_TOC
 */

typedef struct xmm_event_toc_s
{
    uint16_t		type;		/**< XMM_EVENT_TOC (R) */

    int			entries;	/**< Number of entries (R) */
    char		**entry;	/**< Entries (R) */
    int			selected;	/**< Select entry (W) */

} XMM_Event_TOC;

/**
 * Auth event.
 * XMM_Event_Auth is a member of the #XMM_Event union.
 * Event of this type will never be queued, as you have to enter
 * a name/user/login and a password.
 * \note It contains valid information if type is #XMM_EVENT_AUTH
 */

typedef struct xmm_event_auth_s
{
    uint16_t		type;		/**< XMM_EVENT_AUTH (R) */

    char		*url;		/**< URL/description for the needed password (R) */
    char		*user;		/**< User (W) */
    char		*passwd;	/**< Password (W) */
    int			maxlen;		/**< Maximum length for user and passwd (R) */

} XMM_Event_Auth;

/**
 * General event type
 * The XMM_Event untion is the core of all events.
 * It is simply a union of all event structures. The first
 * member of each event structure is type. Using this member
 * it can be decided which union member has to be used.
 */

typedef union xmm_event_u
{
    uint16_t		type;		/**< XMM_EVENT_AUTH (R) */

    XMM_Event_Key	key;		/**< Key event */
    XMM_Event_TOC	toc;		/**< Table of contents event */
    XMM_Event_Auth	auth;		/**< Authorization event */
} XMM_Event;

/*
 * Prototypes
 */

	/**
	 * Initialize event handling.
	 * This call will initialize the event handling subsystem.
	 * ( e.g. the keymap will be read ).
	 *
	 * \param xmm pointer to XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_InitEvent( void *_xmm );

	/**
	 * Stop/Free event handling.
	 * This call will stop the event handling subsystem and free
	 * the used resources.
	 *
	 * \param xmm pointer to XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_ExitEvent( void *_xmm );

	/**
	 * Push event.
	 * This call will add an event to the event queue or will cal the
	 * event handler if event is not queueable.
	 *
	 * \param xmm pointer to XMM object
	 * \param event pointer to general event union
	 * \param queue if queue == 1 event will be queued, else the
	 * event handler will be called.
	 *
	 * \return Upon successful completion XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 *
	 * \note For key events, the members type, scode and state have
	 * to be set. sym and mode will be set by xmm_PushEvent()
	 * according to the keymap.
	 */
int	xmm_PushEvent( void *_xmm, XMM_Event *event, int queue );

	/**
	 * Poll event.
	 * This call will get an event from the event queue.
	 *
	 * \param xmm pointer to XMM object
	 * \param mask Event mask. NOT implemented, yet.
	 * \param event pointer to general event union
	 *
	 * \return Upon successful completion 1 is returned, if event was
	 * available, or 0 if none available.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_PollEvent( void *_xmm, int mask, XMM_Event *event );

	/**
	 * Set event handler.
	 * This call will set the event handler of your application.
	 *
	 * \param xmm pointer to XMM object
	 * \param eventCB event handler
	 * \param priv_data pointer, that will be passed to the event handler
	 * \param onlyCB if onlyCB == 1 event will never be called,
	 * instead the event handler will be called always.
	 *
	 * \return Upon successful completion XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 *
	 * \note Setting the event callback is required. If not set, the non
	 * queueable events will be lost.
	 * e.g. TOC and Authorization request will be lost and module causing
	 * the event will abort with an error message.
	 */
int	xmm_SetEventCB( void *_xmm, int (*eventCB)( void *priv, XMM_Event *event ), void *priv_data, int onlyCB );

#ifdef __cplusplus
}
#endif

#endif
