/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpcodeca_h
#define INC_libxmm_lpcodeca_h

/** @file lpcodeca.h
 * Linux MultiMedia Project API: Audio Codec Layer ( Plugin )
 *
 * Codec Plugins are used to decode (compressed) data and encode (uncompressed)
 * data. The codec plugins have no knowledge of the context/origin of the data
 * they work on.
 */

#include <libxmm/plugin.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_CODEC_MODE_ Codec mode
 *
 * Codec mode
 *
 * @{
 */

#ifndef	_XMM_CODEC_MODES_
#define	_XMM_CODEC_MODES_

	/** Open plugin for decoding */
#define	XMM_CODEC_MODE_DECODE		0x01
	/** Open plugin for encoding */
#define	XMM_CODEC_MODE_ENCODE		0x02
	/** Only query plugin */
#define	XMM_CODEC_MODE_QUERY		0x80

#endif

/** @} */

/** @defgroup XMM_CODEC_ACF_ Audio Codec Capabilities / Flags
 *
 * Audio Codec Capabilities / Flags
 *
 * @{
 */

	/** Plugin is able to decode data */
#define	XMM_CODEC_ACF_DECODE		0x0010
	/** Plugin is able to encode data */
#define	XMM_CODEC_ACF_ENCODE		0x0020

/** @} */

/**
 * Audio format.
 * Contains information about the audio data
 */

typedef struct xmm_AudioFormat_s
{
    uint16_t			formatID;	/**< Format ID */
    uint16_t			channels;	/**< channels */
    uint32_t			samprate;	/**< Sample rate */
    int32_t			bitrate;	/**< Bitrate ( bits / sec ) */
    uint32_t			format;		/**< Sample format ( XMM ID ) */
    uint16_t			blockSize;	/**< block size of data */
    uint16_t			extraSize;	/**< extra data size */
    char			desc[64];	/**< description */
} XMM_AudioFormat;

/**
 * Audio codec info.
 * Information on the codec supported by a plugin. Returned by xmm_CodecInfo()
 */

typedef struct xmm_CodecAudioInfo_s
{
    uint32_t			caps;		/**< codec capabilities */
    char			desc[64];	/**< codec description */
    char			file[64];	/**< codec plugin file name */

    int				aFormatNum;	/**< number of supported audio formats */
    XMM_AudioFormat		*aFormat;	/**< supported audio formats */

} XMM_CodecAudioInfo;

/**
 * Audio Codec Plugin.
 * This structure / type represents a video codec plugin.
 */

typedef struct xmm_PluginCodecAudio_s	XMM_PluginCodecAudio;

struct xmm_PluginCodecAudio_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize / Open codec.
     * Check if audio format supported by codec / Open codec.
     *
     * \param xmm XMM object
     * \param mode codec mode: Definitions from the #XMM_CODEC_MODE_ group.
     * #XMM_CODEC_MODE_DECODE or #XMM_CODEC_MODE_ENCODE. Optionally
     * #XMM_CODEC_MODE_QUERY is possible.
     * \param af audio format. Data format.
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     * If #XMM_CODEC_MODE_QUERY was used in mode, NULL will be returned if
     * codec supports desired format and mode, otherwise an error code
     * ( negative value ).
     */
    XMM_PluginCodecAudio *	(*Open)( void *xmm, int mode, XMM_AudioFormat *af );
    /** xmm_CodecAudioClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginCodecAudio *codec );
    /** xmm_CodecAudioControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg );
    /** xmm_CodecAudioDecode() is a wrapper for this member */
    int				(*Decode)( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags );
    /** xmm_CodecAudioEncode() is a wrapper for this member */
    int				(*Encode)( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags );
    /** xmm_CodecAudioInfo() is a wrapper for this member */
    XMM_CodecAudioInfo *	(*Info)( void *xmm );
};

/*
 * Prototypes
 */

	/**
	 * Open codec.
	 * Find plugin, that is able to handle audio format and open it.
	 *
	 * \param xmm XMM object
	 * \param mode codec mode: Definitions from the #XMM_CODEC_MODE_ group.
	 * #XMM_CODEC_MODE_DECODE or #XMM_CODEC_MODE_ENCODE.
	 * \param af audio format. Data format.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginCodecAudio	*xmm_CodecAudioOpen( void *xmm, int mode, XMM_AudioFormat *af );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Close codec.
	 * Close codec and free all resources used by plugin.
	 *
	 * \param codec Pointer to a audio codec plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecAudioClose( XMM_PluginCodecAudio *codec );

	/**
	 * Control call.
	 * This call is used to control the plugin.
	 *
	 * \param codec Pointer to a audio codec plugin.
	 * \param cmd command.
	 * \param arg argument. For #XMM_CTLSET_ and #XMM_CTLQUERY_ call this
	 * may be an integer, for #XMM_CTLGET_ command it is always a pointer.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecAudioControl( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg );

	/**
	 * Decode data.
	 * Decode (compressed) audio data.
	 *
	 * \param codec Pointer to a audio codec plugin.
	 * \param src Pointer to input data (compressed).
	 * \param isize Size of input data.
	 * \param dest Pointer to output data.
	 * \param samples Used to store the number of output samples.
	 * \param flags Flags.
	 *
	 * \return Upon successful completion the size of the used input data is
	 * returned. Otherwise an error code ( negative value ).
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecAudioDecode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags );

	/**
	 * Encode data.
	 * Encode (uncompressed) audio data.
	 *
	 * \param codec Pointer to a audio codec plugin.
	 * \param src Pointer to input data (uncompressed).
	 * \param samples Number of input samples.
	 * \param dest Pointer to output data.
	 * \param osize Used to store the size of output data.
	 * \param flags Flags.
	 *
	 * \return Upon successful completion the size of the used input data
	 * is returned. Otherwise an error code ( negative value ).
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecAudioEncode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags );

	/**
	 * Get codec info.
	 * Get audio codec info.
	 *
	 * \param codec Pointer to a audio codec plugin.
	 * \param xmm XMM object
	 *
	 * \param Upon successful completion the audio codec info is returned.
	 * Otherwise NULL is returned.
	 *
	 * \note This is only a wrapper.
	 */
XMM_CodecAudioInfo	*xmm_CodecAudioInfo( XMM_PluginCodecAudio *codec, void *xmm );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ------------------------- Misc functions ------------------------ * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Get codec info list.
	 * Get information about installed audio codecs.
	 *
	 * \param xmm XMM object
	 * \param flags Flags from the #XMM_CODEC_ACF_ group.
	 *
	 * \return List of XMM_CodecAudioInfo objects.
	 */
XMM_List *xmm_CodecAudioInfoList( void *xmm, uint32_t flags );

#ifdef __cplusplus
}
#endif

#endif
