/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpcodecv_h
#define INC_libxmm_lpcodecv_h

/** @file lpcodecv.h
 * Linux MultiMedia Project API: Video Codec Layer ( Plugin )
 *
 * Codec Plugins are used to decode (compressed) data and encode (uncompressed)
 * data. The codec plugins have no knowledge of the context/origin of the data
 * they work on.
 */

#include <libxmm/plugin.h>
#include <libxmm/lpgraph.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_CODEC_MODE_ Codec mode
 *
 * Codec mode
 *
 * @{
 */

#ifndef	_XMM_CODEC_MODES_
#define	_XMM_CODEC_MODES_

	/** Open plugin for decoding */
#define	XMM_CODEC_MODE_DECODE		0x01
	/** Open plugin for encoding */
#define	XMM_CODEC_MODE_ENCODE		0x02
	/** Only query plugin */
#define	XMM_CODEC_MODE_QUERY		0x80

#endif

/** @} */

/** @defgroup XMM_CODEC_VCF_ Video Codec Capabilities / Flags
 *
 * Video Codec Capabilities / Flags
 *
 * @{
 */

	/** Plugin is able to decode data */
#define	XMM_CODEC_VCF_DECODE		0x0010
	/** Plugin is able to encode data */
#define	XMM_CODEC_VCF_ENCODE		0x0020
	/** Codec has delta frames */
#define	XMM_CODEC_VCF_DELTA		0x0100

/** @} */

/**
 * Video format.
 * Contains information about the video data
 */

typedef struct xmm_VideoFormat_s
{
    uint32_t			codec;		/**< compression FOURCC */
    uint32_t			width;		/**< Image width */
    uint32_t			height;		/**< Image height */
    uint16_t			planes;		/**< Planes */
    uint16_t			bpp;		/**< Bits / Plane */
    uint32_t			imgSize;	/**< Image size */
    uint32_t			extraSize;	/**< extra data size */
    char			desc[64];	/**< description */
} XMM_VideoFormat;

/**
 * Video codec info.
 * Information on the codec supported by a plugin. Returned by xmm_CodecInfo()
 */

typedef struct xmm_CodecVideoInfo_s
{
    uint32_t			caps;		/**< codec capabilities */
    char			desc[64];	/**< codec description */
    char			file[64];	/**< codec plugin file name */

    uint32_t			fourcc;		/**< codec fourcc */

} XMM_CodecVideoInfo;

/**
 * Video Codec Plugin.
 * This structure / type represents a video codec plugin.
 */

typedef struct xmm_PluginCodecVideo_s	XMM_PluginCodecVideo;

struct xmm_PluginCodecVideo_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize / Open plugin.
     * If url matches I/O type of plugin, initialize plugin and open stream.
     *
     * \param xmm XMM object
     * \param mode codec mode: Definitions from the #XMM_CODEC_MODE_ group.
     * #XMM_CODEC_MODE_DECODE or #XMM_CODEC_MODE_ENCODE. Optionally
     * #XMM_CODEC_MODE_QUERY is possible.
     * \param vf video format. Data format.
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginCodecVideo *	(*Open)( void *xmm, int mode, XMM_VideoFormat *vf );
    /** xmm_CodecVideoClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginCodecVideo *codec );
    /** xmm_CodecVideoControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg );
    /** xmm_CodecVideoDecode() is a wrapper for this member */
    int				(*Decode)( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags );
    /** xmm_CodecVideoEncode() is a wrapper for this member */
    int				(*Encode)( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags );
    /** xmm_CodecVideoInfo() is a wrapper for this member */
    XMM_CodecVideoInfo *	(*Info)( void *xmm );
};

/*
 * Prototypes
 */

	/**
	 * Open codec.
	 * Find plugin, that is able to handle video format and open it.
	 *
	 * \param xmm XMM object
	 * \param mode codec mode: Definitions from the #XMM_CODEC_MODE_ group.
	 * #XMM_CODEC_MODE_DECODE or #XMM_CODEC_MODE_ENCODE.
	 * \param vf video format. Data format.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginCodecVideo	*xmm_CodecVideoOpen( void *xmm, int mode, XMM_VideoFormat *vf );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Close file.
	 * Close file and free all resources used by plugin.
	 *
	 * \param codec Pointer to a video codec plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecVideoClose( XMM_PluginCodecVideo *codec );

	/**
	 * Control call.
	 * This call is used to control the plugin.
	 *
	 * \param codec Pointer to a video codec plugin.
	 * \param cmd command.
	 * \param arg argument. For #XMM_CTLSET_ and #XMM_CTLQUERY_ call this may
	 * be an integer, for #XMM_CTLGET_ command it is always a pointer.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecVideoControl( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg );

	/**
	 * Decode data.
	 * Decode (compressed) video data.
	 *
	 * \param codec Pointer to a video codec plugin.
	 * \param src Pointer to compressed data.
	 * \param isize Size of input data.
	 * \param dest Pointer(s) to the data or planes. When using RGB data or packed YUV only
	 * data[0] is used.
	 * \param osize Used to store the size of output data.
	 * \param flags Flags.
	 *
	 * \return Upon successful completion the size of the used input data is
	 * returned. Otherwise an error code ( negative value ).
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecVideoDecode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags );

	/**
	 * Encode data.
	 * Encode (uncompressed) video data.
	 *
	 * \param codec Pointer to a video codec plugin.
	 * \param src Pointer(s) to the data or planes. When using RGB data or
	 * packed YUV only data[0] is used.
	 * \param isize Size of input data.
	 * \param dest Pointer to compressed data.
	 * \param osize Used to store the size of output data.
	 * \param flags Flags.
	 *
	 * \return Upon successful completion the size of the used input data is
	 * returned. Otherwise an error code ( negative value ).
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_CodecVideoEncode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags );

	/**
	 * Get codec info.
	 * Get video codec info.
	 *
	 * \param codec Pointer to a video codec plugin.
	 * \param xmm XMM object
	 *
	 * \param Upon successful completion the video codec info is returned.
	 * Otherwise NULL is returned.
	 *
	 * \note This is only a wrapper.
	 */
XMM_CodecVideoInfo	*xmm_CodecVideoInfo( XMM_PluginCodecVideo *codec, void *xmm );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ------------------------- Misc functions ------------------------ * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Get codec info list.
	 * Get information about installed video codecs
	 *
	 * \param xmm XMM object
	 * \param flags Flags from the #XMM_CODEC_ACF_ group.
	 *
	 * \return List of XMM_CodecVideoInfo objects.
	 */
XMM_List *xmm_CodecVideoInfoList( void *xmm, uint32_t flags );

#ifdef __cplusplus
}
#endif

#endif
