/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpfiltera_h
#define INC_libxmm_lpfiltera_h

/** @file lpfiltera.h
 * Linux MultiMedia Project API: Audio Filter Layer ( Plugin )
 *
 * Audio Filter Plugins are used to modify (raw/uncompressed ) audio
 * data. For example libxmm uses this plugin type to do conversions
 * if sound format is not supported by Sound Output Plugin.
 * As well, it can be used to apply some effect to the data or
 * it can be used for visualization purposes ( e.g. scope ).
 */

#include <libxmm/plugin.h>
#include <libxmm/lpsound.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Audio Filter Plugin.
 * This structure / type represents a audio filter plugin.
 */

typedef struct xmm_PluginFilterAudio_s	XMM_PluginFilterAudio;

struct xmm_PluginFilterAudio_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize / Open filter.
     * Initialize / Open filter plugin. This call makes a copy of the
     * plugin data and initializes the plugin.
     *
     * \param xmm XMM object
     * \param saf source audio format
     * \param daf destination audio format
     * \param isize size of input data block
     * \param osize Pointer to integer, that will contain size of output data
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginFilterAudio *	(*Open)( void *xmm, XMM_SoundFormat *saf, XMM_SoundFormat *daf, uint32_t isize, uint32_t *osize );
    /** xmm_FilterAudioClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginFilterAudio *filter );
    /** xmm_FilterAudioProc() is a wrapper for this member */
    int				(*Proc)( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata );
};

/*
 * Prototypes
 */

	/**
	 * Open audio filter.
	 * Load audio filter plugin and initialize it.
	 *
	 * \param xmm XMM object
	 * \param filename audio filter plugin
	 * \param saf source audio format
	 * \param daf destination audio format
	 * \param isize size of input data block
	 * \param osize Pointer to integer, that will contain size of output data
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginFilterAudio	*xmm_FilterAudioOpen( void *_xmm, char *filename, XMM_SoundFormat *saf, XMM_SoundFormat *daf, uint32_t isize, uint32_t *osize );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Close filter.
	 * Close filter and free all resources used by plugin.
	 *
	 * \param filter Pointer to an audio filter plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterAudioClose( XMM_PluginFilterAudio *filter );

	/**
	 * Process data.
	 * Process some data.
	 *
	 * \param filter Pointer to an audio filter plugin.
	 * \param sdata Pointer to source data
	 * \param ddata Pointer to a Pointer. Pointer to the destination data
	 * will be stored there.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterAudioProc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Audio Filter Chain ---------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Init chain.
	 * Initialize audio filter chain/subsystem.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainInit( void *xmm );

	/**
	 * Free chain.
	 * Free audio filter chain/subsystem.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainExit( void *xmm );

	/**
	 * Lock chain.
	 * Lock audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainLock( void *xmm );

	/**
	 * Unlock chain.
	 * Unlock audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainUnlock( void *xmm );

	/**
	 * Chain status.
	 * Check if the audio filter chain is active.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion 1 is returned if chain is active,
	 * 0 if not. Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainActive( void *xmm );

	/**
	 * Input format.
	 * Set Input / Source format for audio filter chain.
	 *
	 * \param xmm XMM object
	 * \param samprate Samplerate
	 * \param channels Channels
	 * \param format Format
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainInput( void *xmm, int samprate, int channels, int format );

	/**
	 * Output format.
	 * Set Output / Target format and buffer size for audio filter chain.
	 * Returns: needed number of samples ( of source data )
	 *
	 * \param xmm XMM object
	 * \param samprate Samplerate
	 * \param channels Channels
	 * \param format Format
	 * \param blocksize Size of the output buffer, for sound output this
	 * normally is the fragment size.
	 *
	 * \return Upon successful completion the buffer size ( samples )
	 * is retured. Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainOutput( void *xmm, int samprate, int channels, int format, int blocksize );

	/**
	 * Process chain.
	 * Process audio filter chain.
	 *
	 * \param xmm XMM object.
	 * \param sdata source data.
	 * \param isamples Size of source data in samples.
	 * \param ddata After call, it contains pointer to destination data.
	 * \param osamples After call, it contains number of output samples.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChain( void *xmm, uint8_t *sdata, uint32_t isamples, uint8_t **ddata, uint32_t *osamples );

	/**
	 * Stop chain.
	 * Stop audio filter chain.
	 *
	 * \note Will stop all stuff started in xmm_FilterAChainOutput().
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainStop( void *xmm );

	/**
	 * Add plugin.
	 * Add plugin to audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainAdd( void *xmm, XMM_PluginFilterAudio *pfa );

	/**
	 * Add plugin list.
	 * Add plugin list to audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainAddList( void *xmm, XMM_List *pAudioFilterList );

	/**
	 * Remove plugins.
	 * Remove all plugins from audio filter chain
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainFree( void *xmm );

#ifdef __cplusplus
}
#endif

#endif
