/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpgraph_h
#define INC_libxmm_lpgraph_h

/** @file lpgraph.h
 * Linux MultiMedia Project API: Graph Output Layer ( Plugin )
 *
 * Graph Output Plugins are used to output graphic data.
 * This maybe the output to screen, or as part of Output Plugins, the
 * data will be written to a file. This plugin type is used for example
 * in in libxmmplay. It may also be used by any application.
 * Check out contrib/samples/graph/gtest.c.
 *
 * \note Graph plugins must provide (key) events. If event handling is
 * not possible with handlers, the plugin has to poll regularly for
 * events. This can be done, for example, with the timer functions.
 */

#include <libxmm/plugin.h>

#ifdef __cplusplus
extern "C"
{
#endif

	/**
	 * Create FourCC from four integers.
	 * FourCC means Four Character Code. It is used by LinuX MultiMedia Project
	 * to identify the video codec. Some file formats ( e.g. AVI ) uses FOURCC's
	 * for chunk ID's, as well.
	 */
#define xmmFOURCC( ch0, ch1, ch2, ch3 )				\
		((uint32_t)(ch0) | ((uint32_t)(ch1) << 8 ) | 		\
		((uint32_t)(ch2) << 16 ) |((uint32_t)(ch3) << 24 ))

/** @defgroup XMM_GRAPH_FMT_ Graphic Formats
 *
 * Graphic Formats
 *
 * @{
 */

	/** RGB with depth of bpp */
#define XMM_GRAPH_FMT_RGB(bpp)		xmmFOURCC( 'R','G','B', bpp )
	/** BGR with depth of bpp */
#define XMM_GRAPH_FMT_BGR(bpp)		xmmFOURCC( 'B','G','R', bpp )
	/** planar: Y V U */
#define XMM_GRAPH_FMT_YV12		xmmFOURCC( 'Y','V','1','2' )
	/** planar: Y U V ( IYUV = I420 ) */
#define XMM_GRAPH_FMT_IYUV		xmmFOURCC( 'I','Y','U','V' )
	/** packed: Y0 U0 Y1 V0 */
#define XMM_GRAPH_FMT_YUY2		xmmFOURCC( 'Y','U','Y','2' )
	/** packed: U0 Y0 V0 Y1 */
#define XMM_GRAPH_FMT_UYVY		xmmFOURCC( 'U','Y','V','Y' )
	/** packed: Y0 V0 Y1 U0 */
#define XMM_GRAPH_FMT_YVYU		xmmFOURCC( 'Y','V','Y','U' )

/** @} */
/** @defgroup XMM_GRAPH_FLAG_ Flags to xmm_GraphStart()
 *
 * Flags
 *
 * @{
 */

	/** Flip image. Image has to be flipped vertical. */
#define	XMM_GRAPH_FLAG_YFLIP		0x00000001

/** @} */
/** @defgroup XMM_GRAPH_RESIZE_ Resize Flags
 *
 * Resize flags.
 *
 * @{
 */

	/** Resize to original size. Image will be of the size specified in xmm_GraphStart() */
#define	XMM_GRAPH_RESIZE_ORIG		0x00000001
	/** Resize to double size. Image will be double of the size specified in xmm_GraphStart()  */
#define XMM_GRAPH_RESIZE_DOUBLE		0x00000002
	/** Resize to fullscreen size. Image will cover the whole screen*/
#define XMM_GRAPH_RESIZE_FULLSCREEN	0x00000004
	/** Resize without scaling. NOT used yet. Maybe removed in future. */
#define	XMM_GRAPH_RESIZE_NOSCALE	0x00000008

/** @} */

/**
 * Graph Output Plugin.
 * This structure / type represents a graph output plugin.
 */

typedef struct xmm_PluginGraph_s	XMM_PluginGraph;

struct xmm_PluginGraph_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize plugin.
     * Initialize / create plugin. This call makes a copy of the
     * plugin data and initializes the plugin.
     *
     * \param xmm XMM object
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginGraph *		(*Init)( void *xmm );
    /** xmm_GraphClose() is a wrapper for this member */
    void			(*Close)( XMM_PluginGraph *graph );
    /** xmm_GraphControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginGraph *graph, uint32_t cmd, uint32_t arg, void *data );
    /** xmm_GraphStart() is a wrapper for this member */
    int				(*Start)( XMM_PluginGraph *graph, int width, int height, uint32_t format, int flags );
    /** xmm_GraphDraw() is a wrapper for this member */
    void			(*Draw)( XMM_PluginGraph *graph, uint8_t *data[], int stride[], int x, int y, int width, int height, int flags );
    /** xmm_GraphBlit() is a wrapper for this member */
    void			(*Blit)( XMM_PluginGraph *graph );
};

/*
 * Prototypes
 */

	/**
	 * Open graph.
	 * Load graph plugin and initialize it.
	 *
	 * \param xmm XMM object
	 * \param filename graph plugin.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginGraph	*xmm_GraphOpen( void *xmm, char *filename );

	/**
	 * Open graph ( memory ).
	 * Load graph plugin ( existing plugin in memory ) and initialize it.
	 *
	 * \param xmm XMM object
	 * \param tGraph graph plugin.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginGraph	*xmm_GraphOpenMP( void *xmm, XMM_PluginGraph *tGraph );

	/**
	 * Free plugin.
	 * Close output and free all resources used by plugin.
	 *
	 * \param graph Pointer to a graph plugin.
	 */
int	xmm_GraphClose( XMM_PluginGraph *graph );

	/**
	 * Control call.
	 * This call is used to control the plugin.
	 *
	 * \param graph pointer to a graph plugin.
	 * \param cmd command.
	 * \param param 1st argument. An integer argument.
	 * \param data 2nd argument. Pointer to some data.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 */
int	xmm_GraphControl( XMM_PluginGraph *graph, int cmd, uint32_t param, void *data );

	/**
	 * Start graph output.
	 * This call will start the graphic output according to the
	 * given arguments.
	 *
	 * \param graph Pointer to a graph plugin
	 * \param width Width of the source image
	 * \param height Height of the source image
	 * \param format Data format of the source image
	 * \param flags Flags
	 *		- #XMM_GRAPH_FLAG_YFLIP Flip image
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note If desired format is not available, the plugin will
	 * start a graph converter using libxmm. The same will
	 * be done for scaling purposes. Only if the source format
	 * is supported by the plugin and no scaling shall be done
	 * the graph plugin will handle the data. Otherwise the conversion
	 * subsystem will handle handle/convert the data.
         */
int	xmm_GraphStart( XMM_PluginGraph *graph, int width, int height, uint32_t format, int flags );

	/**
	 * Output some graph data.
	 * This function, will output a complete frame, slices or any part of the
	 * image. 
	 *
	 * \param graph Pointer to a graph plugin
	 * \param data Pointer(s) to the data or planes. When using RGB data or packed YUV only
	 * data[0] is used.
	 * \param stride Bytes per line. Two reasons for this:
	 * - The internal buffer of a decoder may be bigger then the image.
	 * - The lines in the internal buffer of a decoder may be aligned.
	 * \param x horizontal position in the source image.
	 * \param y vertical position in the source image.
	 * \param width horizontal pixels that should be processed.
	 * \param height vertical pixels that should be processed.
	 * \param flags Flags. ( NOT used yet. For future extensions )
	 *
	 * \note Draw() will process the data ( e.g. do needed conversions ).
	 * Normally it will be stored in an internal buffer. To put this content
	 * to the screen we need another function: Blit().
	 *
	 * \note In YUV modes only complete lines can be drawn. x and width are ignored.
	 */
int	xmm_GraphDraw( XMM_PluginGraph *graph, uint8_t *data[], int stride[], int x, int y, int width, int height, int flags );

	/**
	 * Blit image to screen.
	 * Normally this call will update/write the frame. If plugin doesn't use
	 * an internal buffer, this call won't do anything, but still has to
	 * be called.
	 *
	 * \param graph Pointer to a graph plugin
	 */
int	xmm_GraphBlit( XMM_PluginGraph *graph );

#ifdef __cplusplus
}
#endif

#endif
