/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpsound_h
#define INC_libxmm_lpsound_h

/** @file lpsound.h
 * Linux MultiMedia Project API: Sound Output Layer ( Plugin )
 *
 * Graph Output Plugins are used to output sound data.
 * This maybe the output to your sound card, or as part of Output Plugins, the
 * data will be written to a file. This plugin type is used for example
 * in in libxmmplay. It may also be used by any application.
 */

#include <libxmm/plugin.h>

#ifdef __cplusplus
extern "C"
{
#endif


/** @defgroup XMM_SOUND_MASK_ Sound Format Masks
 *
 * Sound Formats Masks
 *
 * @{
 */
	/** Format size. Format size in bits ( e.g. 8 or 16 ) */
#define	XMM_SOUND_MASK_SIZE	0x000000FF
	/** Format is signed */
#define	XMM_SOUND_MASK_SIGNED	0x00000100
	/** Format is big endian */
#define	XMM_SOUND_MASK_BE	0x00000200

/** @} */
/** @defgroup XMM_SOUND_FMT_ Sound Formats
 *
 * Sound Formats
 *
 * @{
 */
	/** Unsigned 4 bit, needed for ADPCM */
#define XMM_SOUND_FMT_U4		( 0x04 )
	/** Unsigned 8 bit */
#define XMM_SOUND_FMT_U8		( 0x08 )
	/** Signed 16 bit Little endian */
#define XMM_SOUND_FMT_S16LE		( 0x10 | XMM_SOUND_MASK_SIGNED )
	/** Signed 16 bit Big endian */
#define XMM_SOUND_FMT_S16BE		( 0x10 | XMM_SOUND_MASK_SIGNED | XMM_SOUND_MASK_BE )
	/** Signed 8 bit */
#define XMM_SOUND_FMT_S8		( 0x08 | XMM_SOUND_MASK_SIGNED )
	/** Unsigned 16 bit Little endian */
#define XMM_SOUND_FMT_U16LE		( 0x10 )
	/** Unsigned 16 bit Big endian */
#define XMM_SOUND_FMT_U16BE		( 0x10 | XMM_SOUND_MASK_BE )

/** @} */

/*
 * Types
 */

/**
 * Sound format.
 * Contains all informations needed for output
 */

typedef struct xmm_SoundFormat_s
{
    /** Data format. One of the #XMM_SOUND_FMT_ values */
    uint16_t			format;
    /** Sample rate. Number of samples per second */
    uint32_t			samprate;
    /** Channels. Number of channels */
    uint16_t			channels;
} XMM_SoundFormat;


/**
 * Sound Output Plugin.
 * This structure / type represents a sound output plugin.
 */

typedef struct xmm_PluginSound_s	XMM_PluginSound;

struct xmm_PluginSound_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize plugin.
     * Initialize / create plugin. This call makes a copy of the
     * plugin data and initializes the plugin.
     *
     * \param xmm XMM object
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginSound *		(*Init)( void *xmm );
    /** xmm_SoundClose() is a wrapper for this member */
    void			(*Close)( XMM_PluginSound *sound );
    /** xmm_SoundControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginSound *graph, uint32_t cmd, uint32_t arg, void *data );
    /** xmm_SoundStart() is a wrapper for this member */
    int				(*Start)( XMM_PluginSound *sound, XMM_SoundFormat *format, int fsize, int fcount );
    /** xmm_SoundWrite() is a wrapper for this member */
    int				(*Write)( XMM_PluginSound *sound, void *data, uint32_t samples );
    /** xmm_SoundFlush() is a wrapper for this member */
    void			(*Flush)( XMM_PluginSound *sound );
};

/*
 * Prototypes
 */

	/**
	 * Open sound.
	 * Load sound plugin and initialize it.
	 *
	 * \param xmm XMM object
	 * \param filename sound plugin.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginSound	*xmm_SoundOpen( void *xmm, char *filename );

	/**
	 * Open sound from memory plugin.
	 * Initialize sound plugin from existing plugin object.
	 *
	 * \param xmm XMM object
	 * \param tSound sound plugin.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginSound *xmm_SoundOpenMP( void *xmm, XMM_PluginSound *tSound );

	/**
	 * Free plugin.
	 * Close output and free all resources used by plugin.
	 *
	 * \param sound Pointer to a sound plugin.
	 */
int	xmm_SoundClose( XMM_PluginSound *sound );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Control call.
	 * This call is used to control the plugin.
	 *
	 * \param sound pointer to a sound plugin.
	 * \param cmd command.
	 * \param param 1st argument. An integer argument.
	 * \param data 2nd argument. Pointer to some data.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 */
int	xmm_SoundControl( XMM_PluginSound *graph, uint32_t cmd, uint32_t param, void *data );

	/**
	 * Start sound output.
	 * This call will start the sound output according to the
	 * given arguments.
	 *
	 * \param sound Pointer to a sound plugin
	 * \param format Data format ( Pointer to a #XMM_SoundFormat object )
	 * \param fsize Desired fragment size
	 * \param fcount Desired fragment number
	 *
	 * \return Upon successful completion the buffer size in samples
	 * is retured. Otherwise an error code ( negative value )
	 *
	 * \note If desired format is not available, the plugin will
	 * use a converter using libxmm.
         */
int	xmm_SoundStart( XMM_PluginSound *sound, XMM_SoundFormat *format, int fsize, int fcount );

	/**
	 * Write data.
	 * Write some data to the sound output ( subsystem ).
	 *
	 * \param sound Pointer to a sound plugin
	 * \param data Pointer to the data
	 * \param samples Number of samples
	 *
	 * \return Upon successful completion the number of written samples
	 * is retured. Otherwise an error code ( negative value )
	 *
	 * \note Data size ( samples ) must be exactly the fragments size.
	 * ( The value returned by xmm_SoundStart() ).
	 * Most sound drivers use internal buffers ( fragments ). Some drivers
	 * are only able to write an amount of data, that is a multiple of the
	 * fragments size. ( Hopefully ) this will change in future.
         */
int	xmm_SoundWrite( XMM_PluginSound *sound, void *data, uint32_t samples );

	/**
	 * Flush data.
	 * Removes remaining data from buffers, what will stop playback
	 * immediatly.
	 *
	 * \param sound pointer to a sound plugin.
	 */
int	xmm_SoundFlush( XMM_PluginSound *sound );

#ifdef __cplusplus
}
#endif

#endif
