/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_util_config_h
#define INC_libxmm_util_config_h

/** @file config.h
 * Linux MultiMedia Project: Config file tools
 */

#include <libxmm/util/list.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Types
 */

/*
 * Config line
 */

typedef struct xmm_ConfigLine_s
{
    char	*name;
    char	*value;
} XMM_ConfigLine;

/*
 * Config section
 */

typedef struct xmm_ConfigSection_s
{
    char	*name;
    XMM_List	*lines;
} XMM_ConfigSection;

/**
 * Config file.
 * This data types represents a config file. It will contain all data,
 * that can be found in the config file.
 */

typedef struct xmm_ConfigFile_s
{
    XMM_List	*sections;
} XMM_ConfigFile;

/*
 * Prototypes
 */
		/**
		 * Create new config file.
		 * xmmCfg_New() will create an empty config file. Not really
		 * a file, it will create the data in memory.
		 *
		 * \return Upon successful completion the config file object
		 * is returned. Otherwise NULL.
		 */
XMM_ConfigFile	*xmmCfg_New( void );
		/**
		 * Free config file.
		 * xmmCfg_Free() destroys the memory represantion of a
		 * config file.
		 *
		 * \param cfg config file
		 */
void		xmmCfg_Free( XMM_ConfigFile *cfg );
		/**
		 * Remove config line.
		 * This funtion will remove a config line.
		 *
		 * \param cfg config file
		 * \param secname section from which to remove
		 * \param name config line to remove
		 */
void		xmmCfg_Remove( XMM_ConfigFile *cfg, char *secname, char *name );

		/**
		 * Read config file.
		 * Read config file into memory.
		 *
		 * \param filename path to config file
		 *
		 * \return Upon successful completion the config file object
		 * is returned. Otherwise NULL.
		 */
XMM_ConfigFile	*xmmCfg_ReadFile( char *filename );
		/**
		 * Write config file.
		 * Write config file from memory to disk.
		 *
		 * \param cfg config file
		 * \param filename path to config file
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int		xmmCfg_WriteFile( XMM_ConfigFile *cfg, char *filename );

		/**
		 * Read string.
		 * Read string from a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value function will store there the string
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_ReadString( XMM_ConfigFile *cfg, char *secname, char *name, char *value );
		/**
		 * Read integer.
		 * Read integer from a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value function will store there the integer value
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_ReadInt( XMM_ConfigFile *cfg, char *secname, char *name, int *value );
		/**
		 * Read boolean.
		 * Read boolean from a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value function will store there the integer value
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_ReadBool( XMM_ConfigFile *cfg, char *secname, char *name, int *value );
		/**
		 * Read double float.
		 * Read double float from a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value function will store there the double float value
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_ReadDouble( XMM_ConfigFile *cfg, char *secname, char *name, double *value );
		/**
		 * Read string list.
		 * Read string list from a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param list list of strings
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_ReadStringList( XMM_ConfigFile *cfg, char *secname, char *name, XMM_List **list );

		/**
		 * Write string.
		 * Write string to a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value string
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_WriteString( XMM_ConfigFile *cfg, char *secname, char *name, char *value );
		/**
		 * Write integer.
		 * Write integer to a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value integer value
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_WriteInt( XMM_ConfigFile *cfg, char *secname, char *name, int value );
		/**
		 * Write boolean.
		 * Write boolean to a config file. This is either TRUE
		 * ( 1 as integer ) or FALSE ( 0 as integer ).
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value integer value. Either zero (0) or one (1).
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_WriteBool( XMM_ConfigFile *cfg, char *secname, char *name, int value );
		/**
		 * Write double float.
		 * Write double float to a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param value double float value
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_WriteDouble( XMM_ConfigFile *cfg, char *secname, char *name, double value );
		/**
		 * Write string list.
		 * Write string list to a config file.
		 *
		 * \param cfg config file
		 * \param secname section name
		 * \param config file entry
		 * \param list list of strings
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 */
int xmmCfg_WriteStringList( XMM_ConfigFile *cfg, char *secname, char *name, XMM_List **list );

/* - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - *
 *   *   *                        Config Block                       *   *   *
 * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - */

/*
 * Config block line types
 */

/** Config file integer entry line */
#define	XMM_CFG_TYPE_INT		0x01
/** Config file string entry line */
#define	XMM_CFG_TYPE_STRING		0x02
/** Config file boolean entry line */
#define	XMM_CFG_TYPE_BOOL		0x03
/** Config file double float entry line */
#define	XMM_CFG_TYPE_DOUBLE		0x04

/** Config file string list entry line */
#define	XMM_CFG_TYPE_STRING_LIST	0x0A

/**
 * Config file block.
 * This structure is used with xmmCfg_BlockLoad() and xmmCfg_BlockSave().
 * Usually, it is defined as array and passed as argument to these funtions.
 * Using this functions, many config entries can be read / written with
 * one call.
 */

typedef struct xmm_ConfigBlock_s
{
		/** Pointer to a variable */
    void	*ptr;
		/** Entry name */
    char	name[256];
		/** Entry type, one of the XMM_CFG_ definitions */
    int		type;
} XMM_ConfigBlock;

/*
 * Config block prototypes
 */

		/**
		 * Load config file block.
		 * Used to load a block of config file entries.
		 *
		 * \param filename full path and name of config file
		 * \param section section name
		 * \param ucfg pointer to a XMM_ConfigBlock array.
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 *
		 * \note The last XMM_ConfigBlock in the array, must
		 * have ptr set to NULL.
		 */
int xmmCfg_BlockLoad( char *filename, char *section, XMM_ConfigBlock *ucfg );
		/**
		 * Save config file block.
		 * Used to save a block of config file entries.
		 *
		 * \param filename full path and name of config file
		 * \param section section name
		 * \param ucfg pointer to a XMM_ConfigBlock array.
		 *
		 * \return Upon successful completion #XMM_RET_OK is returned.
		 * Otherwise an error code ( negative value ).
		 *
		 * \note The last XMM_ConfigBlock in the array, must
		 * have ptr set to NULL.
		 */
int xmmCfg_BlockSave( char *filename, char *section, XMM_ConfigBlock *ucfg );

#ifdef __cplusplus
}
#endif

#endif
