/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_xmmctl_h
#define INC_libxmm_xmmctl_h

/** @file xmmctl.h
 * Linux MultiMedia Project API: control interface
 *
 * Some plugin types have a special call. The control() call:
 * int Control( XMM_Plugin<type> *plugin, int cmd, void *arg );
 *
 * - plugin Pointer to the called plugin
 * - cmd Command
 * - arg Argument. Maybe an integer ( 32 bits ) or a pointer,
 *   depending on the individual command
 *
 * There are 4 command types:
 * - #XMM_CTLQUERY_ Query a feature. Upon successful completion #XMM_CTLRET_TRUE
 *   is returned if feature supported, #XMM_CTLRET_FALSE if not supported.
 *   Otherwise an error code ( negative value )
 * - #XMM_CTLGET_ Get some data/information. Upon successful completion the data
 *   is stored in the memory pointed by arg and #XMM_CTLRET_ARG is returned.
 *   Otherwise an error code ( negative value ).
 * - #XMM_CTLSET_ Set some data. Upon successful completion #XMM_CTL_TRUE is
 *   returned. Otherwise an error code ( negative value ). Argument maybe
 *   an integer or a pointer ( usually to a struct ),
 * - #XMM_CTLINP_ Special input commands for direct output mode.
 */

#include <inttypes.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_CTLRET_ Control call return values
 *
 * Control call return values
 *
 * @{
 */

	/** Unkown command. Plugin does not know the command. */
#define XMM_CTLRET_UNKNOWN		-7
	/** Invalid command. Command is not valid for this plugin type. */
#define XMM_CTLRET_INVALID		-6
	/** Invalid argument. On of the arguments is invalid. */
#define XMM_CTLRET_INVALID_ARG		-5
	/** Not supported. Command / Feature is not supported */
#define XMM_CTLRET_NOTSUPPORTED		-2
	/** Unknown error. Error, not categorized. */
#define XMM_CTLRET_ERROR		-1
	/** Failure */
#define XMM_CTLRET_FALSE		0
	/** Success in query/set */
#define XMM_CTLRET_TRUE			1
	/** Success in get, result can be found in arg */
#define XMM_CTLRET_ARG			2

/** @} */
/** @defgroup XMM_CTLMASK_ Control layer type masks
 *
 * Control layer type masks
 *
 * @{
 */

	/** Graph subsystem */
#define XMM_CTLMASK_GRAPH		0x01000000
	/** Sound subsystem */
#define XMM_CTLMASK_SOUND		0x02000000
	/** Codec subsystem */
#define XMM_CTLMASK_CODEC		0x04000000
	/** Input subsystem */
#define XMM_CTLMASK_INPUT		0x08000000
	/** video filter subsystem */
#define XMM_CTLMASK_VFILTER		0x10000000

/** @} */
/** @defgroup XMM_CTLQUERY_ Query control commands
 *
 * Query control commands
 *
 * @{
 */

	/** Graph format. Query if graph format supported ( by hardware ).
	 * Arg is a pointer to an uint32_t containing the FOURCC. */
#define XMM_CTLQUERY_GFORMAT		( 0x101 | XMM_CTLMASK_GRAPH | XMM_CTLMASK_CODEC | XMM_CTLMASK_INPUT )
	/** Sound format. Query if sound format supported ( by driver ).
	 * Arg is a pointer to a XMM_SoundFormat struct.
	 * \note Not supported by input plugins in direct out mode */
#define XMM_CTLQUERY_SFORMAT		( 0x102 | XMM_CTLMASK_SOUND | XMM_CTLMASK_CODEC | XMM_CTLMASK_INPUT )
	/** Y-Flipping. Query if plugin supports Y-Flipping. For graph plugins,
	 * this means hardware support. Arg will be ignored. */
#define XMM_CTLQUERY_YFLIP		( 0x103 | XMM_CTLMASK_GRAPH | XMM_CTLMASK_CODEC | XMM_CTLMASK_INPUT )
	/** Hardware scaling. Query if driver support hardware scaling.
	 * Arg will be ignored. */
#define XMM_CTLQUERY_HWSCALE		( 0x104 | XMM_CTLMASK_GRAPH )
	/** Fullscreen. Query if driver supports fullscreen.
	 * Arg will be ignored. */
#define XMM_CTLQUERY_FULLSCREEN		( 0x105 | XMM_CTLMASK_GRAPH )
	/** Surface reading. Query if reading from surface is possible.
	 * Arg will be ignored. \note This command maybe removed in future,
	 * as I'm not sure we really need it */
#define XMM_CTLQUERY_SURFACE_READ	( 0x106 | XMM_CTLMASK_GRAPH )


/** @} */
/** @defgroup XMM_CTLGET_ Get control commands
 *
 * Get control commands
 *
 * @{
 */

	/** Get best graph format. This should be the fastest format.
	 * Arg is a pointer to an uint32_t, which will contain the FOURCC. */
#define XMM_CTLGET_GFORMAT		( 0x201 | XMM_CTLMASK_GRAPH | XMM_CTLMASK_CODEC | XMM_CTLMASK_INPUT )
	/** Get native sound format. Format of the sound data.
	 * Arg is a pointer to a XMM_SoundFormat struct, which will contain the format.
	 * \note Not supported by input plugins in direct out mode */
#define XMM_CTLGET_SFORMAT		( 0x202 | XMM_CTLMASK_CODEC | XMM_CTLMASK_INPUT )
	/** Get information and address of the graph surface / frame buffer
	 * Arg is a pointer to a XMM_ControlSurface struct, which will contain
	 * the info. This call makes it possible to write directly into the
	 * drivers internal buffer. This command will fail if conversion
	 * ( includes stretching ) has to be done
	 * \note After calls changing the internal buffer ( e.g. scaling )
	 * this call has to be done again, to get new information on the
	 * internal buffer. However, probably the surface will no longer
	 * be available */
#define XMM_CTLGET_SURFACE		( 0x203 | XMM_CTLMASK_GRAPH )
	/** Get delay. Get sound output delay in seconds. The delay
	 * is caused by the internal buffer ( fragments ) of the sound
	 * driver. Arg is a pointer to a double, which will contain the delay. */
#define XMM_CTLGET_DELAY		( 0x204 | XMM_CTLMASK_SOUND )
	/** Get maximum delay. Get maximum sound output delay in seconds.
	 * The delay is caused by the internal buffer ( fragments ) of the sound
	 * driver. Arg is a pointer to a double, which will contain the delay. */
#define XMM_CTLGET_MAX_DELAY		( 0x205 | XMM_CTLMASK_SOUND )
	/** Get buffer size. Usually this is frag_size * frag_num.
	 * Arg is a pointer to a uint32_t, which will contain the delay */
#define	XMM_CTLGET_BUFFER_TOTAL		( 0x206 | XMM_CTLMASK_SOUND )
	/** Get free buffer amount. Writing more data to output usually
	 * means blocking.
	 * Arg is a pointer to a uint32_t, which will contain the delay */
#define	XMM_CTLGET_BUFFER_FREE		( 0x207 | XMM_CTLMASK_SOUND )
	/** Screen size. Get size of the screen.
	 * Arg is a pointer to a XMM_ControlRect struct, the members
	 * 'width' and 'height' will contain the screen size */
#define	XMM_CTLGET_SCREEN		( 0x208 | XMM_CTLMASK_GRAPH )
	/** Window position. Get position of the output window.
	 * Arg is a pointer to a XMM_ControlRect struct, the members
	 * 'x' and 'y' will contain the screen size. REMOVED (temporary?) */
#if 0	/* Really needed ??? */
#define	XMM_CTLGET_POSITION		( 0x209 | XMM_CTLMASK_GRAPH )
#endif
	/** Get volume. Get the volume setting.
	 * Arg is a poiner to a uint32_t. The lower 16 bits will contain
	 * the volume of the left channel, the higher 16 bits will
	 * contain the volume of the right channel. */
#define XMM_CTLGET_VOLUME		( 0x20A | XMM_CTLMASK_INPUT | XMM_CTLMASK_SOUND )
	/** Get capabilities. This will return the supported modes of
	 * an input plugin. ( XMM_INPUT_CF_MODE_READ, XMM_INPUT_CF_MODE_DOUT )
	 * Arg is a pointer to a uint32_t. */
#define	XMM_CTLGET_CAPS			( 0x20B | XMM_CTLMASK_INPUT )
	/** Get source data size. Compressed audio data has a fixed bitrate.
	 * This call will return the amount of compressed data needed for
	 * an amount of uncompressed data.
	 * Arg is a pointer to a uint32_t. Before the call it has to
	 * contain the uncompressed data size. After the call it will
	 * contain the needed amount of compressed data. */
#define XMM_CTLGET_DATA_SSIZE		( 0x20C | XMM_CTLMASK_CODEC )
	/** Size of video format. Get size of video format.
	 * Arg is a pointer to a uint32_t, which will contain the format size. */
#define	XMM_CTLGET_FORMAT_SIZE		( 0x20D | XMM_CTLMASK_CODEC )
	/** Video format. Get video format. Needed when encoding data.
	 * Arg is a pointer to a XMM_VideoFormat pointer, which will contain
	 * a the address of a XMM_VideoFormat struct. This must NOT be
	 * modified. */
#define	XMM_CTLGET_FORMAT_DATA		( 0x20E | XMM_CTLMASK_CODEC )

/** @} */
/** @defgroup XMM_CTLSET_ Set control commands
 *
 * Set control commands
 *
 * @{
 */

	/** Set graph format. This call will set the graph format used for output.
	 * Arg is a uint32_t, containing the FOURCC */
#define XMM_CTLSET_GFORMAT		( 0x301 | XMM_CTLMASK_CODEC | XMM_CTLMASK_INPUT )
	/** Set graph scaling. This call will set a new size of the graph image.
	 * Arg is a pointer to a XMM_ControlScale struct. If input plugin is in
	 * direct out mode, the input plugin has to do scaling. Else scaling is
	 * done by the graph output subsystem ( including graph conversion ). */
#define XMM_CTLSET_SCALE		( 0x302 | XMM_CTLMASK_GRAPH | XMM_CTLMASK_INPUT | XMM_CTLMASK_VFILTER )
	/** Set position. Set position of the output window. 
	 * Arg is a pointer to a XMM_ControlRect struct. The position has to
	 * be specified in the 'x' and 'y' members. */
#define XMM_CTLSET_POSITION		( 0x303 | XMM_CTLMASK_GRAPH )
	/** Set title. Set window title.
	 * Arg is a pointer to the string (char *). If NULL the titlebar
	 * will be disabled. */
#define XMM_CTLSET_TITLE		( 0x304 | XMM_CTLMASK_GRAPH )
	/** Set framerate. Set output frame rate. This usually MUST
	 * be set, as file output plugins need a correct value.
	 * Arg is a uint32_t, containing the framerate */
#define XMM_CTLSET_FRAMERATE		( 0x305 | XMM_CTLMASK_GRAPH )
	/** Set volume. Set the volume. If plugin is in direct out mode,
	 * the input plugin has to set the volume, else the sound output
	 * plugin will set the volume.
	 * Arg is a uint32_t. The lower 16 bits will contain the volume
	 * of the left channel, the higher 16 bits will contain the
	 * volume of the right channel. */
#define XMM_CTLSET_VOLUME		( 0x0306 | XMM_CTLMASK_SOUND | XMM_CTLMASK_INPUT )
	/** Seeking notify. Notify the codec about seeking done in the media.
	 * This call is needed, as codecs may buffer some data and after
	 * seeking this data must NOT be used by the nexte decode call.
	 * \note This call is only needed for audio codecs. */
#define	XMM_CTLSET_SEEK			( 0x0307 | XMM_CTLMASK_CODEC )

/** @} */
/** @defgroup XMM_CTLINP_ Special commands for direct out
 *
 * Special commands for direct out
 *
 * @{
 */

/*
 * Commands for some 'special' input plugins:
 * ( that do not have AudioRead(), VideoRead() these do their own output )
 */

	/** Play. Start playback. Command has no argument, and will be ignored.
	 * Upon successful completion #XMM_CTL_TRUE is returned.
	 * Otherwise an error code ( negative value ). */
#define	XMM_CTLINP_PLAY			( 0x401 | XMM_CTLMASK_INPUT )
	/** Stop. Stop playback. Command has no argument, and will be ignored.
	 * Upon successful completion #XMM_CTL_TRUE is returned.
	 * Otherwise an error code ( negative value ). */
#define	XMM_CTLINP_STOP			( 0x402 | XMM_CTLMASK_INPUT )
	/** Pause. Pause playback. Command has no argument, and will be ignored.
	 * Upon successful completion #XMM_CTL_TRUE is returned.
	 * Otherwise an error code ( negative value ). */
#define	XMM_CTLINP_PAUSE		( 0x403 | XMM_CTLMASK_INPUT )
	/** Get status. Get playback status.
	 * Argument is a pointer to uint32_t, and will contain the current
	 * playback status after successful call. Upon successful completion
	 * #XMM_CTL_ARG is returned. Otherwise an error code ( negative value ). */
#define	XMM_CTLINP_STATUS		( 0x404 | XMM_CTLMASK_INPUT )

/** @} */
/** @defgroup XMM_CTLDLG_ Dialogue interface
 *
 * Dialogue interface
 *
 * @{
 */

	/** Query dialogue capability. Check if dialogue available.
	 * Argument is a uint32_t, containing the desired dialogue code.
	 * A definition from the #XMM_GTKDLG group */
#define	XMM_CTLDLG_QUERY		( 0x501 )

	/** Display dialogue. This call will advise the plugin to display a
	 * dialogue. Argument is a uint32_t, containing the desired dialogue
	 * code. A definition from the #XMM_GTKDLG_ group */
#define	XMM_CTLDLG_DISPLAY		( 0x502 )

/** @} */

/** @defgroup XMM_GTKDLG_ Gtk+ Dialogues
 *
 * Gtk+ Dialogues
 *
 * @{
 */

	/** Display Gtk+ About window */
#define	XMM_GTKDLG_ABOUT		0x00000001
	/** Display Gtk+ Configuration window */
#define	XMM_GTKDLG_CONFIG		0x00000002
	/** Display Gtk+ Info window */
#define	XMM_GTKDLG_INFO			0x00000003

/** @} */

/**
 * Type for scaling.
 * This type is needed for scaling the graph output.
 */

typedef struct xmm_ControlScale_s
{
    int				width;		/**< New width of the output image */
    int				height;		/**< New Height of the output image */
    int				flags;		/**< Flags of the #XMM_GRAPH_RESIZE group */

    /** Width of the destination buffer. \note Only used for graph conversion */
    int				bwidth;	
    /** Height of the destination buffer. \note Only used for graph conversion */
    int				bheight;
} XMM_ControlScale;

/**
 * Rectangle.
 * This type is needed to set sizes and positions.
 */

typedef struct xmm_ControlRect_s
{
    int				x;		/**< X-Ordinate */
    int				y;		/**< Y-Ordinate */
    int				width;		/**< Width */
    int				height;		/**< Height */
} XMM_ControlRect;

/**
 * Surface.
 * This type is needed for direct graph surface writing.
 */

typedef struct xmm_ControlSurface_s
{
    uint32_t			fourcc;		/**< Type of the surface */
    uint8_t			*data[3];	/**< Data pointers */
    int				stride[3];	/**< Bytes per line of the internal buffer */
} XMM_ControlSurface;

#ifdef __cplusplus
}
#endif

#endif
