/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpcodeca.c
 * Audio codec manager/wrapper
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libxmm/xmmp.h"
#include "libxmm/lpcodeca.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/list.h"
#include "../xmmpriv.h"

/*
 * Open codec.
 */
XMM_PluginCodecAudio *xmm_CodecAudioOpen( void *_xmm, int mode, XMM_AudioFormat *af )
{
  XMM_List		*le;
  XMM_PluginCodecAudio	*pACodec = NULL, *tACodec = NULL;
  XMM			*xmm = (XMM *) _xmm;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( af == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() url = NULL\n" );
	return NULL;
  }

  if( xmm->pCodecList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() pCodecList: libXMM not initialized or no codec plugins found ?\n" );
	return NULL;
  }

  /* Find plugin */
  for( le = xmm->pCodecList; le; le = le->next )
  {
	/* Only audio codecs */
	if( !(((XMM_Plugin *)le->data)->Flags & XMM_PLUGIN_TYPE_ACODEC ))	continue;

	if(((XMM_PluginCodecAudio *)le->data)->Open( xmm, mode | XMM_CODEC_MODE_QUERY, af ) == NULL )
	{
		tACodec = (XMM_PluginCodecAudio *)le->data;
		break;
	}
  }

  if( tACodec == NULL || ( le == NULL ))
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() No matching audio codec plugin found ( formatID = 0x%x, mode %i )", af->formatID, mode );
	return NULL;
  }

  /* Initialize  */
  if(( pACodec = tACodec->Open( xmm, mode, af )) == NULL )
  {
	return NULL;
  }

  return pACodec;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Close codec.
 */
int xmm_CodecAudioClose( XMM_PluginCodecAudio *codec )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  codec->Close( codec );

  return XMM_RET_OK;
}

/*
 * Control call.
 */
int xmm_CodecAudioControl( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  return codec->Control( codec, cmd, arg );
}

/*
 * Decode data.
 */
int xmm_CodecAudioDecode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  return codec->Decode( codec, src, isize, dest, samples, flags );
}

/*
 * Encode data.
 */
int xmm_CodecAudioEncode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  return codec->Encode( codec, src, samples, dest, osize, flags );
}

/*
 * Get codec info.
 */
XMM_CodecAudioInfo *xmm_CodecAudioInfo( XMM_PluginCodecAudio *codec, void *xmm )
{
  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( codec == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() codec == NULL\n" );
	return NULL;
  }

  return codec->Info( xmm );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ------------------------- Misc functions ------------------------ * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Get Information about installed audio codecs
 */
XMM_List *xmm_CodecAudioInfoList( void *_xmm, uint32_t flags )
{
  XMM_List		*le, *ailist = NULL;
  XMM_CodecAudioInfo	*ai;
  XMM			*xmm = (XMM *) _xmm;
  char			*ptr, *tmp;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( xmm->pCodecList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() pCodecList: libXMM not initialized or no codec plugins found ?\n" );
	return NULL;
  }

  /* Find plugins */
  for( le = xmm->pCodecList; le; le = le->next )
  {
	/* Only audio codecs */
	if( !(((XMM_Plugin *)le->data)->Flags & XMM_PLUGIN_TYPE_ACODEC ))	continue;

	/* Get codec info */
	ai = ((XMM_PluginCodecAudio *)le->data)->Info( xmm );

	/* If codec according to flags, fill file member and appene codec */
	if( ai && (( ai->caps & flags ) == flags ))
	{
		tmp = strdup( ((XMM_Plugin *)le->data)->Filename );

		ptr = strrchr( tmp, '/' );
		if( ptr )
		{
			if( strlen( ptr + 1 ) > 63 )	ptr[63] = '\0';
			strcpy( ai->file, ptr + 1 );
		}
		else
		{
			if( strlen( tmp ) > 63 )	tmp[63] = '\0';
			strcpy( ai->file, tmp );
		}

		free( tmp );

		ailist = xmmList_Append( ailist, ai );
	}
  }

  return ailist;
}
