/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpcodecv.c
 * Video codec manager/wrapper
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libxmm/xmmp.h"
#include "libxmm/lpcodecv.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/list.h"
#include "../xmmpriv.h"

/*
 * Open file.
 */
XMM_PluginCodecVideo *xmm_CodecVideoOpen( void *_xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_List		*le;
  XMM_PluginCodecVideo	*pVCodec = NULL, *tVCodec = NULL;
  XMM			*xmm = (XMM *) _xmm;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( vf == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() url = NULL\n" );
	return NULL;
  }

  if( xmm->pCodecList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_NULL, __FUNCTION__ "() pCodecList: libXMM not initialized or no codec plugins found ?\n" );
	return NULL;
  }

  /* Find plugin */
  for( le = xmm->pCodecList; le; le = le->next )
  {
	/* Only video codecs */
	if( !(((XMM_Plugin *)le->data)->Flags & XMM_PLUGIN_TYPE_VCODEC ))	continue;

	if(((XMM_PluginCodecVideo *)le->data)->Open( xmm, mode | XMM_CODEC_MODE_QUERY, vf ) == NULL )
	{
		tVCodec = (XMM_PluginCodecVideo *)le->data;
		break;
	}
  }

  if( tVCodec == NULL || ( le == NULL ))
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() No matching video codec plugin found ( FourCC = 0x%x [%s] )", vf->codec, xmm_FOURCC_string( vf->codec ));
	return NULL;
  }

  /* Initialize  */
  if(( pVCodec = tVCodec->Open( xmm, mode, vf )) == NULL )
  {
	return NULL;
  }

  return pVCodec;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Close file.
 */
int xmm_CodecVideoClose( XMM_PluginCodecVideo *codec )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  codec->Close( codec );

  return XMM_RET_OK;
}

/*
 * Control call.
 */
int xmm_CodecVideoControl( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  return codec->Control( codec, cmd, arg );
}

/*
 * Decode data.
 */
int xmm_CodecVideoDecode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  return codec->Decode( codec, src, isize, dest, osize, flags );
}

/*
 * Encode data.
 */
int xmm_CodecVideoEncode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  if( codec == NULL )	return XMM_RET_INVALID_ARG;

  return codec->Encode( codec, src, isize, dest, osize, flags );
}

/*
 * Get codec info.
 */
XMM_CodecVideoInfo *xmm_CodecVideoInfo( XMM_PluginCodecVideo *codec, void *xmm )
{
  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( codec == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() codec == NULL\n" );
	return NULL;
  }

  return codec->Info( xmm );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ------------------------- Misc functions ------------------------ * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Get Information about installed video codecs
 */
XMM_List *xmm_CodecVideoInfoList( void *_xmm, uint32_t flags )
{
  XMM_List		*le, *vilist = NULL;
  XMM_CodecVideoInfo	*vi;
  XMM			*xmm = (XMM *) _xmm;
  char			*ptr, *tmp;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( xmm->pCodecList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() pCodecList: libXMM not initialized or no codec plugins found ?\n" );
	return NULL;
  }

  /* Find plugins */
  for( le = xmm->pCodecList; le; le = le->next )
  {
	/* Only video codecs */
	if( !(((XMM_Plugin *)le->data)->Flags & XMM_PLUGIN_TYPE_ACODEC ))	continue;

	/* Get codec info */
	vi = ((XMM_PluginCodecVideo *)le->data)->Info( xmm );

	/* If codec according to flags, fill file member and appene codec */
	if( vi && (( vi->caps & flags ) == flags ))
	{
		tmp = strdup( ((XMM_Plugin *)le->data)->Filename );

		ptr = strrchr( tmp, '/' );
		if( ptr )
		{
			if( strlen( ptr + 1 ) > 63 )	ptr[63] = '\0';
			strcpy( vi->file, ptr + 1 );
		}
		else
		{
			if( strlen( tmp ) > 63 )	tmp[63] = '\0';
			strcpy( vi->file, tmp );
		}

		free( tmp );

		vilist = xmmList_Append( vilist, vi );
	}
  }

  return vilist;
}
