/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Resampling and Format conversion code is taken from ALSA Project
 * ( www.alsa-project.org )
 */

/*
 * Channel conversion
 */

typedef void (*channel_t)( void *src, void *dst, int size );

#define DIVIDE_FUNC( name, type ) \
static void divide_##name( void *src, void *dst, int size ) \
{ \
  type *src_ptr = (type *)src; \
  type *dst_ptr = (type *)dst; \
 \
  while( size-- > 0 ) \
  { \
	*dst_ptr++ = *src_ptr; \
	*dst_ptr++ = *src_ptr++; \
  } \
}

DIVIDE_FUNC( 8bit, char )
DIVIDE_FUNC( 16bit, short )

#define MERGE_FUNC( name, type, min, max ) \
static void merge_##name( void *src, void *dst, int size ) \
{ \
  int tmp; \
  type *src_ptr = (type *)src; \
  type *dst_ptr = (type *)dst; \
 \
  while( size-- > 0 ) \
  { \
	tmp = ((int)src_ptr[0] + (int)src_ptr[1] ) / 2; \
 \
	if( tmp < min )	tmp = min; \
	else if( tmp > max )	tmp = max; \
	*dst_ptr++ = tmp; \
	src_ptr += 2; \
  } \
}

MERGE_FUNC( 8bit_signed, signed char, -128, 127 )
MERGE_FUNC( 16bit_signed, signed short, -32768, 32767 )
MERGE_FUNC( 8bit_unsigned, unsigned char, 0, 255 )
MERGE_FUNC( 16bit_unsigned, unsigned short, 0, 65535 )

/*
 * Format conversion ( ALSA: kernel/plugin/linear.c )
 */

typedef void (*convert_t)( void *src, void *dst, size_t size );

#define CONV_FUNC(name, srctype, dsttype, val) \
static void conv_##name(void *src_ptr, void *dst_ptr, size_t size) \
{ \
	srctype *srcp = src_ptr; \
	dsttype *dstp = dst_ptr; \
	while (size--) { \
		srctype src = *srcp++; \
		*dstp++ = val; \
	} \
}

CONV_FUNC(8_sign, u_int8_t, u_int8_t, src ^ 0x80)

CONV_FUNC(8_16, u_int8_t, u_int16_t, (u_int16_t)src << 8)
CONV_FUNC(8_16_end, u_int8_t, u_int16_t, (u_int16_t)src)
CONV_FUNC(8_16_sign, u_int8_t, u_int16_t, (u_int16_t)(src ^ 0x80) << 8)
CONV_FUNC(8_16_sign_end, u_int8_t, u_int16_t, (u_int16_t)src ^ 0x80)

CONV_FUNC(16_8, u_int16_t, u_int8_t, src >> 8)
CONV_FUNC(16_end_8, u_int16_t, u_int8_t, src)
CONV_FUNC(16_8_sign, u_int16_t, u_int8_t, (src >> 8) ^ 0x80)
CONV_FUNC(16_end_8_sign, u_int16_t, u_int8_t, src ^ 0x80)

CONV_FUNC(16_sign, u_int16_t, u_int16_t, src ^ 0x8000)
CONV_FUNC(16_end, u_int16_t, u_int16_t, bswap_16(src))
CONV_FUNC(16_end_sign, u_int16_t, u_int16_t, bswap_16(src) ^ 0x8000)
CONV_FUNC(16_sign_end, u_int16_t, u_int16_t, bswap_16(src ^ 0x8000))
CONV_FUNC(16_end_sign_end, u_int16_t, u_int16_t, src ^ 0x80)

static convert_t convert_functions[32] =
{
 	NULL,			/* 8->8: Nothing to do */
	conv_8_sign,		/* 8->8 sign: conv_8_sign */
	NULL,			/* 8->8 dst_end: Nothing to do */
	conv_8_sign,		/* 8->8 dst_end sign: conv_8_sign */
	NULL,			/* 8->8 src_end: Nothing to do */
	conv_8_sign,		/* 8->8 src_end sign: conv_8_sign */
	NULL,			/* 8->8 src_end dst_end: Nothing to do */
	conv_8_sign,		/* 8->8 src_end dst_end sign: conv_8_sign */
	conv_8_16,		/* 8->16: conv_8_16 */
	conv_8_16_sign,		/* 8->16 sign: conv_8_16_sign */
	conv_8_16_end,		/* 8->16 dst_end: conv_8_16_end */
	conv_8_16_sign_end,	/* 8->16 dst_end sign: conv_8_16_sign_end */
	conv_8_16,		/* 8->16 src_end: conv_8_16 */
	conv_8_16_sign,		/* 8->16 src_end sign: conv_8_16_sign */
	conv_8_16_end,		/* 8->16 src_end dst_end: conv_8_16_end */
	conv_8_16_sign_end,	/* 8->16 src_end dst_end sign: conv_8_16_sign_end */
	conv_16_8,		/* 16->8: conv_16_8 */
	conv_16_8_sign,		/* 16->8 sign: conv_16_8_sign */
	conv_16_8,		/* 16->8 dst_end: conv_16_8 */
	conv_16_8_sign,		/* 16->8 dst_end sign: conv_16_8_sign */
	conv_16_end_8,		/* 16->8 src_end: conv_16_end_8 */
	conv_16_end_8_sign,	/* 16->8 src_end sign: conv_16_end_8_sign */
	conv_16_end_8,		/* 16->8 src_end dst_end: conv_16_end_8 */
	conv_16_end_8_sign,	/* 16->8 src_end dst_end sign: conv_16_end_8_sign */
	NULL,			/* 16->16: Nothing to do */
	conv_16_sign,		/* 16->16 sign: conv_16_sign */
	conv_16_end,		/* 16->16 dst_end: conv_16_end */
	conv_16_sign_end,	/* 16->16 dst_end sign: conv_16_sign_end */
	conv_16_end,		/* 16->16 src_end: conv_16_end */
	conv_16_end_sign,	/* 16->16 src_end sign: conv_16_end_sign */
	NULL,			/* 16->16 src_end dst_end: Nothing to do */
	conv_16_end_sign_end,	/* 16->16 src_end dst_end sign: conv_16_end_sign_end */
};

/*
 * Resampling ( ALSA: kernel/plugin/rate.c )
 */

/*
 * Definitions
 */

#define SHIFT			11
#define BITS			(1 << SHIFT)
#define MASK			(BITS - 1)

#define MAX_VOICES		6

/*
 * Types
 */

struct priv_alsa_csr_s
{
    unsigned int		pitch;
    unsigned int		pos;
    signed short		last_S1[MAX_VOICES];
    signed short		last_S2[MAX_VOICES];
};

typedef void (*resample16_t)( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size );
typedef void (*resample8_t)( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size );

/*
 * Code
 */

/*
 * Shrink ( 16 bit )
 */

static void resample16_shrink( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  signed short *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	while( size > 0 )
	{
		S1 = S2;

		if(( src - src_ptr ) < ( src_size * voices ))
		{
			S2 = *src;
			src += voices;
		}

		if( pos >> SHIFT )
		{
			pos &= MASK;
			val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

			if( val < -32768 )	val = -32768;
			else if( val > 32767 )	val = 32767;

    			*dst = val;
			dst += voices;
			size--;
		}
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}

/*
 * Expand ( 16 bit )
 */

static void resample16_expand( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  signed short *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	if( pos >> SHIFT )
	{
		pos &= MASK;
		S1 = S2;
		S2 = *src;
	}

	while( size-- > 0 )
	{
		if( pos >> SHIFT )
		{
			src += voices;
			pos &= MASK;
			S1 = S2;
			if(( src - src_ptr ) < src_size * voices)	S2 = *src;
		}

		val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

		if( val < -32768 )	val = -32768;
		else if( val > 32767 )	val = 32767;

		*dst = val;
		dst += voices;
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}

/*
 * Shrink ( 8 bit )
 */


static void resample8_shrink( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  unsigned char *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	while( size > 0 )
	{
		S1 = S2;

		if(( src - src_ptr ) < ( src_size * voices ))
		{
			S2 = (*src << 8) ^ 0x8000;
			src += voices;
		}

		if( pos >> SHIFT )
		{
			pos &= MASK;
			val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

			if( val < -32768 )	val = -32768;
			else if( val > 32767 )	val = 32767;

			*dst = (val >> 8) ^ 0x0080;
			dst += voices;
			size--;
		}
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}

/*
 * Expand ( 8 bit )
 */

static void resample8_expand( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  unsigned char *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	if( pos >> SHIFT )
	{
		pos &= MASK;
		S1 = S2;
		S2 = (*src << 8) ^ 0x8000;
	}

	while( size-- > 0 )
	{
		if( pos >> SHIFT )
		{
			src += voices;
			pos &= MASK;
			S1 = S2;
			if(( src - src_ptr ) < src_size * voices )
				S2 = (*src << 8) ^ 0x8000;
		}

		val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

		if( val < -32768 )	val = -32768;
		else if( val > 32767 )	val = 32767;

		*dst = (val >> 8) ^ 0x0080;
		dst += voices;
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}
