/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpfiltera.c
 * Manage Audio Filter Chain ( and provide internal plugins for conversion )
 */

#include <stdio.h>
#include <stdlib.h>
#include <byteswap.h>

#include "libxmm/xmmp.h"
#include "libxmm/version.h"
#include "libxmm/lpsound.h"
#include "libxmm/lpfiltera.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/mutex.h"
#include "../xmmpriv.h"

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/*
 * Types
 */

struct priv_audio_filter_s
{
    XMM_Mutex			*mutex;

    /* internal convesion plugins */
    XMM_PluginFilterAudio	*cAF_cfmt, *pAF_cfmt;
    XMM_PluginFilterAudio	*cAF_ccn, *pAF_ccn;
    XMM_PluginFilterAudio	*cAF_csr, *pAF_csr;

    /* conversion paramters */
    XMM_SoundFormat		ssf;
    XMM_SoundFormat		dsf;

    /* filter list */
    XMM_List			*cAFlist;
    XMM_List			*pAFlist;
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info_cfmt;
extern XMM_PluginFilterAudio	plugin_info_ccn;
extern XMM_PluginFilterAudio	plugin_info_csr;

/*
 * Code
 */

/*
 * Open filter
 */
XMM_PluginFilterAudio *xmm_FilterAudioOpen( void *_xmm, char *filename, XMM_SoundFormat *saf, XMM_SoundFormat *daf, uint32_t isize, uint32_t *osize )
{
  XMM_PluginFilterAudio	*pFA = NULL, *tFA;
  XMM			*xmm = (XMM *) _xmm;

  if( filename == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filename == NULL\n" );
	return NULL;
  }

  /* Register plugin */
  if(( tFA = (XMM_PluginFilterAudio *)xmm_PluginRegister( filename )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to load plugin '%s'\n", filename );
	return NULL;
  }

  /* Initialize plugin */
  pFA = tFA->Open( xmm, saf, daf, isize, osize );
  if( pFA == NULL )	return NULL;

  return pFA;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Close filter
 */
int xmm_FilterAudioClose( XMM_PluginFilterAudio *filter )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Close( filter );
}

/*
 * Process data
 */
int xmm_FilterAudioProc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Proc( filter, sdata, ddata );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Audio Filter Chain ---------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Initialize Filter chain/subsystem
 */
int xmm_FilterAChainInit( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  if( xmm->pAFpriv != NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Audio filter chain already initialized" );
	return XMM_RET_ERROR;
  }

  /* Allocate private data */
  if(( xmm->pAFpriv = malloc( sizeof( struct priv_audio_filter_s ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to allocate memory for audio filter chain" );
	return XMM_RET_ERROR;
  }
  memset( xmm->pAFpriv, 0, sizeof( struct priv_audio_filter_s ));

  priv = xmm->pAFpriv;

  /* Load internal plugins */
  priv->cAF_cfmt = (XMM_PluginFilterAudio *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_cfmt );
  if( priv->cAF_cfmt == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  priv->cAF_ccn = (XMM_PluginFilterAudio *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_ccn );
  if( priv->cAF_ccn == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  priv->cAF_csr = (XMM_PluginFilterAudio *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_csr );
  if( priv->cAF_csr == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  /* Create mutex */
  priv->mutex = xmmMutex_Create();

  return XMM_RET_OK;
}

/*
 * Free Filter chain/subsystem
 */
int xmm_FilterAChainExit( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Stop the unstopped chain ( only to be sure, if plugin doesn't stop it ) */
  xmm_FilterAChainStop( xmm );

  /* Free internal plugins */
  if( priv->cAF_cfmt )		xmm_PluginRemove( (XMM_Plugin *)priv->cAF_cfmt );
  if( priv->cAF_ccn )		xmm_PluginRemove( (XMM_Plugin *)priv->cAF_ccn );
  if( priv->cAF_csr )		xmm_PluginRemove( (XMM_Plugin *)priv->cAF_csr );

  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex );

  /* Free private chain data */
  free( xmm->pAFpriv );
  xmm->pAFpriv = NULL;

  return XMM_RET_OK;
}

/*
 * Lock audio filter chain
 */
int xmm_FilterAChainLock( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  return xmmMutex_Lock( priv->mutex );
}

/*
 * Unlock audio filter chain
 */
int xmm_FilterAChainUnlock( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  return xmmMutex_Unlock( priv->mutex );
}

/*
 * Check if the audio filter chain is active
 */
int xmm_FilterAChainActive( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Check for conversion */
  if( priv->ssf.samprate != priv->dsf.samprate )	return 1;
  if( priv->ssf.format != priv->dsf.format )		return 1;
  if( priv->ssf.channels != priv->dsf.channels )	return 1;

  /* Check for filters */
  if( priv->pAFlist != NULL )	return 1;

  return 0;
}

/*
 * Set Input / Source format
 */
int xmm_FilterAChainInput( void *_xmm, int samprate, int channels, int format )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  priv->ssf.samprate = priv->dsf.samprate = samprate;
  priv->ssf.format = priv->dsf.format = format;
  priv->ssf.channels = priv->dsf.channels = channels;

  return XMM_RET_OK;
}

/*
 * Set Output / Target format and buffer size
 * Returns: needed number of samples ( of source data )
 */
int xmm_FilterAChainOutput( void *_xmm, int samprate, int channels, int format, int blocksize )
{
  struct priv_audio_filter_s	*priv;
  int				nblocksize = blocksize;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;
  XMM_PluginFilterAudio		*tAudioFilter;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Stop the unstopped chain ( only to be sure, if plugin doesn't stop it ) */
  xmm_FilterAChainStop( xmm );

  /* Lock audio chain, filter list may not be changed sound plugin unlocks the chain */
  xmm_FilterAChainLock( xmm );

  /* Initialize dest format values */
  if( format == -1 )	priv->dsf.format = priv->ssf.format;
  else	priv->dsf.format = format;

  if( channels == -1 )	priv->dsf.channels = priv->ssf.channels;
  else	priv->dsf.channels = channels;

  if( samprate == -1 )	priv->dsf.samprate = priv->ssf.samprate;
  else	priv->dsf.samprate = samprate;

  /*
   * We need to do this in reverse order, the last conversions output must be
   * blocksize
   */

  /* Process Filter List */
  for( le = priv->cAFlist; le; le = le->next )
  {
	tAudioFilter = ((XMM_PluginFilterAudio *)le->data)->Open( xmm, &priv->dsf, NULL, nblocksize, NULL );
	if( tAudioFilter == NULL )
	{
	    xmm_logging( 1, "FAC! WARNING: Unable to open filter '%s'\n", ((XMM_Plugin *)le->data)->Name );
	    continue;
	}

	priv->pAFlist = xmmList_Append( priv->pAFlist, (void *)tAudioFilter );
  }

  /* Samprate */
  if( priv->ssf.samprate != priv->dsf.samprate )
  {
	if(( priv->pAF_csr = priv->cAF_csr->Open( xmm, &priv->ssf, &priv->dsf, nblocksize, &nblocksize )) == NULL )
	    return XMM_RET_ERROR;
  }

  /* Channels */
  if( priv->ssf.channels != priv->dsf.channels )
  {
	if(( priv->pAF_ccn = priv->cAF_ccn->Open( xmm, &priv->ssf, &priv->dsf, nblocksize, &nblocksize )) == NULL )
	    return XMM_RET_ERROR;
  }

  /* Format */
  if( priv->ssf.format != priv->dsf.format )
  {
	if(( priv->pAF_cfmt = priv->cAF_cfmt->Open( xmm, &priv->ssf, &priv->dsf, nblocksize, &nblocksize )) == NULL )
	    return XMM_RET_ERROR;
  }

  xmm_logging( 2, "FAC! Input: %i Hz, %i Channels, %i Bits ( Size = %i )\n", priv->ssf.samprate, priv->ssf.channels, priv->ssf.format & XMM_SOUND_MASK_SIZE, nblocksize );
  xmm_logging( 2, "FAC! Output: %i Hz, %i Channels, %i Bits ( Size = %i )\n", priv->dsf.samprate, priv->dsf.channels, priv->dsf.format & XMM_SOUND_MASK_SIZE, blocksize );

  return nblocksize / ( priv->ssf.channels * ( priv->ssf.format & XMM_SOUND_MASK_SIZE ) / 8 );
}

/*
 * Process filter chain
 */
int xmm_FilterAChain( void *_xmm, uint8_t *sdata, uint32_t isamples, uint8_t **ddata, uint32_t *osamples )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Sample format conversion */
  if( priv->pAF_cfmt )	priv->pAF_cfmt->Proc( priv->pAF_cfmt, sdata, &sdata );

  /* Channel conversion */
  if( priv->pAF_ccn )	priv->pAF_ccn->Proc( priv->pAF_ccn, sdata, &sdata );

  /* Do rate resampling */
  if( priv->pAF_csr )	priv->pAF_csr->Proc( priv->pAF_csr, sdata, &sdata );

  /* Process Filter List */
  for( le = priv->pAFlist; le; le = le->next )
  {
	printf( "Doing '%s'\n", ((XMM_Plugin *)le->data)->Name );
	((XMM_PluginFilterAudio *)le->data)->Proc((XMM_PluginFilterAudio *)le->data, sdata, &sdata );
  }

  /* Calculate number of output samples */
  *osamples = isamples * priv->dsf.samprate / priv->ssf.samprate;

  *ddata = sdata;
  return XMM_RET_OK;
}

/*
 * Stop chain ( will stop all stuff started in xmm_FilterAChainOutput()
 */
int xmm_FilterAChainStop( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Free existing conversion filters */
  if( priv->pAF_csr )	priv->pAF_csr->Close( priv->pAF_csr );
  priv->pAF_csr = NULL;
  if( priv->pAF_ccn )	priv->pAF_ccn->Close( priv->pAF_ccn );
  priv->pAF_ccn = NULL;
  if( priv->pAF_cfmt )	priv->pAF_cfmt->Close( priv->pAF_cfmt );
  priv->pAF_cfmt = NULL;

  /* Free filters */
  for( le = priv->pAFlist; le; le = le->next )
  {
	((XMM_PluginFilterAudio *)le->data)->Close((XMM_PluginFilterAudio *)le->data );
  }
  xmmList_Free( priv->pAFlist );
  priv->pAFlist = NULL;

  /* Unlock audio chain */
  xmm_FilterAChainUnlock( xmm );

  return XMM_RET_OK;
}

/*
 * Add plugin to list
 */
int xmm_FilterAChainAdd( void *_xmm, XMM_PluginFilterAudio *pfa )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  if( pfa == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filter plugin == NULL\n" );
	return XMM_RET_ERROR;
  }

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  /* Add plugin */
  priv->cAFlist = xmmList_Append( priv->cAFlist, (void *)pfa );

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Add list to filter list
 */
int xmm_FilterAChainAddList( void *_xmm, XMM_List *_pAudioFilterList )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  if( _pAudioFilterList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filter plugin list == NULL\n" );
	return XMM_RET_ERROR;
  }

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  for( le = _pAudioFilterList; le; le = le->next )
	xmm_FilterAChainAdd( xmm, (XMM_PluginFilterAudio *)le->data );

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Remove filters from chain
 */
int xmm_FilterAChainFree( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Audio Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  xmmList_Free( priv->cAFlist );
  priv->cAFlist = NULL;

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/* * * * * * * * *
 * Internal code *
 * * * * * * * * */

#include "lpfiltera-alsa.h"

/*
 * Sample format converter
 * ( internal plugin )
 */

/*
 * Prototypes
 */

static XMM_PluginFilterAudio *cfmt_Open( void *_xmm, XMM_SoundFormat *ssf, XMM_SoundFormat *dsf, uint32_t isize, uint32_t *osize );
static int cfmt_Close( XMM_PluginFilterAudio *filter );
static int cfmt_Proc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata );

/*
 * Plugin info
 */

static XMM_PluginFilterAudio	plugin_info_cfmt =
				    { { NULL,
				    XMM_PLUGIN_ID,
				    XMM_PLUGIN_TYPE_AFILTER,
				    0,
				    XMM_VERSION_NUM,
				    "",
				    "STDACONV-FORMAT",
				    "libxmm sample format converter",
				    "Copyright (c) 2001 by Arthur Kleer",
				    NULL, NULL },
				    cfmt_Open, cfmt_Close, cfmt_Proc };

/*
 * Private type
 */

struct priv_cfmt_t
{
    convert_t			convert;
    void			*data;
    uint32_t			dSS;
    uint32_t			sSS;
    uint32_t			channels;
    uint32_t			tsamples;
};

/*
 * Code
 */

static XMM_PluginFilterAudio *cfmt_Open( void *_xmm, XMM_SoundFormat *ssf, XMM_SoundFormat *dsf, uint32_t isize, uint32_t *osize )
{
  XMM_PluginFilterAudio	*pAFilter;
  struct priv_cfmt_t	*priv;
  XMM			*xmm = (XMM *) _xmm;

  /* Allocate plugin data */
  if(( pAFilter = xmm_memdup_x( &plugin_info_cfmt, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_cfmt_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_cfmt_t *) &pAFilter[1];
  pAFilter->sys.priv = (void *) priv;
  pAFilter->sys.xmm = xmm;

  /* Save data */
  priv->sSS = ssf->format & XMM_SOUND_MASK_SIZE;
  priv->dSS = dsf->format & XMM_SOUND_MASK_SIZE;
  priv->channels = ssf->channels;
  priv->tsamples = isize / ( priv->dSS / 8 );	/* Number of (mono) samples in all channels */

  /* Set conversion function */
  priv->convert = ((convert_t(*)[2][2][2][2])convert_functions)
	    [(( ssf->format & XMM_SOUND_MASK_SIZE ) == 16 ) ? 1 : 0 ]
	    [(( dsf->format & XMM_SOUND_MASK_SIZE ) == 16 ) ? 1 : 0 ]
	    [( ssf->format & XMM_SOUND_MASK_BE ) ? 1 : 0 ]
	    [( dsf->format & XMM_SOUND_MASK_BE ) ? 1 : 0 ]
	    [( ssf->format & XMM_SOUND_MASK_SIGNED ) != ( dsf->format & XMM_SOUND_MASK_SIGNED )];

  xmm_logging( 3, "FAC-cfmt! Allocating Buffer: Size = %i\n", isize );

  /* Allocate conversion buffer */
  if(( priv->data = malloc( isize )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() conversion buffer" );
	free( pAFilter );
	return NULL;
  }

  /* Set output size */
  *osize = isize * ( ssf->format & XMM_SOUND_MASK_SIZE ) / ( dsf->format & XMM_SOUND_MASK_SIZE );

  return pAFilter;
}

static int cfmt_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_cfmt_t	*priv = (struct priv_cfmt_t *)filter->sys.priv;

  /* Free data */
  if( priv->data )	free( priv->data );

  /* Free plugin */
  free( filter );
  return XMM_RET_OK;
}

static int cfmt_Proc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata )
{
  struct priv_cfmt_t	*priv = (struct priv_cfmt_t *)filter->sys.priv;

#ifdef DEBUG
  xmm_logging( 1, "FAC-cfmt! Input %i bytes ( %i samples )\n", priv->tsamples * priv->sSS / 8, priv->tsamples * priv->channels );
#endif

  priv->convert( sdata, priv->data, priv->tsamples );
  
  *ddata = priv->data;

#ifdef DEBUG
  xmm_logging( 1, "FAC-cfmt! Output %i bytes ( %i samples )\n", priv->tsamples * priv->dSS / 8, priv->tsamples * priv->channels );
#endif

  return XMM_RET_OK;
}

/*
 * Channel number converter
 * ( internal plugin )
 */

/*
 * Prototypes
 */

static XMM_PluginFilterAudio *ccn_Open( void *_xmm, XMM_SoundFormat *ssf, XMM_SoundFormat *dsf, uint32_t isize, uint32_t *osize );
static int ccn_Close( XMM_PluginFilterAudio *filter );
static int ccn_Proc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata );

/*
 * Plugin info
 */

static XMM_PluginFilterAudio	plugin_info_ccn =
				    { { NULL,
				    XMM_PLUGIN_ID,
				    XMM_PLUGIN_TYPE_AFILTER,
				    0,
				    XMM_VERSION_NUM,
				    "",
				    "STDACONV-CHANNEL",
				    "libxmm channel number converter",
				    "Copyright (c) 2001 by Arthur Kleer",
				    NULL, NULL },
				    ccn_Open, ccn_Close, ccn_Proc };

/*
 * Private type
 */

struct priv_ccn_t
{
    channel_t			channel;
    void			*data;
    uint32_t			dSS;
    uint16_t			dChannels;
    uint16_t			sChannels;
    uint32_t			samples;
};

/*
 * Code
 */

static XMM_PluginFilterAudio *ccn_Open( void *_xmm, XMM_SoundFormat *ssf, XMM_SoundFormat *dsf, uint32_t isize, uint32_t *osize )
{
  XMM_PluginFilterAudio	*pAFilter;
  struct priv_ccn_t	*priv;
  XMM			*xmm = (XMM *) _xmm;

  /* Allocate plugin data */
  if(( pAFilter = xmm_memdup_x( &plugin_info_ccn, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_ccn_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_ccn_t *) &pAFilter[1];
  pAFilter->sys.priv = (void *) priv;
  pAFilter->sys.xmm = xmm;

  /* Save data */
  priv->dSS = dsf->format & XMM_SOUND_MASK_SIZE;
  priv->dChannels = dsf->channels;
  priv->sChannels = ssf->channels;
  priv->samples = isize / ( priv->dChannels * priv->dSS / 8 );

  /* Initialize conversion */
  if(( dsf->format & XMM_SOUND_MASK_SIZE ) ==  16 )
  {
	if(( ssf->channels == 2 ) && ( dsf->channels == 1 ))	
	{
		if( dsf->format & XMM_SOUND_MASK_SIGNED )	priv->channel = merge_16bit_signed;
		else	priv->channel = merge_16bit_unsigned;
	}
	else	if(( ssf->channels == 1 ) && ( dsf->channels == 2 ))	priv->channel = divide_16bit;
  }
  else
  {
	if(( ssf->channels == 2 ) && ( dsf->channels == 1 ))	
	{
		if( dsf->format & XMM_SOUND_MASK_SIGNED )	priv->channel = merge_8bit_signed;
		else	priv->channel = merge_8bit_unsigned;
	}
	else	if(( ssf->channels == 1 ) && ( dsf->channels == 2 ))	priv->channel = divide_8bit;
  }

  if( priv->channel == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unsupported channel conversion: %i --> %i at %i bit\n", ssf->channels, dsf->channels, ( dsf->format & XMM_SOUND_MASK_SIZE ));
	free( pAFilter );
	return NULL;
  }

  xmm_logging( 3, "FAC-ccn! Allocating Buffer: Size = %i\n", isize );

  /* Allocate conversion buffer */
  if(( priv->data = malloc( isize )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() conversion buffer" );
	free( pAFilter );
	return NULL;
  }

  /* Set output size */
  *osize = isize * ssf->channels / dsf->channels;

  return pAFilter;
}

static int ccn_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_ccn_t	*priv = (struct priv_ccn_t *)filter->sys.priv;

  /* Free data */
  if( priv->data )	free( priv->data );

  /* Free plugin */
  free( filter );
  return XMM_RET_OK;
}

static int ccn_Proc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata )
{
  struct priv_ccn_t	*priv = (struct priv_ccn_t *)filter->sys.priv;

#ifdef DEBUG
  xmm_logging( 1, "FAC-ccn! Input %i bytes ( %i samples )\n", priv->samples * priv->sChannels * priv->dSS / 8, priv->samples );
#endif

  priv->channel( sdata, priv->data, priv->samples );

  *ddata = priv->data;

#ifdef DEBUG
  xmm_logging( 1, "FAC-ccn! Output %i bytes ( %i samples )\n", priv->samples * priv->dChannels * priv->dSS / 8, priv->samples );
#endif

  return XMM_RET_OK;
}

/*
 * Sample rate converter
 * ( internal plugin )
 */

/*
 * Prototypes
 */

static XMM_PluginFilterAudio *csr_Open( void *_xmm, XMM_SoundFormat *ssf, XMM_SoundFormat *dsf, uint32_t isize, uint32_t *osize );
static int csr_Close( XMM_PluginFilterAudio *filter );
static int csr_Proc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata );

/*
 * Plugin info
 */

static XMM_PluginFilterAudio	plugin_info_csr =
				    { { NULL,
				    XMM_PLUGIN_ID,
				    XMM_PLUGIN_TYPE_AFILTER,
				    0,
				    XMM_VERSION_NUM,
				    "",
				    "STDACONV-SAMPLERATE",
				    "libxmm sample rate converter",
				    "Copyright (c) 2001 by Arthur Kleer",
				    NULL, NULL },
				    csr_Open, csr_Close, csr_Proc };

/*
 * Private type
 */

struct priv_csr_t
{
    struct priv_alsa_csr_s	alsa;
    void			*data;
    resample16_t		resample16;
    resample8_t			resample8;
    uint32_t			dSS;
    uint16_t			dChannels;
    uint32_t			isamples;
    uint32_t			osamples;
};

/*
 * Code
 */

static XMM_PluginFilterAudio *csr_Open( void *_xmm, XMM_SoundFormat *ssf, XMM_SoundFormat *dsf, uint32_t isize, uint32_t *osize )
{
  XMM_PluginFilterAudio	*pAFilter;
  struct priv_csr_t	*priv;
  XMM			*xmm = (XMM *) _xmm;
  int			ch;

  /* Allocate plugin data */
  if(( pAFilter = xmm_memdup_x( &plugin_info_csr, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_csr_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_csr_t *) &pAFilter[1];
  pAFilter->sys.priv = (void *) priv;
  pAFilter->sys.xmm = xmm;

  /* Save data */
  priv->dSS = dsf->format & XMM_SOUND_MASK_SIZE;
  priv->dChannels = dsf->channels;
  priv->osamples = isize / ( priv->dChannels * ( priv->dSS / 8 ));
  priv->isamples = ( isize * ssf->samprate / dsf->samprate ) / ( priv->dChannels * ( priv->dSS / 8 ));

  /* Initialize conversion */
  if( ssf->samprate < dsf->samprate )
  {
	priv->resample16 = resample16_expand;
	priv->resample8 = resample8_expand;
	priv->alsa.pitch = (( ssf->samprate << SHIFT ) + ( dsf->samprate >> 1 )) / dsf->samprate;
  }
  else
  {
	priv->resample16 = resample16_shrink;
	priv->resample8 = resample8_shrink;
	priv->alsa.pitch = (( dsf->samprate << SHIFT ) + ( ssf->samprate >> 1 )) / ssf->samprate;
  }

  priv->alsa.pos = 0;

  for( ch = 0; ch < dsf->channels; ch++ )
	priv->alsa.last_S1[ch] = priv->alsa.last_S2[ch] = 0;

  xmm_logging( 3, "FAC-csr! Allocating Buffer: Size = %i\n", isize );

  /* Allocate conversion buffer */
  if(( priv->data = malloc( isize )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() conversion buffer" );
	free( pAFilter );
	return NULL;
  }

  /* Set output size */
  *osize = isize * ssf->samprate / dsf->samprate;

  return pAFilter;
}

static int csr_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_csr_t	*priv = (struct priv_csr_t *)filter->sys.priv;

  /* Free data */
  if( priv->data )	free( priv->data );

  /* Free plugin */
  free( filter );
  return XMM_RET_OK;
}

static int csr_Proc( XMM_PluginFilterAudio *filter, uint8_t *sdata, uint8_t **ddata )
{
  struct priv_csr_t	*priv = (struct priv_csr_t *)filter->sys.priv;

#ifdef DEBUG
  xmm_logging( 1, "FAC-csr: Input %i bytes ( %i samples )\n", priv->isamples * priv->dSS * priv->dChannels / 8, priv->isamples );
#endif

  if( priv->dSS == 16 )
  {
	priv->resample16( &priv->alsa, priv->dChannels, (signed short *)sdata, priv->isamples,
			(signed short *)priv->data, priv->osamples );
  }
  else
  {
	priv->resample8( &priv->alsa, priv->dChannels, sdata, priv->isamples,
			priv->data, priv->osamples );
  }

  *ddata = priv->data;

#ifdef DEBUG
  xmm_logging( 1, "FAC-csr! Output %i bytes ( %i samples )\n", priv->osamples * priv->dSS * priv->dChannels / 8, priv->osamples );
#endif

  return XMM_RET_OK;
}
