/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpfilterv.c
 * Manage Video Filter Chain ( and provide internal plugins for conversion )
 *
 * TODO:
 *	- xmm_FilterVChainStop(): segmentation fault at free( priv->image[0] ).
 *	  Very strange! I was not able to find a reason. I don't think it
 *	  is caused by a overflow of priv->image[0] itself.
 */

#include <stdio.h>
#include <stdlib.h>
#include <byteswap.h>

#include "libxmm/xmmp.h"
#include "libxmm/version.h"
#include "libxmm/xmmctl.h"
#include "libxmm/lpgraph.h"
#include "libxmm/lpfilterv.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/mutex.h"
#include "../xmmpriv.h"

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/*
 * Definitions
 */

#define	divUP( a, b )		(((a) + (b) - 1 ) / (b))

/*
 * Types
 */

struct priv_video_filter_s
{
    XMM_Mutex			*mutex;

    /* internal / current conversion plugin */
    XMM_PluginFilterVideo	*cVF_cfmt_default, *pVF_cfmt;

    /* conversion paramters */
    XMM_ImageFormat		sif;
    XMM_ImageFormat		dif;

    /* filter list */
    XMM_List			*cVFlist;
    XMM_List			*pVFlist;

    /* image buffer */
    uint8_t			*image[3];
    int				image_bpp;
    int				image_size;
};

/*
 * Global data
 */

extern XMM_PluginFilterVideo	plugin_info_cfmt;

/*
 * Code
 */

/*
 * Open filter
 */
XMM_PluginFilterVideo *xmm_FilterVideoOpen( void *_xmm, char *filename, XMM_ImageFormat *sif, XMM_ImageFormat *dif )
{
  XMM_PluginFilterVideo	*pFV = NULL, *tFV;
  XMM			*xmm = (XMM *) _xmm;

  if( filename == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filename == NULL\n" );
	return NULL;
  }

  /* Register plugin */
  if(( tFV = (XMM_PluginFilterVideo *)xmm_PluginRegister( filename )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to load plugin '%s'\n", filename );
	return NULL;
  }

  /* Initialize plugin */
  pFV = tFV->Open( xmm, sif, dif );
  if( pFV == NULL )	return NULL;

  return pFV;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Close filter
 */
int xmm_FilterVideoClose( XMM_PluginFilterVideo *filter )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Close( filter );
}

/*
 * Control call
 */
int xmm_FilterVideoControl( XMM_PluginFilterVideo *filter, uint32_t cmd, uint32_t arg, void *data )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Control( filter, cmd, arg, data );
}

/*
 * Process data
 */
int xmm_FilterVideoProc( XMM_PluginFilterVideo *filter, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Proc( filter, sdata, stride, width, height, ddata, dx, dy );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Audio Filter Chain ---------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Initialize Filter chain/subsystem
 */
int xmm_FilterVChainInit( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  if( xmm->pVFpriv != NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Video filter chain already initialized" );
	return XMM_RET_ERROR;
  }

  /* Allocate private data */
  if(( xmm->pVFpriv = malloc( sizeof( struct priv_video_filter_s ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to allocate memory for video filter chain" );
	return XMM_RET_ERROR;
  }
  memset( xmm->pVFpriv, 0, sizeof( struct priv_video_filter_s ));

  priv = xmm->pVFpriv;

  /* Load internal plugins */
  priv->cVF_cfmt_default = (XMM_PluginFilterVideo *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_cfmt );
  if( priv->cVF_cfmt_default == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  /* Create mutex */
  priv->mutex = xmmMutex_Create();

  return XMM_RET_OK;
}

/*
 * Free Filter chain/subsystem
 */
int xmm_FilterVChainExit( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Stop the unstopped chain ( only to be sure, if plugin doesn't stop it ) */
  xmm_FilterVChainStop( xmm );

  /* Free internal / user plugins */
  if( priv->cVF_cfmt_default )	xmm_PluginRemove( (XMM_Plugin *)priv->cVF_cfmt_default );

  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex );

  /* Free private chain data */
  free( xmm->pVFpriv );
  xmm->pVFpriv = NULL;

  return XMM_RET_OK;
}

/*
 * Lock video filter chain
 */
int xmm_FilterVChainLock( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  return xmmMutex_Lock( priv->mutex );
}

/*
 * Unlock video filter chain
 */
int xmm_FilterVChainUnlock( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  return xmmMutex_Unlock( priv->mutex );
}

/*
 * Check if the video filter chain is active
 */
int xmm_FilterVChainActive( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Check for conversion */
  if( priv->pVF_cfmt )		return 1;

  /* Check for filters */
  if( priv->pVFlist != NULL )	return 1;

  return 0;
}

/*
 * Set Input / Source format and size
 */
int xmm_FilterVChainInput( void *_xmm, uint32_t format, int width, int height, int flags )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  priv->sif.codec = priv->dif.codec = format;
  priv->sif.width = priv->dif.width = width;
  priv->sif.height = priv->dif.height = height;
  priv->sif.flags = priv->dif.flags = flags;
  priv->sif.bwidth = priv->dif.bwidth = width;
  priv->sif.bheight = priv->dif.bheight = height;

  return XMM_RET_OK;
}

/*
 * Set Output / Destination format and size ( also buffer size )
 */
int xmm_FilterVChainOutput( void *_xmm, uint32_t format, int width, int height, int flags, int bwidth, int bheight )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_PluginFilterVideo		*tFV = NULL;
  XMM_ControlScale		scale;
  char				buffer[5];
  int				ret;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Stop the unstopped chain ( only to be sure, if plugin doesn't stop it ) */
  xmm_FilterVChainStop( xmm );

  /* Lock video chain, filter list may not be changed graph plugin unlocks the chain */
  xmm_FilterVChainLock( xmm );

  /* Initialize dest format values */
  priv->dif.codec = format;
  priv->dif.width = width;
  priv->dif.height = height;
  priv->dif.flags = flags;
  priv->dif.bwidth = bwidth;
  priv->dif.bheight = bheight;

  /* No conversion */
  if(( priv->sif.codec == priv->dif.codec ) &&
	(( priv->sif.flags & XMM_GRAPH_FLAG_YFLIP ) ==
	( priv->dif.flags & XMM_GRAPH_FLAG_YFLIP )) &&
	( priv->sif.width == priv->dif.width ) &&
	( priv->sif.height == priv->dif.height ))
  {
#ifdef DEBUG
	xmm_logging( 1, "FVC! No conversion ( %x [%s] )\n", priv->sif.codec, xmm_FOURCC_string( priv->sif.codec ));
#endif

	xmm_FilterVChainUnlock( xmm );
	return XMM_RET_OK;
  }

  /* Only conversion to RGB supported */
  if(( priv->dif.codec & XMM_GRAPH_FMT_RGB(0)) != XMM_GRAPH_FMT_RGB(0))
  {
	xmm_SetError( xmm, XMM_ERR_NOTSUPPORTED, __FUNCTION__ "() Conversion only to RGB supported: %x [%s]\n", priv->dif.codec, xmm_FOURCC_string( priv->dif.codec ));

	xmm_FilterVChainUnlock( xmm );
	return XMM_RET_ERROR;
  }

  /*
   * Image buffer needed
   */
  priv->image_bpp = divUP(( priv->dif.codec >> 24 ) & 0xFF, 8 );
  priv->image_size = priv->dif.bwidth * priv->dif.bheight * priv->image_bpp;

  if(( priv->image[0] = malloc( priv->image_size )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to allocate memory for video filter chain" );
	return XMM_RET_ERROR;
  }

  /*
   * Initialize / Load conversion filter plugin
   */

  if( xmm->gconv_plugin[0] != '\0' )
	tFV = (XMM_PluginFilterVideo *) xmm_PluginRegister( xmm->gconv_plugin );

  /* Try user conversion filter plugin */
  if( tFV  == NULL )
  {
#ifdef DEBUG
	xmm_logging( 1, "FVC! Unable to load conversion filter plugin '%s'", filename );
#endif
  }
  else
  {
	if(( priv->pVF_cfmt = tFV->Open( xmm, &priv->sif, &priv->dif )) == NULL )
	{
	    strcpy( buffer, xmm_FOURCC_string( priv->dif.codec ));
	    xmm_logging( 2, "FVC! Conversion not supported: %x [%s] --> %x [%s]\n", priv->sif.codec, xmm_FOURCC_string( priv->sif.codec ), priv->dif.codec, buffer );
	    xmm_PluginRemove( &tFV->sys );
	}
  }

  /* Use libxmm internal conversion filter */
  if( priv->pVF_cfmt == NULL )
  {
	xmm_logging( 2, "FVC! Using libXMM for conversion...\n" );

	/* Default Graphic Plugins */
	if(( priv->pVF_cfmt = priv->cVF_cfmt_default->Open( xmm, &priv->sif, &priv->dif )) == NULL )
	{
	    strcpy( buffer, xmm_FOURCC_string( priv->dif.codec ));
	    xmm_SetError( xmm, XMM_ERR_NOTSUPPORTED, __FUNCTION__ "() Conversion not supported: %x [%s] --> %x [%s]\n", priv->sif.codec, xmm_FOURCC_string( priv->sif.codec ), priv->dif.codec, buffer );

	    xmm_FilterVChainUnlock( xmm );
	    return XMM_RET_ERROR;
	}
  }

  scale.width = priv->dif.width;
  scale.height = priv->dif.height;
  scale.bwidth = priv->dif.bwidth;
  scale.bheight = priv->dif.bheight;
  ret = priv->pVF_cfmt->Control( priv->pVF_cfmt, XMM_CTLSET_SCALE, 0, &scale );
  if( ret != XMM_CTLRET_TRUE )	return ret;

  xmm_logging( 2, "FVC! Input: %x [%s] %ix%i\n", priv->sif.codec, xmm_FOURCC_string( priv->sif.codec ), priv->sif.width, priv->sif.height );
  xmm_logging( 2, "FVC! Output: %x [%s] %ix%i\n", priv->dif.codec, xmm_FOURCC_string( priv->dif.codec ), priv->dif.width, priv->dif.height );

  xmm_FilterVChainUnlock( xmm );
  return XMM_RET_OK;
}

/*
 *
 */
int xmm_FilterVChainData( void *_xmm, uint8_t *data[], int stride[], int x, int y, int width, int height, int flags )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  uint8_t			*src, *dest;
  int				sstride, dstride, i;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Conversion needed ? */
  if( priv->pVF_cfmt )
  {
  	priv->pVF_cfmt->Proc( priv->pVF_cfmt, data, stride, width, height, priv->image, x, y );
  }
  else
  {
	src = data[0];
	dest = priv->image[0] + ( y * priv->dif.bwidth + x ) * priv->image_bpp;

	width *= priv->image_bpp;
	dstride = priv->dif.bwidth * priv->image_bpp;
	sstride = (( stride == NULL ) ? priv->sif.width * priv->image_bpp : stride[0] );

	if( priv->dif.flags & XMM_GRAPH_FLAG_YFLIP )
	{
	    dest = priv->image[0] + (( priv->dif.height - y - 1 ) * priv->dif.bwidth + x ) * priv->image_bpp;
	    dstride = -dstride;
	}

	for( i = 0; i < height; i++, src += sstride, dest += dstride )
		memcpy( dest, src, width );
  }

  return XMM_RET_OK;
}

/*
 * Process filter chain
 */
int xmm_FilterVChain( void *_xmm, uint8_t *data[] )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Copy image to destination */
  if( priv->image[0] )	memcpy( data[0], priv->image[0], priv->image_size );

  return XMM_RET_OK;
}

/*
 * Stop chain ( will stop all stuff started in xmm_FilterVChainOutput()
 */
int xmm_FilterVChainStop( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;
  XMM_Plugin			*plugin;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pVFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  if( priv->pVF_cfmt )
  {
	/* Get plugin ID */
	plugin = xmm_memdup( &priv->pVF_cfmt->sys, sizeof( XMM_Plugin ));

	/* Free plugin */
	priv->pVF_cfmt->Close( priv->pVF_cfmt );
	priv->pVF_cfmt = NULL;

	/* unload plugin */
	if( plugin )
	{
	    xmm_PluginRemove( plugin );
	    free( plugin );
	}
  }

  /* Free image buffer */
  if( priv->image[0] )	free( priv->image[0] );
  priv->image[0] = NULL;

  /* Free filters */
  for( le = priv->pVFlist; le; le = le->next )
  {
	((XMM_PluginFilterVideo *)le->data)->Close((XMM_PluginFilterVideo *)le->data );
  }
  xmmList_Free( priv->pVFlist );
  priv->pVFlist = NULL;

  /* Unlock audio chain */
  xmm_FilterVChainUnlock( xmm );

  return XMM_RET_OK;
}

/*
 * Add plugin to list
 */
int xmm_FilterVChainAdd( void *_xmm, XMM_PluginFilterVideo *pfv )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  if( pfv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filter plugin == NULL\n" );
	return XMM_RET_ERROR;
  }

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  /* Add plugin */
  priv->cVFlist = xmmList_Append( priv->cVFlist, (void *)pfv );

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Add list to filter list
 */
int xmm_FilterVChainAddList( void *_xmm, XMM_List *_pVideoFilterList )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  if( _pVideoFilterList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filter plugin list == NULL\n" );
	return XMM_RET_ERROR;
  }

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  for( le = _pVideoFilterList; le; le = le->next )
	xmm_FilterVChainAdd( xmm, (XMM_PluginFilterVideo *)le->data );

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Remove filters from chain
 */
int xmm_FilterVChainFree( void *_xmm )
{
  struct priv_video_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pAFpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Video Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  xmmList_Free( priv->cVFlist );
  priv->cVFlist = NULL;

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/* * * * * * * * *
 * Internal code *
 * * * * * * * * */

/* * * * * * * * * * * * * * * * * * * * *
 * Default colorspace converter / scaler *
 * ( internal plugin )                   *
 * * * * * * * * * * * * * * * * * * * * */

/*
 * Prototypes
 */

#include "libxmm/xmmctl.h"
#include "libxmm/version.h"

/*
 * Prototypes
 */

static XMM_PluginFilterVideo *cfmt_Open( void *_xmm, XMM_ImageFormat *sif, XMM_ImageFormat *dif );
static int cfmt_Close( XMM_PluginFilterVideo *filter );
static int cfmt_Control( XMM_PluginFilterVideo *filter, uint32_t cmd, uint32_t param, void *data );

/*
 * Plugin info
 */

static XMM_PluginFilterVideo	plugin_info_cfmt = {{
				NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_VFILTER,
				0,
				XMM_VERSION_NUM,
				"",
				"STDVCONV",
				"libxmm graph converter / scaler ( default )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				cfmt_Open, cfmt_Close, cfmt_Control, NULL };

/*
 * Private data type
 */

struct priv_t
{
    int				swidth;		/* Source */
    int				sheight;
    int				dwidth;		/* Dest */
    int				dheight;
    int				wwidth;		/* Output ( screen size ) */
    int				wheight;
    int				gflags;		/* Flags ( e.g. Y-Flip ) */
    int (*Proc)( XMM_PluginFilterVideo *pfv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
    int (*ProcS)( XMM_PluginFilterVideo *pfv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
};

/*
 * Prototypes ( converter )
 */

#define CONV_FUNC_PROTO( NAME ) \
static int c_##NAME( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );

#define CONV_FUNC_PROTO_STRETCH( NAME ) \
static int cS_##NAME( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );

/*
 * RGB 16 Output
 */

CONV_FUNC_PROTO( rgb15_rgb16 )
CONV_FUNC_PROTO_STRETCH( rgb15_rgb16 )
CONV_FUNC_PROTO_STRETCH( rgb16_rgb16 )
CONV_FUNC_PROTO( rgb24_rgb16 )
CONV_FUNC_PROTO_STRETCH( rgb24_rgb16 )
CONV_FUNC_PROTO( rgb32_rgb16 )
CONV_FUNC_PROTO_STRETCH( rgb32_rgb16 )
CONV_FUNC_PROTO( yv12_rgb16 )
CONV_FUNC_PROTO_STRETCH( yv12_rgb16 )

/*
 * RGB 24 Output
 */

CONV_FUNC_PROTO( rgb15_rgb24 )
CONV_FUNC_PROTO_STRETCH( rgb15_rgb24 )
CONV_FUNC_PROTO( rgb16_rgb24 )
CONV_FUNC_PROTO_STRETCH( rgb16_rgb24 )
CONV_FUNC_PROTO_STRETCH( rgb24_rgb24 )
CONV_FUNC_PROTO( rgb32_rgb24 )
CONV_FUNC_PROTO_STRETCH( rgb32_rgb24 )
CONV_FUNC_PROTO( yv12_rgb24 )
CONV_FUNC_PROTO_STRETCH( yv12_rgb24 )

/*
 * RGB 32 Output
 */

CONV_FUNC_PROTO( rgb15_rgb32 )
CONV_FUNC_PROTO_STRETCH( rgb15_rgb32 )
CONV_FUNC_PROTO( rgb16_rgb32 )
CONV_FUNC_PROTO_STRETCH( rgb16_rgb32 )
CONV_FUNC_PROTO( rgb24_rgb32 )
CONV_FUNC_PROTO_STRETCH( rgb24_rgb32 )
CONV_FUNC_PROTO_STRETCH( rgb32_rgb32 )
CONV_FUNC_PROTO( yv12_rgb32 )
CONV_FUNC_PROTO_STRETCH( yv12_rgb32 )

/*
 * Prototypes
 */

static void InitTables( int bpp );

/*
 * Init filter
 */
static XMM_PluginFilterVideo *cfmt_Open( void *_xmm, XMM_ImageFormat *sif, XMM_ImageFormat *dif )
{
  XMM_PluginFilterVideo	*pVF;
  struct priv_t		*priv;
  XMM			*xmm = (XMM *) _xmm;
  char			buffer[5];
  int			i;
  struct
  {
    uint32_t		Src;
    uint32_t		Dst;
    int (*Proc)( XMM_PluginFilterVideo *pfv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
    int (*ProcS)( XMM_PluginFilterVideo *pfv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
  } aCv[] =
  {
  { XMM_GRAPH_FMT_RGB(15), XMM_GRAPH_FMT_RGB(16), c_rgb15_rgb16, cS_rgb15_rgb16 },
  { XMM_GRAPH_FMT_RGB(15), XMM_GRAPH_FMT_RGB(24), c_rgb15_rgb24, cS_rgb15_rgb24 },
  { XMM_GRAPH_FMT_RGB(15), XMM_GRAPH_FMT_RGB(32), c_rgb15_rgb32, cS_rgb15_rgb32 },
  { XMM_GRAPH_FMT_RGB(16), XMM_GRAPH_FMT_RGB(16), NULL,		 cS_rgb16_rgb16 },
  { XMM_GRAPH_FMT_RGB(16), XMM_GRAPH_FMT_RGB(24), c_rgb16_rgb24, cS_rgb16_rgb24 },
  { XMM_GRAPH_FMT_RGB(16), XMM_GRAPH_FMT_RGB(32), c_rgb16_rgb32, cS_rgb16_rgb32 },
  { XMM_GRAPH_FMT_RGB(24), XMM_GRAPH_FMT_RGB(16), c_rgb24_rgb16, cS_rgb24_rgb16 },
  { XMM_GRAPH_FMT_RGB(24), XMM_GRAPH_FMT_RGB(24), NULL,		 cS_rgb24_rgb24 },
  { XMM_GRAPH_FMT_RGB(24), XMM_GRAPH_FMT_RGB(32), c_rgb24_rgb32, cS_rgb24_rgb32 },
  { XMM_GRAPH_FMT_RGB(32), XMM_GRAPH_FMT_RGB(16), c_rgb32_rgb16, cS_rgb32_rgb16 },
  { XMM_GRAPH_FMT_RGB(32), XMM_GRAPH_FMT_RGB(24), c_rgb32_rgb24, cS_rgb32_rgb24 },
  { XMM_GRAPH_FMT_RGB(32), XMM_GRAPH_FMT_RGB(32), NULL,		 cS_rgb32_rgb32 },
  { XMM_GRAPH_FMT_YV12, XMM_GRAPH_FMT_RGB(16), c_yv12_rgb16,	 cS_yv12_rgb16 },
  { XMM_GRAPH_FMT_YV12, XMM_GRAPH_FMT_RGB(24), c_yv12_rgb24,	 cS_yv12_rgb24 },
  { XMM_GRAPH_FMT_YV12, XMM_GRAPH_FMT_RGB(32), c_yv12_rgb32,	 cS_yv12_rgb32 },
  };

  /* Allocate plugin data */
  if(( pVF = xmm_memdup_x( &plugin_info_cfmt, sizeof( XMM_PluginFilterVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pVF[1];
  pVF->sys.priv = (void *) priv;
  pVF->sys.xmm = xmm;

  /* Save data */
  priv->swidth = sif->width;
  priv->sheight = sif->height;
  priv->gflags = sif->flags;
  priv->dwidth = dif->width;
  priv->dheight = dif->height;
  priv->wwidth = dif->bwidth;
  priv->wheight = dif->bheight;

  xmm_logging( 2, "FVC-STD! Source: %x [%s]\n", sif->codec, xmm_FOURCC_string( sif->codec ));
  xmm_logging( 2, "FVC-STD! Dest: %x [%s]\n", dif->codec, xmm_FOURCC_string( dif->codec ));

  /* Find converter */
  for( i = 0; aCv[i].Src; i++ )
  {
	if(( sif->codec == aCv[i].Src ) && ( dif->codec == aCv[i].Dst ))
	{
	    priv->Proc = pVF->Proc = aCv[i].Proc;
	    priv->ProcS = aCv[i].ProcS;
	    return pVF;
	}
  }

  free( pVF );

  strcpy( buffer, xmm_FOURCC_string( sif->codec ));
  xmm_SetError( pVF->sys.xmm, XMM_ERR_NOTSUPPORTED, __FUNCTION__ "() Needed conversion not supported. %x [%s] --> %x [%s]", sif->codec, buffer, dif->codec, xmm_FOURCC_string( dif->codec ));

  return (XMM_PluginFilterVideo *) NULL;
}

/*
 * Free converter
 */
static int cfmt_Close( XMM_PluginFilterVideo *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Control function
 */
static int cfmt_Control( XMM_PluginFilterVideo *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;
  XMM_ControlScale	*scale;

  switch( cmd )
  {
    case XMM_CTLSET_SCALE:
	    scale = (XMM_ControlScale *) data;
	    if(( priv->swidth - scale->width ) || ( priv->sheight - scale->height ))	filter->Proc = priv->ProcS;
	    else	filter->Proc = priv->Proc;
	    priv->dwidth = scale->width;
	    priv->dheight = scale->height;
	    priv->wwidth = scale->bwidth;
	    priv->wheight = scale->bheight;
	    return XMM_CTLRET_TRUE;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_VFILTER )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No VFILTER command */
}

/*
 * Global data
 */

static long *pR_tab, *pG_tab, *pB_tab;

/*
 * Internal code 
 */


/*
 * RGB --> RGB Conversion
 */

/* * * Description * * *

  Usable variables:
    dest	DSTTYPE pointer. Has to be incremented for every written pixel
    src		SRCTYPE pointer. Has to be incremented for every read pixel

  Input:
    NAME	Fuction Name
    SRCTYPE	Type of source data ( e.g. uint8_t or uint16_t )
    SRCMUL	'*' number of bytes per pixel
    SRCDIV	'/' size of SRCTYPE in bytes
    DSTTYPE	Type of source data ( e.g. uint8_t or uint16_t )
    DSTMUL	'*' number of bytes per pixel
    VARS	Specia vars needed for conversion
    CONVERSION	Conversion
*/


#define CONV_FUNC( NAME, SRCTYPE, SRCMUL, SRCDIV, DSTTYPE, DSTMUL, VARS, CONVERSION ) \
static int c_##NAME( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy ) \
{ \
  struct priv_t *priv = conv->sys.priv; \
  DSTTYPE *dest; \
  SRCTYPE *src; \
  int i, j, sstride_add, dstride_add; \
  VARS \
 \
  src = (SRCTYPE *) sdata[0]; \
  dest = (DSTTYPE *)(ddata[0]) + ( dy * priv->wwidth + dx ) DSTMUL; \
 \
  sstride_add = (( stride == NULL ) ? 0 : ( stride[0] - width SRCMUL ) SRCDIV ); \
  dstride_add = ( priv->wwidth - width ) DSTMUL; \
 \
  if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )\
  { \
	dest = (DSTTYPE *)(ddata[0]) + (( priv->dheight - dy - 1 ) * priv->wwidth + dx ) DSTMUL; \
	dstride_add = - ( priv->wwidth + width ) DSTMUL; \
  } \
 \
  for( i = 0; i < height; i++, src += sstride_add, dest += dstride_add ) \
  { \
	for( j = 0; j < width; j++ ) \
	{ \
		CONVERSION \
	} \
  } \
 \
  return XMM_RET_OK; \
}

/*
 * RGB --> RGB Conversion ( with stretching )
 */

/* * * Description * * *

  Usable variables:
    dest	DSTTYPE pointer. Has to be incremented for every written pixel
    src		SRCTYPE pointer. MUST not be incremented.
    i		line counter ( destination )
    j		column counter ( destination )
    sstride	width of source image ( maybe internal buffer )
		number of SRCTYPE items

  Input:
    NAME	Fuction Name
    SRCTYPE	Type of source data ( e.g. uint8_t or uint16_t )
    SRCMUL	'*' number of bytes per pixel
    SRCDIV	'/' size of SRCTYPE in bytes
    DSTTYPE	Type of source data ( e.g. uint8_t or uint16_t )
    DSTMUL	'*' number of bytes per pixel
    VARS	Specia vars needed for conversion
    CONVERSION	Conversion
*/

#define CONV_FUNC_STRETCH( NAME, SRCTYPE, SRCMUL, SRCDIV, DSTTYPE, DSTMUL, VARS, CONVERSION ) \
static int cS_##NAME( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy ) \
{ \
  struct priv_t *priv = conv->sys.priv; \
  int i, j, dstride_add, sstride; \
  SRCTYPE *src; \
  DSTTYPE *dest; \
  VARS \
 \
  src = (SRCTYPE *) sdata[0]; \
  dest = (DSTTYPE *)(ddata[0]) + (( dy * priv->dheight / priv->sheight ) * priv->wwidth + ( dx * priv->dwidth / priv->swidth )) DSTMUL; \
 \
  sstride = (( stride == NULL ) ? priv->swidth SRCMUL : stride[0] ) SRCDIV; \
  dstride_add = ( priv->wwidth - ( width * priv->dwidth / priv->swidth )) DSTMUL; \
 \
  height = height * priv->dheight / priv->sheight; \
  width = width * priv->dwidth / priv->swidth; \
 \
  if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )\
  { \
	dest = (DSTTYPE *)(ddata[0]) + (( priv->dheight - ( dy * priv->dheight / priv->sheight ) - 1 ) * priv->wwidth + ( dx * priv->dwidth / priv->swidth )) DSTMUL; \
	dstride_add = - ( priv->wwidth + width ) DSTMUL; \
  } \
 \
  for( i = 0; i < height; i++, dest += dstride_add ) \
  { \
	for( j = 0; j < width; j++ ) \
	{ \
		CONVERSION \
	} \
  } \
 \
  return XMM_RET_OK; \
}

/*
 * YUV --> RGB Conversion
 */

#define CONV_FUNC_YUV( NAME, DSTTYPE, DSTMUL, VARS, CONVERSION ) \
static int c_##NAME( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy ) \
{ \
  struct priv_t *priv = conv->sys.priv; \
  int j, i, dstride_add, sstride_add[3]; \
  int CR, CB, Yo, Cr_R, Cr_G, Cb_G, Cb_B; \
  DSTTYPE *row1, *row2; \
  uint8_t *py1, *py2, *pu, *pv; \
 \
  row1 = (DSTTYPE *)(ddata[0]) + ( dy * priv->wwidth + dx ) DSTMUL; \
  row2 = row1 + priv->wwidth DSTMUL; \
  dstride_add = (( priv->wwidth << 1 ) - width ) DSTMUL; \
 \
  py1 = sdata[0]; \
  py2 = py1 + (( stride == NULL ) ? priv->swidth : stride[0] ); \
  pu = sdata[2]; \
  pv = sdata[1]; \
 \
  sstride_add[0] = ((( stride == NULL ) ? priv->swidth : stride[0] ) << 1 ) - width; \
  sstride_add[1] = ((( stride == NULL ) ? priv->swidth : stride[1] ) - width ) >> 1; \
  sstride_add[2] = ((( stride == NULL ) ? priv->swidth : stride[2] ) - width ) >> 1; \
 \
  if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )\
  { \
	row2 = (DSTTYPE *)(ddata[0]) + (( priv->dheight - dy - 1 ) * priv->wwidth + dx ) DSTMUL; \
	row1 = row2 - priv->wwidth DSTMUL; \
	dstride_add = - (( priv->wwidth << 1 ) + width ) DSTMUL; \
  } \
 \
  for( i = 0; i < height; i += 2, py1 += sstride_add[0], py2 += sstride_add[0], pu += sstride_add[1], pv += sstride_add[2], row1 += dstride_add, row2 += dstride_add ) \
  { \
	for( j = 0; j < width; j += 2 ) \
	{ \
		CR = *pu++; \
		CB = *pv++; \
		Cr_R = Cr_R_tab[ CR ]; \
		Cr_G = Cr_G_tab[ CR ]; \
		Cb_G = Cb_G_tab[ CB ]; \
		Cb_B = Cb_B_tab[ CB ]; \
 \
		CONVERSION \
	} \
  } \
 \
  return XMM_RET_OK; \
}

/*
 * YUV --> RGB Conversion ( with stretching )
 */

#define CONV_FUNC_YUV_STRETCH( NAME, DSTTYPE, DSTMUL, VARS, CONVERSION ) \
static int cS_##NAME( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy ) \
{ \
  struct priv_t *priv = conv->sys.priv; \
  int i, j, dstride_add, swidth[3], idx; \
  int CR, CB, Yo, Cr_R, Cr_G, Cb_G, Cb_B; \
  DSTTYPE *dest; \
  uint8_t *py, *pu, *pv; \
 \
  height = height * priv->dheight / priv->sheight; \
  width = width * priv->dwidth / priv->swidth; \
 \
  swidth[0] = (( stride == NULL ) ? priv->swidth : stride[0] ); \
  swidth[1] = (( stride == NULL ) ? ( priv->swidth >> 1 ) : ( stride[1] >> 1 )); \
  swidth[2] = (( stride == NULL ) ? ( priv->swidth >> 1 ) : ( stride[2] >> 1 )); \
 \
  dest = (DSTTYPE *)(ddata[0]) + (( dy * priv->dheight / priv->sheight ) * priv->wwidth + ( dx * priv->dwidth / priv->swidth )) DSTMUL; \
  dstride_add = ( priv->wwidth - width ) DSTMUL; \
 \
  if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )\
  { \
	dest = (DSTTYPE *)(ddata[0]) + (( priv->dheight - ( dy * priv->dheight / priv->sheight ) - 1 ) * priv->wwidth + ( dx * priv->dwidth / priv->swidth )) DSTMUL; \
	dstride_add = - ( priv->wwidth + width ) DSTMUL; \
  } \
 \
  for( i = 0; i < height; i++, dest += dstride_add ) \
  { \
	py = sdata[0] + (i * priv->sheight / priv->wheight) * swidth[0]; \
	pu = sdata[2] + (((i * priv->sheight / priv->wheight) * swidth[2] ) >> 1 ); \
	pv = sdata[1] + (((i * priv->sheight / priv->wheight) * swidth[1] ) >> 1 ); \
 \
	for( j = 0; j < width; j++ ) \
	{ \
		idx = j * priv->swidth / priv->wwidth; \
 \
		CR = pu[idx >> 1]; \
		CB = pv[idx >> 1]; \
		Cr_R = Cr_R_tab[ CR ]; \
		Cr_G = Cr_G_tab[ CR ]; \
		Cb_G = Cb_G_tab[ CB ]; \
		Cb_B = Cb_B_tab[ CB ]; \
		Yo = Y_tab[ py[idx]]; \
 \
		CONVERSION \
	} \
  } \
 \
  return XMM_RET_OK; \
}

/*
 * Definitions
 */

#define	oRED	Yo + Cr_R
#define	oGREEN	Yo + Cr_G + Cb_G
#define	oBLUE	Yo + Cb_B

/*
 * Data
 */

static int Y_tab[256], Cr_R_tab[256], Cr_G_tab[256], Cb_G_tab[256], Cb_B_tab[256];
static long R_tab[768], G_tab[768], B_tab[768];

/*
 * Initialize tables
 */

static void InitTables( int bpp )
{
  int i, CB, CR;
  
  for( i=0; i<256; i++ )
  {
	Y_tab[i] = i;
      
	CR = i - 128;
	CB = i - 128;
      
	Cr_R_tab[i] = 1.402 * CR;
	Cr_G_tab[i] = -0.71414 * CR;
	Cb_G_tab[i] = -0.34414 * CB;
	Cb_B_tab[i] = 1.772 * CB;
  }
  
  for( i=0; i<256; i++ )
  {
	if( bpp == 15 )	R_tab[i+256] = (i & 248 ) << 7;
	if( bpp == 16 )	R_tab[i+256] = (i & 248 ) << 8;
	if( bpp == 24 )	R_tab[i+256] = (i & 255 );
	if( bpp == 32 )	R_tab[i+256] = (i & 255 );
	
	if( bpp == 15 )	G_tab[i+256] = (i & 248 ) << 2;
	if( bpp == 16 )	G_tab[i+256] = (i & 252 ) << 3;
	if( bpp == 24 )	G_tab[i+256] = (i & 255 );
	if( bpp == 32 )	G_tab[i+256] = (i & 255 );

	if( bpp == 15 )	B_tab[i+256] = i >> 3;
	if( bpp == 16 )	B_tab[i+256] = i >> 3;
	if( bpp == 24 )	B_tab[i+256] = i;
	if( bpp == 32 )	B_tab[i+256] = i;
  }

  for( i=0; i<256; i++ )
  {
	R_tab[i] |= R_tab[256];
	R_tab[i+512] |= R_tab[511];
	G_tab[i] |= G_tab[256];
	G_tab[i+512] |= G_tab[511];
	B_tab[i] |= B_tab[256];
	B_tab[i+512] |= B_tab[511];
  }
  
  pR_tab = R_tab + 256;
  pG_tab = G_tab + 256;
  pB_tab = B_tab + 256;
}

/*
 * Internal ( Converter code )
 */

/*
 * RGB 16 Output
 */

CONV_FUNC( rgb15_rgb16, uint16_t, * 2, / 2, uint16_t,,
	register uint16_t tmp;,
	tmp = *src++;
	*dest++ = ( tmp & 0x001F ) | (( tmp & 0x7FE0 ) << 1 ); )

CONV_FUNC_STRETCH( rgb15_rgb16, uint16_t, * 2, / 2, uint16_t,,
	register uint16_t tmp;,
	tmp = *( src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth));
	*dest++ = ( tmp & 0x001F ) | (( tmp & 0x7FE0 ) << 1 ); )

CONV_FUNC_STRETCH( rgb16_rgb16, uint16_t, * 2, / 2, uint16_t,,
	register uint16_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth);
	*dest++ = p[0]; )

CONV_FUNC( rgb24_rgb16, uint8_t, * 3,, uint16_t,,,
	*dest++ = pB_tab[src[0]] | pG_tab[src[1]] | pR_tab[src[2]];
	src += 3; )

CONV_FUNC_STRETCH( rgb24_rgb16, uint8_t, * 3,, uint16_t,,
	uint8_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + 3 * (j * priv->swidth / priv->dwidth);
	*dest++ = pB_tab[p[0]] | pG_tab[p[1]] | pR_tab[p[2]]; )

CONV_FUNC( rgb32_rgb16, uint8_t, * 4,, uint16_t,,,
	*dest++ = pB_tab[src[0]] | pG_tab[src[1]] | pR_tab[src[2]];
	src += 4; )

CONV_FUNC_STRETCH( rgb32_rgb16, uint8_t, * 4,, uint16_t,,
	uint8_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + 4 * (j * priv->swidth / priv->dwidth);
	*dest++ = pB_tab[p[0]] | pG_tab[p[1]] | pR_tab[p[2]]; )

CONV_FUNC_YUV( yv12_rgb16, uint16_t,,,
	Yo = Y_tab[ *py1++ ];
	*row1++ = pR_tab[oRED] | pG_tab[oGREEN] | pB_tab[oBLUE];
	Yo = Y_tab[ *py1++ ];
	*row1++ = pR_tab[oRED] | pG_tab[oGREEN] | pB_tab[oBLUE];
	Yo = Y_tab[ *py2++ ];
	*row2++ = pR_tab[oRED] | pG_tab[oGREEN] | pB_tab[oBLUE];
	Yo = Y_tab[ *py2++ ];
	*row2++ = pR_tab[oRED] | pG_tab[oGREEN] | pB_tab[oBLUE]; )

CONV_FUNC_YUV_STRETCH( yv12_rgb16, uint16_t,,,
	*dest++ = pR_tab[oRED] | pG_tab[oGREEN] | pB_tab[oBLUE]; )

/*
 * RGB 24 Output
 */

CONV_FUNC( rgb15_rgb24, uint16_t, * 2, / 2, uint8_t, * 3,
	register uint16_t tmp;,
	tmp = *src++;
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x03E0 ) >> 2;	/* G */
	*dest++ = ( tmp & 0x7C00 ) >> 7; 	/* B */
	)

CONV_FUNC_STRETCH( rgb15_rgb24, uint16_t, * 2, / 2, uint8_t, * 3,
	register uint16_t tmp;,
	tmp = *(src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth));
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x03E0 ) >> 2;	/* G */
	*dest++ = ( tmp & 0x7C00 ) >> 7;	/* R */
	)

CONV_FUNC( rgb16_rgb24, uint16_t, * 2, / 2, uint8_t, * 3,
	register uint16_t tmp;,
	tmp = *src++;
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x07E0 ) >> 3;	/* G */
	*dest++ = ( tmp & 0xF800 ) >> 8;	/* R */
	)

CONV_FUNC_STRETCH( rgb16_rgb24, uint16_t, * 2, / 2, uint8_t, * 3,
	register uint16_t tmp;,
	tmp = *(src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth));
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x07E0 ) >> 3;	/* G */
	*dest++ = ( tmp & 0xF800 ) >> 8;	/* R */
	)

CONV_FUNC_STRETCH( rgb24_rgb24, uint8_t, * 3,, uint8_t, * 3,
	uint8_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + 3 * (j * priv->swidth / priv->dwidth);
	*dest++ = p[0];
	*dest++ = p[1];
	*dest++ = p[2]; )

CONV_FUNC( rgb32_rgb24, uint8_t, * 4,, uint8_t, * 3,,
	*dest++ = *src++;
	*dest++ = *src++;
	*dest++ = *src++;
	src++; )

CONV_FUNC_STRETCH( rgb32_rgb24, uint8_t, * 4,, uint8_t, * 3,
	uint8_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + 4 * (j * priv->swidth / priv->dwidth);
	*dest++ = p[0];
	*dest++ = p[1];
	*dest++ = p[2]; )

CONV_FUNC_YUV( yv12_rgb24, uint8_t, * 3,,
	Yo = Y_tab[ *py1++ ];
	*row1++ = pB_tab[oBLUE]; *row1++ = pG_tab[oGREEN]; *row1++ = pR_tab[oRED];
	Yo = Y_tab[ *py1++ ];
	*row1++ = pB_tab[oBLUE]; *row1++ = pG_tab[oGREEN]; *row1++ = pR_tab[oRED];
	Yo = Y_tab[ *py2++ ];
	*row2++ = pB_tab[oBLUE]; *row2++ = pG_tab[oGREEN]; *row2++ = pR_tab[oRED];
	Yo = Y_tab[ *py2++ ];
	*row2++ = pB_tab[oBLUE]; *row2++ = pG_tab[oGREEN]; *row2++ = pR_tab[oRED];
	)

CONV_FUNC_YUV_STRETCH( yv12_rgb24, uint8_t, * 3,,
	*dest++ = pB_tab[oBLUE];
	*dest++ = pG_tab[oGREEN];
	*dest++ = pR_tab[oRED];
	)

/*
 * RGB 32 Output
 */

CONV_FUNC( rgb15_rgb32, uint16_t, * 2, / 2, uint8_t, * 4,
	register uint16_t tmp;,
	tmp = *src++;
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x03E0 ) >> 2;	/* G */
	*dest++ = ( tmp & 0x7C00 ) >> 7;	/* R */
	dest++; )

CONV_FUNC_STRETCH( rgb15_rgb32, uint16_t, * 2, / 2, uint8_t, * 4,
	register uint16_t tmp;,
	tmp = *(src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth));
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x03E0 ) >> 2;	/* G */
	*dest++ = ( tmp & 0x7C00 ) >> 7;	/* R */
	dest++; )

CONV_FUNC( rgb16_rgb32, uint16_t, * 2, / 2, uint8_t, * 4,
	register uint16_t tmp;,
	tmp = *src++;
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x07E0 ) >> 3;	/* G */
	*dest++ = ( tmp & 0xF800 ) >> 8;	/* R */
	dest++; )

CONV_FUNC_STRETCH( rgb16_rgb32, uint16_t, * 2, / 2, uint8_t, * 4,
	register uint16_t tmp;,
	tmp = *(src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth));
	*dest++ = ( tmp & 0x001F ) << 3;	/* B */
	*dest++ = ( tmp & 0x07E0 ) >> 3;	/* G */
	*dest++ = ( tmp & 0xF800 ) >> 8;	/* R */
	dest++; )

CONV_FUNC( rgb24_rgb32, uint8_t, * 3,, uint8_t, * 4 ,,
	*dest++ = *src++;
	*dest++ = *src++;
	*dest++ = *src++;
	dest++; )

CONV_FUNC_STRETCH( rgb24_rgb32, uint8_t, * 3,, uint8_t, * 4,
	uint8_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + 3 * (j * priv->swidth / priv->dwidth);
	*dest++ = p[0];
	*dest++ = p[1];
	*dest++ = p[2];
	dest++; )

CONV_FUNC_STRETCH( rgb32_rgb32, uint8_t, * 4,, uint8_t, * 4,
	uint8_t *p;,
	p = src + (i * priv->sheight / priv->dheight) * sstride + 4 * (j * priv->swidth / priv->dwidth);
	*dest++ = p[0];
	*dest++ = p[1];
	*dest++ = p[2];
	*dest++ = p[3]; )

CONV_FUNC_YUV( yv12_rgb32, uint8_t, * 4,,
	Yo = Y_tab[ *py1++ ];
	*row1++ = pB_tab[oBLUE]; *row1++ = pG_tab[oGREEN]; *row1++ = pR_tab[oRED]; row1++;
	Yo = Y_tab[ *py1++ ];
	*row1++ = pB_tab[oBLUE]; *row1++ = pG_tab[oGREEN]; *row1++ = pR_tab[oRED]; row1++;
	Yo = Y_tab[ *py2++ ];
	*row2++ = pB_tab[oBLUE]; *row2++ = pG_tab[oGREEN]; *row2++ = pR_tab[oRED]; row2++;
	Yo = Y_tab[ *py2++ ];
	*row2++ = pB_tab[oBLUE]; *row2++ = pG_tab[oGREEN]; *row2++ = pR_tab[oRED]; row2++;
	)

CONV_FUNC_YUV_STRETCH( yv12_rgb32, uint8_t, * 4,,
	*dest++ = pB_tab[oBLUE];
	*dest++ = pG_tab[oGREEN];
	*dest++ = pR_tab[oRED];
	dest++; )
