/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpinput.c
 * Input (DeMUX) manager/wrapper
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "libxmm/xmmp.h"
#include "libxmm/xmmctl.h"
#include "libxmm/lpinput.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/list.h"
#include "../xmmpriv.h"

/*
 * Open file.
 */
XMM_PluginInput *xmm_InputOpen( void *_xmm, char *filename, int flags, int *mode )
{
  XMM_List		*le;
  XMM_PluginInput	*pInput = NULL, *tInput = NULL;
  XMM			*xmm = (XMM *) _xmm;
  uint32_t		caps;

  /* Check arguments */
  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( filename == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() url = NULL\n" );
	return NULL;
  }

  if( xmm->pInputList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_NULL, __FUNCTION__ "() pInputList: libXMM not initialized or no Input plugins found ?\n" );
	return NULL;
  }

  if( xmm->iFilename )
  {
	xmm_SetError( xmm, XMM_ERR_NULL, __FUNCTION__ "() XMM object contains opened file. This file must be closed first." );
	return NULL;
  }

  /* Find plugin */
  for( le = xmm->pInputList; le; le = le->next )
  {
	if(((XMM_PluginInput *)le->data)->Check( xmm, filename ) == 1 )
	{
		tInput = (XMM_PluginInput *)le->data;
		break;
	}
  }

  if( tInput == NULL || ( le == NULL ))
  {
	xmm_SetError( xmm, XMM_ERR_NOTSUPPORTED, __FUNCTION__ "() No matching input plugin found for %s", filename );
	return NULL;
  }

  /* Initialize  */
  if(( pInput = tInput->Init( xmm )) == NULL )
  {
	return NULL;
  }

  /* Determine mode: read or direct out */
  caps = 0;
  if( pInput->Control( pInput, XMM_CTLGET_CAPS, 0, &caps ) != XMM_CTLRET_ARG )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to get input plugin capabilities" );
	return NULL;
  }

  /* store supported of the desired modes in xmmplay->modes */
  *mode = caps & ( flags & ( XMM_INPUT_CF_MODE_DOUT | XMM_INPUT_CF_MODE_READ ));

  /* Check if any desired mode supported */
  if(( *mode & ( XMM_INPUT_CF_MODE_DOUT | XMM_INPUT_CF_MODE_READ )) == 0 )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Desired mode not supported by input plugin." );
	return NULL;
  }

  /* Prefer DOUT mode if both supported */
  if(( *mode & XMM_INPUT_CF_MODE_DOUT ) && ( *mode & XMM_INPUT_CF_MODE_READ ))
  {
	xmm_logging( 2, "xmmplay_Open(): DOUT and READ mode desired and supported. Prefering DOUT.\n" );
	*mode &= ~XMM_INPUT_CF_MODE_READ;
  }

  /* Set mode bits in flags according to xmmplay->mode */
  flags &= ~( XMM_INPUT_CF_MODE_DOUT | XMM_INPUT_CF_MODE_READ ) | *mode;

  /* Open file */
  if( pInput->Open( pInput, filename, flags ) < 0 )
  {
	pInput->Close( pInput );
	return NULL;
  }

  /* Dublicate filename */
  xmm->iFilename = strdup( filename );

  return pInput;
}

/*
 * Close file.
 */
int xmm_InputClose( XMM_PluginInput *input )
{
  XMM		*xmm = (XMM *) input->sys.xmm;

  if( input == NULL )	return XMM_RET_INVALID_ARG;

  /* Free filename */
  if( xmm->iFilename ) free( xmm->iFilename );
  xmm->iFilename = NULL;
  if( xmm->oFilename ) free( xmm->oFilename );
  xmm->oFilename = NULL;

  return input->Close( input );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Controll call.
 */
int xmm_InputControl( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->Control( input, cmd, param, data );
}

/*
 * Seek.
 */
int xmm_InputSeek( XMM_PluginInput *input, double seek )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->Seek( input, seek );
}

/*
 * File information.
 */
double xmm_InputInfo( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->Info( input, ci, avdiff, seekval );
}

/*
 * Audio streams.
 */
int xmm_InputAudioStreams( XMM_PluginInput *input )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->AudioStreams( input );
}

/*
 * Audio information.
 */
int xmm_InputAudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->AudioInfo( input, stream, ai, cSample );
}

/*
 * Audio data.
 */
int xmm_InputAudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->AudioRead( input, stream, buffer, samples );
}

/*
 * Seek audio stream.
 */
int xmm_InputAudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->AudioSeek( input, stream, sample );
}

/*
 * Video streams.
 */
int xmm_InputVideoStreams( XMM_PluginInput *input )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->VideoStreams( input );
}

/*
 * Video information.
 */
int xmm_InputVideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->VideoInfo( input, stream, vi, cFrame );
}

/*
 * Video data.
 */
int xmm_InputVideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->VideoRead( input, stream, buffer );
}

/*
 * Video PTS.
 */
int xmm_InputVideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->VideoPTS( input, stream, videoPTS );
}

/*
 * Seek video stream.
 */
int xmm_InputVideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->VideoSeek( input, stream, frame );
}

/*
 * Check.
 */
int xmm_InputCheck( XMM_PluginInput *input, void *xmm, char *filename )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->Check( xmm, filename );
}

/*
 * File information.
 */
int xmm_InputFileInfo( XMM_PluginInput *input, void *xmm, char *filename, XMM_ClipInfo *ci )
{
  if( input == NULL )	return XMM_RET_INVALID_ARG;

  return input->FileInfo( xmm, filename, ci );
}
