/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpio.c
 * I/O manager/wrapper
 */

#include <stdio.h>
#include "libxmm/xmmp.h"
#include "libxmm/lpio.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/list.h"
#include "../xmmpriv.h"

/*
 * Open matching I/O
 */
XMM_PluginIO *xmm_IOOpen( void *_xmm, const char *url, int mode )
{
  XMM_List	*le;
  XMM_PluginIO	*pIO = NULL;
  XMM		*xmm = (XMM *) _xmm;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( url == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() url = NULL\n" );
	return NULL;
  }

  if( xmm->pIOList == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_NULL, __FUNCTION__ "() pIOList: libXMM not initialized or no I/O plugins found ?\n" );
	return NULL;
  }


  for( le = xmm->pIOList; le; le = le->next )
	if(( pIO = ((XMM_PluginIO *)le->data)->Open( xmm, url, mode )))	break;

  if( pIO == NULL || ( le == NULL ))
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() No matching I/O plugin found for '%s' ( mode %i )\n", url, mode );
	return NULL;
  }

  return pIO;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Close I/O.
 */
int xmm_IOClose( XMM_PluginIO *io )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Close( io );
}

/*
 * Read data.
 */
int xmm_IORead( XMM_PluginIO *io, void *ptr, int size, int nelem )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Read( io, ptr, size, nelem );
}

/*
 * Write data.
 */
int xmm_IOWrite( XMM_PluginIO *io, void *ptr, int size, int nelem )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Write( io, ptr, size, nelem );
}

/*
 * Get I/O stream position.
 */
long xmm_IOTell( XMM_PluginIO *io )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Tell( io );
}

/*
 * Seek in I/O stream.
 */
int xmm_IOSeek( XMM_PluginIO *io, long offset, int mode )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Seek( io, offset, mode );
}

/*
 * Check for end of I/O stream.
 */
int xmm_IOEof( XMM_PluginIO *io )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Eof( io );
}

/*
 * Get I/O stream size.
 */
long xmm_IOSize( XMM_PluginIO *io )
{
  if( io == NULL )	return XMM_RET_INVALID_ARG;

  return io->Size( io );
}
