/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpsound.c
 * Sound manager/wrapper
 */

#include <stdio.h>
#include <stdlib.h>
#include "libxmm/xmmp.h"
#include "libxmm/lpsound.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "../xmmpriv.h"

/*
 * Open sound
 */
XMM_PluginSound *xmm_SoundOpen( void *_xmm, char *filename )
{
  XMM_PluginSound	*pSound = NULL, *tSound;
  XMM			*xmm = (XMM *) _xmm;

  if( filename == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() filename == NULL" );
	return NULL;
  }

  /* Register plugin */
  if(( tSound = (XMM_PluginSound *)xmm_PluginRegister( filename )) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to load plugin '%s'", filename );
	return NULL;
  }

  /* Initialize plugin */
  pSound = tSound->Init( xmm );
  if( pSound == NULL )	return NULL;

  return pSound;
}

/*
 * Open sound from memory plugin
 */
XMM_PluginSound *xmm_SoundOpenMP( void *xmm, XMM_PluginSound *tSound )
{
  if( tSound == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() tSound == NULL" );
	return NULL;
  }

  /* Initialize plugin */
  return tSound->Init( xmm );
}

/*
 * Free plugin.
 */
int xmm_SoundClose( XMM_PluginSound *sound )
{
  XMM_Plugin	*plugin;

  if( sound == NULL )	return XMM_ERR_INVALID_ARG;

  /* Get plugin ID */
  plugin = xmm_memdup( &sound->sys, sizeof( XMM_Plugin ));

  /* Free plugin */
  sound->Close( sound );

  /* unload plugin */
  if( plugin )
  {
	xmm_PluginRemove( plugin );
	free( plugin );
  }

  return XMM_RET_OK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Control call.
 */
int xmm_SoundControl( XMM_PluginSound *sound, uint32_t cmd, uint32_t param, void *data )
{
  if( sound == NULL )	return XMM_ERR_INVALID_ARG;

  return sound->Control( sound, cmd, param, data );
}

/*
 * Start sound output.
 */
int xmm_SoundStart( XMM_PluginSound *sound, XMM_SoundFormat *format, int fsize, int fcount )
{
  if( sound == NULL )	return XMM_ERR_INVALID_ARG;

  return sound->Start( sound, format, fsize, fcount );
}

/*
 * Write data.
 */
int xmm_SoundWrite( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  if( sound == NULL )	return XMM_ERR_INVALID_ARG;

  return sound->Write( sound, data, samples );
}

/*
 * Flush data.
 */
int xmm_SoundFlush( XMM_PluginSound *sound )
{
  if( sound == NULL )	return XMM_ERR_INVALID_ARG;

  sound->Flush( sound );

  return XMM_RET_OK;
}
