/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/config.h"
#include "libxmm/error.h"
#include "libxmm/util/config.h"
#include "libxmm/util/utils.h"
#include "xmmpriv.h"

/*
 * Get codec properties 
 */
int xmm_AcCodecFile( void *xmm, uint32_t format, int type, XMM_AcCodecProperties *cp )
{
  char			id[5], *section, cmap[XMM_MAXPATHLEN];
  int			i, ret;
  XMM_AcCodecProperties	icp = { "", "", "", 0, "" };
  XMM_ConfigBlock	ccfg[] =
  {
    { icp.info,		"xxxx.info",		XMM_CFG_TYPE_STRING },
    { icp.file,		"xxxx.file",		XMM_CFG_TYPE_STRING },
    { icp.stat,		"xxxx.stat",		XMM_CFG_TYPE_STRING },
    { &icp.flags,	"xxxx.flag",		XMM_CFG_TYPE_INT },
    { icp.ofmt,		"xxxx.ofmt",		XMM_CFG_TYPE_STRING },
    { cmap,		"xxxx.cmap",		XMM_CFG_TYPE_STRING },
    { NULL, "", 0 }
  };

  /* Build section and line name */
  switch( type )
  {
	case XMM_CFG_CODEC_VFW:
		section = "vfw";
		strcpy( id, xmm_FOURCC_string( format ));
		break;

	case XMM_CFG_CODEC_ACM:
		section = "acm";
		sprintf( id, "%4.4x", format );
		break;

	default:
		return xmm_SetError( xmm, XMM_ERR_INVALID_ARG, __FUNCTION__ "() Unknown type" );
  }

  while( id[0] != '\0' )
  {
	cmap[0] = '\0';
	/* Initialize properties */
	icp.info[0] = '\0';
	icp.file[0] = '\0';
	icp.stat[0] = '\0';
	icp.flags = 0;
	icp.ofmt[0] = '\0';

	/* Set format ID */
	for( i = 0; ccfg[i].ptr; i++ )	strncpy( ccfg[i].name, id, 4 );

	/* Load codec info */
	ret = xmmCfg_BlockLoad( "codecs-w32", section, ccfg );
	if( ret < 0 )
	{
	    ret = xmmCfg_BlockLoad( LIBDIR "codecs-w32", section, ccfg );
	}
  
	if( ret < 0 )
	    return xmm_SetError( xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Unable to read codec config." );

	/* Do mapping */
	strcpy( id, cmap );
  }

  /* Copy properties */
  if( cp )	memcpy( cp, &icp, sizeof( XMM_AcCodecProperties ));

  if( icp.file[0] == '\0' )	return XMM_RET_NOTSUPPORTED;

  return XMM_RET_OK;
}

/*
 * Get/Set name of sound plugin
 */
const char *xmm_AcSoundPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )	strcpy( xmm->sound_plugin, name );

  return xmm->sound_plugin;
}

/*
 * Get/Set name of graph plugin
 */
const char *xmm_AcGraphPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )	strcpy( xmm->graph_plugin, name );

  return xmm->graph_plugin;
}

/*
 * Get/Set name of output plugin
 */
const char *xmm_AcOutputPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )	strcpy( xmm->output_plugin, name );

  return xmm->output_plugin;
}

/*
 * Get/Set name of graph conversion filter plugin
 */
const char *xmm_AcGConvPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )	strcpy( xmm->gconv_plugin, name );

  return xmm->gconv_plugin;
}

/*
 * Get/Set name of output file
 */
const char *xmm_AcOutputFilename( void *_xmm, char *name, char *ext )
{
  XMM	*xmm = (XMM *) _xmm;
  char	*ptr, *ifn = NULL;
  int	len;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  /* Set file name */
  if( name )	ifn = strdup( name );

  if( ifn == NULL )	/* outputFilename set ??? */
  {
	if( xmm->oFilename )
	{
	    if( xmm->oFilename[0] != '\0' )
		ifn = strdup( xmm->oFilename );
	}
  }

  if( ifn == NULL )	/* Use input file name */
  {
	if( xmm->iFilename )	ifn = strdup( xmm->iFilename );
  }

  if( ifn == NULL )	/* default name */
  {
	ifn = strdup( "xmmp-output" );
  }

  /* Remove extension */
  if(( ptr = strrchr( ifn, '.' )) == NULL )	len = strlen( ifn );
  else	len = ptr - ifn;

  /* Free output file name */
  if( xmm->oFilename != NULL )	free( xmm->oFilename );

  /* Allocate output file name */
  if(( xmm->oFilename = malloc( len + 7 )) == NULL )
  {
	free( ifn );
	xmm_SetError( xmm, XMM_ERR_ALLOC, __FUNCTION__ "() memory for output filename." );
	return NULL;
  }

  memcpy( xmm->oFilename, ifn, len );
  xmm->oFilename[len] = '\0';

  if( ext )
  {
	if( !strcmp( &ifn[len], ext ))	strcat( xmm->oFilename, ".2" );
	strcat( xmm->oFilename, ext );
  }

  free( ifn );
  return xmm->oFilename;
}
