/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <dirent.h>
#include <string.h>
#include "libxmm/util/list.h"
#include "libxmm/util/utils.h"
#include "libxmm/version.h"
#include "libxmm/plugin.h"

/*
 * Register plugin of specified type
 */

XMM_Plugin *xmm_PluginRegisterType( char *filename, int type )
{
  void		*handle;
  XMM_Plugin	*plugin;
  const char	*error;

  if( filename == NULL )	return NULL;

  if(( handle = dlopen( filename, RTLD_LAZY )) == NULL )
  {
	xmm_logging( 1, "DLL! Error loading plugin '%s': %s )\n", filename, dlerror());
	return NULL;
  }

  plugin = (XMM_Plugin *) dlsym( handle, "plugin_info" );
  if((error = dlerror()) != NULL )
  {
	xmm_logging( 1, "DLL! Error loading symbol ( %s ) from '%s'\n", error, filename );
	return NULL;
  }

  if( strncmp( plugin->ID, "XMMP", 4 ))
  {
	xmm_logging( 1, "DLL! Plugin Error ( Wrong ID ): %s\n", filename );
	return NULL;
  }

  if( !( plugin->Type & type ))
  {
	xmm_logging( 1, "DLL! Wrong Type ( %x ). Requested was %x\n", plugin->Type, type );
	dlclose( (void *) plugin );
	return NULL;
  }

  plugin->Handle = handle;
  strcpy( plugin->Filename, filename );

  xmm_logging( 2, "DLL! Plugin loaded ( type 0x%x ): %s ( v%i.%i.%i.%i ) [%s]\n",
		plugin->Type, plugin->Name,
		XMM_VERSION_GETMAJOR( plugin->Version ), XMM_VERSION_GETMINOR( plugin->Version ),
		XMM_VERSION_GETMICRO( plugin->Version ), XMM_VERSION_GETPATCH( plugin->Version ),
		plugin->Filename );

  return plugin;
}

/*
 * Register Plugin
 */

XMM_Plugin *xmm_PluginRegister( char *filename )
{
  return xmm_PluginRegisterType( filename, 0xFFFF );
}

/*
 * Register plugin from data
 */

XMM_Plugin *xmm_PluginRegisterFromData( XMM_Plugin *plugin )
{
  if( plugin == NULL )	return (XMM_Plugin *)NULL;

  plugin->Handle = NULL;
  strcpy( plugin->Filename, "Plugin from data" );

  xmm_logging( 2, "DLL! Plugin loaded ( type 0x%x ): %s ( v%i.%i.%i,%i ) [%s]\n",
		    plugin->Type, plugin->Name,
		    XMM_VERSION_GETMAJOR( plugin->Version ),
		    XMM_VERSION_GETMINOR( plugin->Version ),
		    XMM_VERSION_GETMICRO( plugin->Version ),
		    XMM_VERSION_GETPATCH( plugin->Version ),
		    plugin->Filename );

  return plugin;
}

/*
 * Remove plugin
 */

void xmm_PluginRemove( XMM_Plugin *plugin )
{
  if( plugin->Handle )	dlclose( plugin->Handle );
}

/*
 * Scan directory for plugins
 */

XMM_List *xmm_PluginRegisterDir( char *dirname, int type )
{
  DIR		*ds;
  struct dirent	*dir;
  char		*filename, *ext, slash[2] = "/";
  void		*plugin;
  XMM_List	*list = NULL;

  /* Check for slash at end of dirname */
  if( dirname[strlen(dirname) - 1] == '/' )	slash[0] = '\0';

  ds = opendir( dirname );
  if( ds )
  {
	while(( dir = readdir( ds )) != NULL )
	{
	    filename = malloc( strlen( dirname ) + strlen( dir->d_name ) + 2 );
	    sprintf( filename, "%s%s%s", dirname, slash, dir->d_name );

	    if(( ext = strrchr( dir->d_name, '.' )) != NULL )
		if( !strcmp( ext, ".so" ))
		{
			plugin = xmm_PluginRegisterType( filename, type );
			if( plugin )	list = xmmList_Append( list, plugin );
			else xmm_logging( 1, "DLL! Failed to register %s\n", filename );
		}

	    free( filename );
	}
  }
  else	xmm_logging( 1, "DLL! Cannot open dir %s\n", dirname );

  return list; 
}

/*
 * Remove plugin list
 */

void xmm_PluginRemoveList( XMM_List *list )
{
  XMM_List *le, *next;

  for( le = list; le; le = next )
  {
	next = le->next;
	xmm_PluginRemove( le->data );
  }
}
