/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * buffer.c
 * Buffer Queue implementations
 */

#include <stdlib.h>
#include "libxmm/util/buffer.h"
#include "libxmm/util/utils.h"

/*
 * Initialize buffer queue
 */

int xmmBQ_Init( XMM_BufferQueue *bq )
{
  bq->head = NULL;
  bq->tail = NULL;
  bq->size = 0;

  return 0;
}

/*
 * Remove all buffers from queue
 */

void xmmBQ_Free( XMM_BufferQueue *bq )
{
  while( bq->head )	xmmBQ_HeadRemove( bq );
}

/*
 * Get size of data in buffer queue
 */

int xmmBQ_Size( XMM_BufferQueue *bq )
{
  return bq->size;
}

/*
 * Add data to queue
 * For data == NULL, buffer will be allocated and will be added to queue
 */

void *xmmBQ_Add( XMM_BufferQueue *bq, void *data, int size )
{
  XMM_Buffer	*nbuf;

  if(( nbuf = malloc( sizeof( XMM_Buffer ) + size )) == NULL )
	return NULL;

  nbuf->data = (void *)&nbuf[1];
  if( data )	memcpy( nbuf->data, data, size );

  nbuf->size = size;
  nbuf->pos = 0;
  nbuf->next = NULL;
  nbuf->prev = bq->tail;

  if( bq->tail != NULL )	bq->tail->next = nbuf;
  else	bq->head = bq->tail = nbuf;

  bq->tail = nbuf;
  bq->size += size;

  return nbuf->data;
}

/*
 * Add complete queue to another queue
 */

int xmmBQ_Cat( XMM_BufferQueue *bq, XMM_BufferQueue *bqadd )
{
  while( bqadd->head )
  {
	/* Add buffer to bq queue */
	bqadd->head->next = NULL;
	bqadd->head->prev = bq->tail;

	if( bq->tail != NULL )	bq->tail->next = bqadd->head;
	else	bq->head = bq->tail = bqadd->head;

	bq->tail = bqadd->head;
	bq->size += ( bqadd->head->size - bqadd->head->pos );

	/* Remove buffer from bqadd queue */
	bqadd->head = bqadd->head->next;

	if( bqadd->head )	bqadd->head->prev = NULL;
	else	bqadd->head = bqadd->tail = NULL;
  }

  bqadd->size = 0;

  return 0;
}

/*
 * Read bytes from queue
 */

int xmmBQ_Read( XMM_BufferQueue *bq, void *data, int size )
{
  int len = 0, nlen, blen;

  while( len < size )
  {
	if( bq->head == NULL )	break;

	blen = bq->head->size - bq->head->pos;

	if(( size - len ) <= blen )	nlen = size - len;
	else	nlen = blen;

	if( data )	memcpy( data + len, bq->head->data + bq->head->pos, nlen );
	len += nlen;

        bq->head->pos += nlen;
	bq->size -= nlen;

	if( bq->head->pos == bq->head->size )	xmmBQ_HeadRemove( bq );
  }

  return len;
}

/*
 * Read first buffer ( head )
 */

int xmmBQ_HeadRead( XMM_BufferQueue *bq, void *data, int maxsize )
{
  int blen;

  if( bq->head == NULL )	return 0;

  blen = bq->head->size - bq->head->pos;
  if( maxsize > blen )	maxsize = blen;

  if( data )	memcpy( data, bq->head->data + bq->head->pos, maxsize );

  xmmBQ_HeadRemove( bq );

  return maxsize;
}

/*
 * Get size of first buffer ( head )
 */

int xmmBQ_HeadSize( XMM_BufferQueue *bq )
{
  return bq->head->size - bq->head->pos;
}

/*
 * Get data pointer of first buffer ( head )
 */

void *xmmBQ_HeadData( XMM_BufferQueue *bq )
{
  if( bq->head == NULL )	return NULL;

  return bq->head->data;
}

/*
 * Remove first buffer ( head )
 */

void xmmBQ_HeadRemove( XMM_BufferQueue *bq )
{
  XMM_Buffer *buf = bq->head;

  bq->size -= ( buf->size - buf->pos );
  bq->head = buf->next;

  if( bq->head )	bq->head->prev = NULL;
  else	bq->head = bq->tail = NULL;

  free( buf );
}

/*
 * Set size of last added buffer ( tail ) to a smaller value
 * May be necessary if you have less data then in your xmmBQ_Add() request
 */

int xmmBQ_TailResize( XMM_BufferQueue *bq, int size )
{
  /* Error: Only smaller values */
  if( bq->tail->size < size )	return -1;

  /* size must NOT me smaller then the amount of used data */
  if( size < bq->tail->pos )	return -1;

  /* No negative values */
  if( size < 0 )	return -1;

  /* Allright, no resizing needed */
  if( bq->tail->size == size )	return 0;

  bq->size -= ( bq->tail->size - size );
  bq->tail->size = size;

  return 0;
}
