/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * config.c
 * Configuration manager
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libxmm/util/list.h"
#include "libxmm/util/config.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/system.h"

/*
 * Prototypes
 */

static XMM_ConfigLine *_xmmCfg_FindLine( XMM_ConfigFile *cfg, XMM_ConfigSection *section, char *name );
static XMM_ConfigSection *_xmmCfg_FindSection( XMM_ConfigFile *cfg, char *name );

static XMM_ConfigSection *_xmmCfg_AddSection( XMM_ConfigFile *cfg, char *name );
static XMM_ConfigLine *_xmmCfg_AddLine( XMM_ConfigFile *cfg, XMM_ConfigSection *section, char *name );

static XMM_ConfigLine *_xmmCfg_GetOrCreateLine( XMM_ConfigFile *cfg, char *secname, char *name );

/*
 * Create new XMM_ConfigFile struct
 */

XMM_ConfigFile *xmmCfg_New( void )
{
  XMM_ConfigFile	*cfg;

  if(( cfg = malloc( sizeof( XMM_ConfigFile ))) == NULL )
  {
	xmm_logging( 1, "CFG! ERROR: Cannot malloc memory for XMM_ConfigFile\n" );
	return NULL;
  }

  cfg->sections = NULL;

  return cfg;
}

/*
 * Free memory used XMM_ConfigFile struct
 */

void xmmCfg_Free( XMM_ConfigFile *cfg )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;
  XMM_List		*sle, *lle;

  for( sle = cfg->sections; sle; sle = sle->next )
  {
	section = (XMM_ConfigSection *)sle->data;

	for( lle = section->lines; lle; lle = lle->next )
	{
	    line = (XMM_ConfigLine *)lle->data;

	    free( line->name );
	    free( line->value );
	    free( line );
	}

	xmmList_Free( section->lines );
	free( section->name );
	free( section );
  }

  xmmList_Free( cfg->sections );
}

/*
 * Remove entry
 */

void xmmCfg_Remove( XMM_ConfigFile *cfg, char *secname, char *name )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;

  section = _xmmCfg_FindSection( cfg, secname );

  if( name == NULL )
  {
	cfg->sections = xmmList_Remove( cfg->sections, section );
	return;
  }

  line = _xmmCfg_FindLine( cfg, section, name );
  if( line )
  {
	free( line->name );
	free( line->value );
	free( line );
	section->lines = xmmList_Remove( section->lines, line );
  }
}

/*
 * Read file into memory
 */

XMM_ConfigFile *xmmCfg_ReadFile( char *filename )
{
  XMM_ConfigFile	*cfg;
  XMM_ConfigSection	*section = NULL;
  XMM_ConfigLine	*line;
  FILE 			*stream;
  char			buffer[256], *ptr;

  if(( stream = fopen( filename, "r" )) == NULL )
  {
	xmm_logging( 1, "CFG! ERROR: Cannot open '%s'\n", filename );
	return NULL;
  }

  if(( cfg = xmmCfg_New()) == NULL )	return NULL;

  while( 1 )
  {
	fgets( buffer, 255, stream );
	if( feof( stream ))	break;

	if( buffer[0] == '#' )	continue;
	if( buffer[0] == ';' )	continue;
	if( buffer[0] == '\n' )	continue;

	ptr = strchr( buffer, ';' );
	if( ptr )	*ptr = '\0';

	ptr = strchr( buffer, '\n' );
	if( ptr )	*ptr = '\0';

	if( buffer[0] == '[' )
	{
		ptr = strchr( buffer, ']' );
		if( ptr == NULL )
		{
			xmm_logging( 1, "CFG! Parse ERROR in section name, no ']' found.\n" );
			return NULL;
		}
		*ptr = '\0';

		section = _xmmCfg_AddSection( cfg, buffer + 1 );

	}
	else
	{
		if( section == NULL )
		{
			xmm_logging( 1, "CFG! Parse ERROR: Entry outside section.\n" );
			return NULL;
		}

		ptr = strchr( buffer, '=' );
		if( ptr == NULL )
		{
			xmm_logging( 1, "CFG! Parse ERROR in line, no '=' found.\n" );
			return NULL;
		}
		*ptr++ = '\0';

		line = _xmmCfg_AddLine( cfg, section, buffer );
		if( line == NULL )	return NULL;

		line->value = strdup( ptr );
	}
  }

  fclose( stream );

  return cfg;
}

/*
 * Write memory into file
 */

int xmmCfg_WriteFile( XMM_ConfigFile *cfg, char *filename )
{
  XMM_ConfigSection 	*section;
  XMM_ConfigLine	*line;
  XMM_List		*sle, *lle;
  FILE			*stream;

  if(( stream = fopen( filename, "w" )) == NULL )
  {
	xmm_logging( 1, "ERROR: Cannot open '%s'\n", filename );
	return -1;
  }

  for( sle = cfg->sections; sle; sle = sle->next )
  {
	section = (XMM_ConfigSection *)sle->data;
	fprintf( stream, "[%s]\n", section->name );
	for( lle = section->lines; lle; lle = lle->next )
	{
	    line = (XMM_ConfigLine *)lle->data;
	    fprintf( stream, "%s=%s\n", line->name, line->value );
	}
	fprintf( stream, "\n" );
  }

  fclose( stream );
  return 0;
}

/*
 * Read functions
 */

int xmmCfg_ReadString( XMM_ConfigFile *cfg, char *secname, char *name, char *value )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;

  if( value == NULL )	return -1;
 
  section = _xmmCfg_FindSection( cfg, secname );
  line = _xmmCfg_FindLine( cfg, section, name );

  if( line )
  {
	strcpy( value, line->value );
	return 0;
  }
  else	return -1;
}

int xmmCfg_ReadInt( XMM_ConfigFile *cfg, char *secname, char *name, int *value )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;

  if( value == NULL )	return -1;
 
  section = _xmmCfg_FindSection( cfg, secname );
  line = _xmmCfg_FindLine( cfg, section, name );

  if( line )
  {
	*value = strtol( line->value, NULL, 10 );
	return 0;
  }
  else	return -1;
}

int xmmCfg_ReadBool( XMM_ConfigFile *cfg, char *secname, char *name, int *value )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;

  if( value == NULL )	return -1;
 
  section = _xmmCfg_FindSection( cfg, secname );
  line = _xmmCfg_FindLine( cfg, section, name );

  if( line )
  {
	if( !strcasecmp( line->value, "TRUE" ))
	{
	    *value = 1;
	    return 0;
	}
	else if( !strcasecmp( line->value, "FALSE" ))
	{
	    *value = 0;
	    return 0;
	}
	else return -1;
  }
  else	return -1;
}

int xmmCfg_ReadDouble( XMM_ConfigFile *cfg, char *secname, char *name, double *value )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;

  if( value == NULL )	return -1;
 
  section = _xmmCfg_FindSection( cfg, secname );
  line = _xmmCfg_FindLine( cfg, section, name );

  if( line )
  {
	*value = strtod( line->value, NULL );
	return 0;
  }
  else	return -1;
}

int xmmCfg_ReadStringList( XMM_ConfigFile *cfg, char *secname, char *name, XMM_List **list )
{
  char		buffer[256], string[256];
  int		n;

  for( n = 0; ; n++ )
  {
	sprintf( buffer, "%s%i", name, n );
	if( xmmCfg_ReadString( cfg, secname, buffer, string ) == -1 )	break;
	*list = xmmList_Append( *list, strdup( string ));
  }

  return 0;
}


/*
 * Write functions
 */

int xmmCfg_WriteString( XMM_ConfigFile *cfg, char *secname, char *name, char *value )
{
  XMM_ConfigLine	*line;

  if( value == NULL )	return -1;

  line = _xmmCfg_GetOrCreateLine( cfg, secname, name );
  if( line )
  {
	if( line->value )	free( line->value );
	line->value = strdup( value );
	return 0;
  }
  else	return -1;
}

int xmmCfg_WriteInt( XMM_ConfigFile *cfg, char *secname, char *name, int value )
{
  XMM_ConfigLine	*line;
  char			*ptr;

  line = _xmmCfg_GetOrCreateLine( cfg, secname, name );
  if( line )
  {
	if(( ptr = malloc( 256 )) == NULL )
	{
	    xmm_logging( 1, "CFG! ERROR: Cannot malloc memory for temp string\n" );
	    return -1;
	}
	sprintf( ptr, "%i", value );
	
	if( line->value )	free( line->value );
	line->value = strdup( ptr );

	free( ptr );
	return 0;
  }
  else	return -1;
}

int xmmCfg_WriteBool( XMM_ConfigFile *cfg, char *secname, char *name, int value )
{
  XMM_ConfigLine	*line;
 
  line = _xmmCfg_GetOrCreateLine( cfg, secname, name );
  if( line )
  {
	if( line->value )	free( line->value );
	if( value )	line->value = strdup( "TRUE" );
	else	line->value = strdup( "FALSE" );

	return 0;
  }
  else	return -1;
}

int xmmCfg_WriteDouble( XMM_ConfigFile *cfg, char *secname, char *name, double value )
{
  XMM_ConfigLine	*line;
  char			*ptr;
 
  line = _xmmCfg_GetOrCreateLine( cfg, secname, name );
  if( line )
  {
	if(( ptr = malloc( 256 )) == NULL )
	{
	    xmm_logging( 1, "CFG! ERROR: Cannot malloc memory for temp string\n" );
	    return -1;
	}
	sprintf( ptr, "%f", value );
	
	if( line->value )	free( line->value );
	line->value = strdup( ptr );

	free( ptr );
	return 0;
  }
  else	return -1;
}

int xmmCfg_WriteStringList( XMM_ConfigFile *cfg, char *secname, char *name, XMM_List **list )
{
  XMM_List	*le;
  char		buffer[256];
  int		n;

  for( n = 0, le = *list; le; le = le->next, n++ )
  {
	sprintf( buffer, "%s%i", name, n );
	xmmCfg_WriteString( cfg, secname, buffer, (char *)le->data );
  }

  return 0;
}

/*
 * Config Block
 */

int xmmCfg_BlockLoad( char *filename, char *section, XMM_ConfigBlock *ucfg )
{
  XMM_ConfigFile *cfg;

  filename = xmm_gethome_filename( filename );
  if( xmmSYS_access( filename, 0 ) != 1 )	return -1;

  cfg = xmmCfg_ReadFile( filename );
  free( filename );
  if( cfg == NULL )	return -1;

  while( ucfg->ptr )
  {
	switch( ucfg->type )
	{
	    case XMM_CFG_TYPE_INT:
		    xmmCfg_ReadInt( cfg, section, ucfg->name, (int *)ucfg->ptr );
		    break;

	    case XMM_CFG_TYPE_STRING:
		    xmmCfg_ReadString( cfg, section, ucfg->name, (char *)ucfg->ptr );
		    break;

	    case XMM_CFG_TYPE_BOOL:
		    xmmCfg_ReadBool( cfg, section, ucfg->name, (int *)ucfg->ptr );
		    break;

	    case XMM_CFG_TYPE_DOUBLE:
		    xmmCfg_ReadDouble( cfg, section, ucfg->name, (double *)ucfg->ptr );
		    break;

	    case XMM_CFG_TYPE_STRING_LIST:
		    xmmCfg_ReadStringList( cfg, section, ucfg->name, (XMM_List **)ucfg->ptr );
		    break;

	    default:
		    break;
	}
	ucfg++;
  }
  
  xmmCfg_Free( cfg );
  return 0;
}

int xmmCfg_BlockSave( char *filename, char *section, XMM_ConfigBlock *ucfg )
{
  XMM_ConfigFile *cfg;

  filename = xmm_gethome_filename( filename );
  cfg = xmmCfg_ReadFile( filename );
  if( cfg == NULL )	cfg = xmmCfg_New();
  if( cfg == NULL )
  {
	free( filename );
	return -1;
  }

  while( ucfg->ptr )
  {
	switch( ucfg->type )
	{
	    case XMM_CFG_TYPE_INT:
		    xmmCfg_WriteInt( cfg, section, ucfg->name, *((int *)ucfg->ptr ));
		    break;

	    case XMM_CFG_TYPE_STRING:
		    xmmCfg_WriteString( cfg, section, ucfg->name, (char *)ucfg->ptr );
		    break;

	    case XMM_CFG_TYPE_BOOL:
		    xmmCfg_WriteBool( cfg, section, ucfg->name, *((int *)ucfg->ptr ));
		    break;

	    case XMM_CFG_TYPE_DOUBLE:
		    xmmCfg_WriteDouble( cfg, section, ucfg->name, *((double *)ucfg->ptr ));
		    break;

	    case XMM_CFG_TYPE_STRING_LIST:
		    xmmCfg_WriteStringList( cfg, section, ucfg->name, (XMM_List **)ucfg->ptr );
		    break;

	    default:
		    break;
	}
	ucfg++;
  }
  
  xmmCfg_WriteFile( cfg, filename );
  xmmCfg_Free( cfg );

  free( filename );
  return 0;
}

/*
 * Internal
 */

static XMM_ConfigSection *_xmmCfg_FindSection( XMM_ConfigFile *cfg, char *name )
{
  XMM_List	*sle;

  if( cfg == NULL )	return NULL;

  for( sle = cfg->sections; sle; sle = sle->next )
	if( !strcmp( ((XMM_ConfigSection *)sle->data)->name, name ))
		return (XMM_ConfigSection *)sle->data;

  return NULL;
}

static XMM_ConfigLine *_xmmCfg_FindLine( XMM_ConfigFile *cfg, XMM_ConfigSection *section, char *name )
{
  XMM_List	*lle;

  if(( cfg == NULL ) || ( section == NULL ))	return NULL;

  for( lle = section->lines; lle; lle = lle->next )
	if( !strcmp( ((XMM_ConfigLine *)lle->data)->name, name ))
		return (XMM_ConfigLine *)lle->data;

 return NULL;
}

static XMM_ConfigSection *_xmmCfg_AddSection( XMM_ConfigFile *cfg, char *name )
{
  XMM_ConfigSection *section;

  if(( section = malloc( sizeof( XMM_ConfigSection ))) == NULL )
  {
    xmm_logging( 1, "CFG! ERROR: Cannot malloc memory for XMM_ConfigSection\n" );
    return NULL;
  }

  section->name = strdup( name );
  section->lines = NULL;

  cfg->sections = xmmList_Append( cfg->sections, section );

  return section;
}

static XMM_ConfigLine *_xmmCfg_AddLine( XMM_ConfigFile *cfg, XMM_ConfigSection *section, char *name )
{
  XMM_ConfigLine *line;

  if(( line = malloc( sizeof( XMM_ConfigLine ))) == NULL )
  {
    xmm_logging( 1, "CFG! ERROR: Cannot malloc memory for XMM_ConfigLine\n" );
    return NULL;
  }

  line->name = strdup( name );
  line->value = NULL;

  section->lines = xmmList_Append( section->lines, line );

  return line;
}

static XMM_ConfigLine *_xmmCfg_GetOrCreateLine( XMM_ConfigFile *cfg, char *secname, char *name )
{
  XMM_ConfigSection	*section;
  XMM_ConfigLine	*line;

  section = _xmmCfg_FindSection( cfg, secname );
  if( section == NULL )	section = _xmmCfg_AddSection( cfg, secname );
  if( section == NULL )
  {
	xmm_logging( 1, "CFG! ERROR adding section\n" );
	return NULL;
  }
  
  line = _xmmCfg_FindLine( cfg, section, name );
  if( line == NULL )	line = _xmmCfg_AddLine( cfg, section, name );
  if( line == NULL )
  {
	xmm_logging( 1, "CFG! ERROR adding line\n" );
	return NULL;
  }

  return line;
}
