/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mmaccelx86-proc.c
 * Check for x86 extensions, using proc file system
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <inttypes.h>
#include "libxmm/util/mmaccel.h"
#include "libxmm/util/utils.h"

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Definitions
 */

#define	CPU_FAMILY			"cpu family"
#define	CPU_CLOCK			"cpu MHz"
#define	CPU_STEPPING			"stepping"
#define	CPU_FLAGS			"flags"

/*
 * Data
 */

struct
{
	uint32_t	mask;
	char		id[12];
	char		desc[240];
    
} flagtab[] =
{
    { XMM_MMACCEL_X86_FPU,	"fpu",		"Floating-point unit on-chip" },
    { XMM_MMACCEL_X86_TSC,	"tsc",		"Time Stamp Counter" },
    { 0,			"msr",		"Pentium Processor MSR" },
    { 0,			"cx8",		"CMPXCHG8B Instruction Supported" },
    { 0,			"mtrr",		"Memory Type Range Registers" },
    { 0,			"cmov",		"Conditional Move/Compare Instruction" },
    { 0,			"cflsh",	"CFLUSH instruction" },
    { XMM_MMACCEL_X86_MMX,	"mmx",		"MMX Technology" },
    { 0,			"fxsr",		"FXSAVE/FXRSTOR" },
    { XMM_MMACCEL_X86_SSE,	"sse",		"SSE Extensions" },
    { XMM_MMACCEL_X86_SSE2,	"sse2",		"SSE2 Extensions" },
    { 0,			"",		 "" },
};

/*
 *
 */
uint32_t accel_x86( uint32_t *cpu_freq )
{
  char		line[256], *v, *end;
  FILE		*stream;
  uint32_t	caps = 0, val, i;
  float		freq;

  /* Open file */
  if(( stream = fopen( "/proc/cpuinfo", "r" )) == NULL )	return 0;

  /* Read lines */
  while( 1 )
  {
	v = fgets( line, 255, stream );
	if(( v == NULL ) && feof( stream ))	break;

	/* Remove newline */
	v = strchr( line, '\n' );
	if( v )	*v = '\0';

	/* Empty line */
	if( line[0] == '\0'  )	continue;

	/* Find value */
	v = strchr( line, ':' );
	if( v == NULL )
	{
	    xmm_logging( 1, "mmaccel-x86! Parse error: no ':' in line\n" );
	    continue;
	}
	for( v++; *v == ' '; v++ );


	/*
	 *
	 */
	if( strncmp( line, CPU_FLAGS, strlen( CPU_FLAGS )) == 0 )
	{
		for( i = 0; flagtab[i].id[0] != '\0'; i++ )
		{
			if( strstr( v, flagtab[i].id ) != NULL )
			{
			    caps |= flagtab[i].mask;
#ifdef VERBOSE
			    xmm_logging( 1, "mmaccel-x86! CPU flag = %s\n", flagtab[i].desc );
#endif
			}
		}
	}

	if( strncmp( line, CPU_FAMILY, strlen( CPU_FAMILY )) == 0 )
	{
		val = strtol( v, &end, 10 );

		caps |= ( val << XMM_MMACCEL_CPUSHIFT );

#ifdef VERBOSE
		xmm_logging( 1, "mmaccel-x86! CPU = i%i86\n", val );
#endif
	}

	if( strncmp( line, CPU_CLOCK, strlen( CPU_CLOCK )) == 0 )
	{
		freq = strtod( v, &end );

		if( cpu_freq )	*cpu_freq = (uint32_t)( freq * 1000 * 1000 );

#ifdef VERBOSE
		xmm_logging( 1, "mmaccel-x86! CPU clock = %f MHz\n", freq );
#endif
	}


#ifdef VERBOSE

	if( strncmp( line, CPU_STEPPING, strlen( CPU_STEPPING )) == 0 )
	{
		val = strtol( v, &end, 10 );

		xmm_logging( 1, "mmaccel-x86! CPU revision = %i\n", val );
	}
#endif
  }

  fclose( stream );

  return caps;
}
