/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Player interface to XMMP
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include <unistd.h>
#include <sys/time.h>

#include <libxmm/libxmm.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/thread.h>
#include <libxmmplay/xmmplay.h>

/*
 * Definitions
 */

#define	FRAG_SIZE	8192
#define	FRAG_COUNT	16

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get sync info */
#ifndef DEBUG_SYNC
#undef	DEBUG_SYNC
#endif

struct XMM_Play
{
    void		*xmm;
    XMM_FileInfo	fi;

    /* Plugins */
    XMM_PluginSound	*pSound;
    XMM_PluginGraph	*pGraph;

    XMM_PluginInput	*pInput;

    /* Play control */
    int			paused;
    int			playing;
    double		seek;

    /* Audio buffer */
    char		*abuffer;
    int			abuffersize;
    int			bps;

    /* Video buffer */
    char		*vbuffer;
    int			vbuffersize;
    uint8_t		*image[3];

    /* Plugin caps */
    uint32_t		mode;

    /* Player thread stuff */
    XMM_Thread		*player_thread_audio;
    XMM_Thread		*player_thread_video;
    int			exit_player_thread;
};

/*
 * Prototypes
 */

static int player_thread_audio( void *arg );
static int player_thread_video( void *arg );

/*
 * Init library
 */
XMM_Play *xmmplay_Init( void )
{
  XMM_Play	*xmmplay;

  /* Allocate object */
  if(( xmmplay = malloc( sizeof( XMM_Play ))) == NULL )
  {
	xmm_logging( 1, __FUNCTION__ "() Cannot allocate XMM_Play object\n" );
	return NULL;
  }
  memset( xmmplay, 0, sizeof( XMM_Play ));

  /* Initialize libxmm */
  if(( xmmplay->xmm = xmm_Init()) == NULL )
  {
	return NULL;
  }

  return xmmplay;
}

/*
 * Exit library
 */
void xmmplay_Exit( XMM_Play *xmmplay )
{
  /* Exit libxmm */
  xmm_Exit( xmmplay->xmm );

  free( xmmplay );
}

/*
 * Open file
 */
int xmmplay_Open( XMM_Play *xmmplay, char *filename, int flags )
{
  uint32_t	iformat, gformat, yflip, csize;
  int		ret;

  /* Open File */
  xmmplay->pInput = xmm_InputOpen( xmmplay->xmm, filename, flags, &xmmplay->mode );
  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  /* Initialize Audio */
  xmmplay->fi.astreams = xmm_InputAudioStreams( xmmplay->pInput );
  if( xmmplay->fi.astreams )
  {
    /* Get audio stream information */

    xmm_InputAudioInfo( xmmplay->pInput, 0, &xmmplay->fi.ai[0], NULL );

    if( xmmplay->mode & XMM_INPUT_CF_MODE_READ )
    {
	/* Initialize sound output */
	xmmplay->pSound = xmm_SoundOpen( xmmplay->xmm, (char *)xmm_AcSoundPluginName( xmmplay->xmm, NULL ));
	if( xmmplay->pSound == NULL )	return XMM_RET_ERROR;

	/* Init sound output */
	xmmplay->abuffersize = xmm_SoundStart( xmmplay->pSound, &xmmplay->fi.ai[0].format, FRAG_SIZE, FRAG_COUNT );
	if( xmmplay->abuffersize < XMM_RET_OK )	return XMM_RET_ERROR;

	/* Calculate bytes per sample */
	xmmplay->bps = xmmplay->fi.ai[0].format.channels * ( xmmplay->fi.ai[0].format.format & XMM_SOUND_MASK_SIZE ) / 8;

	/* Allocate audio buffer */
	if(( xmmplay->abuffer = malloc( xmmplay->abuffersize * xmmplay->bps )) == NULL )
	{
		xmm_SoundClose( xmmplay->pSound );
		return xmm_SetError( xmmplay->xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Audio buffer" );
	}
    }
  }

  /* Initialize Video */
  xmmplay->fi.vstreams = xmm_InputVideoStreams( xmmplay->pInput );
  if( xmmplay->fi.vstreams )
  {
    /* Get video stream information */
    xmm_InputVideoInfo( xmmplay->pInput, 0, &xmmplay->fi.vi[0], NULL );

    if( xmmplay->mode & XMM_INPUT_CF_MODE_READ )
    {
	/* Initialize graph output */
	xmmplay->pGraph = xmm_GraphOpen( xmmplay->xmm, (char *)xmm_AcGraphPluginName( xmmplay->xmm, NULL ));
	if( xmmplay->pGraph == NULL )	return XMM_RET_ERROR;

	/* Find format */
	xmm_GraphControl( xmmplay->pGraph, XMM_CTLGET_GFORMAT, 0, &gformat );
	ret = xmm_InputControl( xmmplay->pInput, XMM_CTLQUERY_GFORMAT, gformat, NULL );
	if( ret != XMM_CTLRET_TRUE )
	{
		xmm_InputControl( xmmplay->pInput, XMM_CTLGET_GFORMAT, 0, &iformat );
		gformat = iformat;
	}
	else	iformat = gformat;

	/* Need Y - Flipping ? */
	ret = xmm_InputControl( xmmplay->pInput, XMM_CTLQUERY_YFLIP, 0, &yflip );
	if( ret != XMM_CTLRET_ARG )	yflip = 0;

	/* Init graph output */
	if( xmm_GraphStart( xmmplay->pGraph, xmmplay->fi.vi[0].width,
					xmmplay->fi.vi[0].height, gformat,
					yflip ? XMM_GRAPH_FLAG_YFLIP : 0 ) < 0 )
						return XMM_RET_ERROR;

	/* Allocate video buffer */
	xmmplay->vbuffersize = xmmplay->fi.vi[0].width * xmmplay->fi.vi[0].height * 4;
	if(( xmmplay->vbuffer = malloc( xmmplay->vbuffersize )) == NULL )
	{
	    xmm_GraphClose( xmmplay->pGraph );
	    return xmm_SetError( xmmplay->xmm, XMM_ERR_ALLOC, __FUNCTION__ "() Video buffer" );
	}

	/* Set frame rate */
	xmm_GraphControl( xmmplay->pGraph, XMM_CTLSET_FRAMERATE, (uint32_t) xmmplay->fi.vi[0].framerate, NULL );

	/* Set input format */
	ret = xmm_InputControl( xmmplay->pInput, XMM_CTLSET_GFORMAT, iformat, NULL );
	if( ret == XMM_CTLRET_FALSE )
	{
	    xmm_GraphClose( xmmplay->pGraph );
	    return xmm_SetError( xmmplay->xmm, XMM_ERR_UNKNOWN, __FUNCTION__ "() Format %s ( 0x%x ) not supported by codec", xmm_FOURCC_string( iformat ), iformat );
	}

	/* Set image pointer(s) */
	xmmplay->image[0] = xmmplay->vbuffer;

	if( gformat == xmmFOURCC( 'Y','V','1','2' ))
	{
		csize = xmmplay->fi.vi[0].width * xmmplay->fi.vi[0].height / 4;
		xmmplay->image[1] = xmmplay->vbuffer + csize * 4;
		xmmplay->image[2] = xmmplay->vbuffer + csize * 5;
	}
    }
  }

  /* Set player thread for non-direct output */
  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	/* Create thread */
	xmmplay->playing = 0;
	xmmplay->seek = 0.0;
	xmmplay->exit_player_thread = 0;

	if( xmmplay->fi.astreams != 0 )
	{
		xmmplay->player_thread_audio = xmmThread_Create( player_thread_audio, (void *)xmmplay );
	}

	if( xmmplay->fi.vstreams != 0 )
	{
		xmmplay->player_thread_video = xmmThread_Create( player_thread_video, (void *)xmmplay );
	}

  }

  return XMM_RET_OK;
}

/*
 * Close file
 */
void xmmplay_Close( XMM_Play *xmmplay )
{
  /* Stop playback */
  xmmplay_Stop( xmmplay );

  /* Stop player thread */
  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	xmmplay->exit_player_thread = 1;
	if( xmmplay->player_thread_audio )
		xmmThread_Wait( xmmplay->player_thread_audio );

	if( xmmplay->player_thread_video )
		xmmThread_Wait( xmmplay->player_thread_video );
  }

  /* Close input plugin */
  xmm_InputClose( xmmplay->pInput );
  xmmplay->pInput = NULL;

  /* Close sound plugin */
  xmm_SoundClose( xmmplay->pSound );
  xmmplay->pSound = NULL;

  /* Close graph plugin */
  xmm_GraphClose( xmmplay->pGraph );
  xmmplay->pGraph = NULL;
}

/*
 * Play control
 */
int xmmplay_Play( XMM_Play *xmmplay )
{
  int		ret;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	xmmplay->paused = 0;
	xmmplay->playing = 1;
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_PLAY, 0, NULL );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Pause ( toggle )
 */
int xmmplay_Pause( XMM_Play *xmmplay )
{
  int		ret;
  uint32_t	paused;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if( xmmplay->playing == 0 )	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	xmmplay->paused = ( xmmplay->paused + 1 ) & 1;
	xmmplay->playing = !xmmplay->playing;

	return xmmplay->paused;
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_PAUSE, 0, &paused );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
	return paused;
  }
}

/*
 * Stop
 */
int xmmplay_Stop( XMM_Play *xmmplay )
{
  int	ret;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	xmmplay->paused = 0;
	xmmplay->playing = 0;

	xmmplay->pInput->Seek( xmmplay->pInput, 0.0 );
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_STOP, 0, NULL );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Get play status
 */
int xmmplay_Status( XMM_Play *xmmplay )
{
  int		ret;
  uint32_t	status;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	if( xmmplay->playing )	ret = XMM_PBS_PLAYING;
	else	ret = XMM_PBS_STOPPED;

	if(( ret & XMM_PBS_STOPPED ) && xmmplay->paused )
		ret = ( ret & ~XMM_PBS_STOPPED ) | XMM_PBS_PAUSED;

	return ret;
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_STATUS, 0, &status );
	if( ret != XMM_CTLRET_ARG )	return XMM_RET_ERROR;
	return (int)status;
  }
}

/*
 * Seek
 */
int xmmplay_Seek( XMM_Play *xmmplay, double percent )
{
  if(( xmmplay->pInput == NULL ) || ( xmmplay->pInput->Seek == NULL ))	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	/* TODO: */
	if(( xmmplay->fi.astreams != 0 ) && ( xmmplay->fi.vstreams != 0 ))
	    return xmm_SetError( xmmplay->xmm, XMM_ERR_NOTSUPPORTED, "libxmmplay: Seeking only for elementary streams possible" );

	xmmplay->seek = percent;
	return XMM_RET_OK;
  }
  else
  {
	return xmmplay->pInput->Seek( xmmplay->pInput, percent );
  }
}

/*
 * Scale
 */
int xmmplay_Scale( XMM_Play *xmmplay, int width, int height, int flags )
{
  XMM_ControlScale	scale;
  int			ret, status;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  /* If no video, we return OK */
  if( xmmplay->fi.vstreams == 0 )	return XMM_RET_OK;

  /* Scale */
  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	if( xmmplay->pGraph == NULL )	return XMM_RET_ERROR;

	/* Get play status */
	status = xmmplay->playing;
	if( status )	xmmplay->playing = 0;

	/* fill XMM_ControlScale structure */
	scale.width = width;
	scale.height = height;
	scale.flags = flags;

	/* Resize using graph plugin */
	ret = xmmplay->pGraph->Control( xmmplay->pGraph, XMM_CTLSET_SCALE, 0, &scale );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;

	/* restore play status */
	if( status )	xmmplay->playing = 1;
  }
  else
  {
	/* fill XMM_ControlScale structure */
	scale.width = width;
	scale.height = height;
	scale.flags = flags;

	/* Resize using input plugin */
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLSET_SCALE, 0, &scale );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Status & Info
 */
double xmmplay_Info( XMM_Play *xmmplay, XMM_FileInfo *fi, XMM_ClipInfo *ci, double *avdiff, double *percent )
{
  double time, delay = 0;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if( fi )	memcpy( fi, &xmmplay->fi, sizeof( XMM_FileInfo ));
  time = xmmplay->pInput->Info( xmmplay->pInput, ci, avdiff, percent );

  if((( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 ) && xmmplay->pSound )
  {
	xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLGET_DELAY, 0, &delay );
  }

  return time /* - delay */;
}


/*
 * Set Volume
 */
int xmmplay_SetVolume( XMM_Play *xmmplay, int left, int right )
{
  uint32_t	volume = ( right << 16 ) | left;
  int		ret;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
	 ret = xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLSET_VOLUME, volume, NULL );
  else	 ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLSET_VOLUME, volume, NULL );

  if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;

  return XMM_RET_OK;
}

/*
 * Get Volume
 */
int xmmplay_GetVolume( XMM_Play *xmmplay, int *left, int *right )
{
  uint32_t	volume;
  int		ret;

  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
	ret = xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLGET_VOLUME, 0, &volume );
  else	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLGET_VOLUME, 0, &volume );

  if( ret != XMM_CTLRET_ARG )	return XMM_RET_ERROR;

  *left = volume & 0xFFFF;
  *right = volume >> 16;

  return XMM_RET_OK;
}

/*
 * Input plugin dialogues
 */
int xmmplay_InputDialogue( XMM_Play *xmmplay, int dialog )
{
  /* Check arguments */
  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  /* */
  xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLDLG_DISPLAY, dialog, NULL );

  return XMM_RET_OK;
}

/*
 * Internal code ( player threads )
 */

#include <libxmm/util/system.h>
#include <libxmm/util/mmaccel.h>

/*
 * Player thread ( audio only )
 */
static int player_thread_audio( void *arg )
{
  XMM_Play		*xmmplay = (XMM_Play *) arg;
  XMM_PluginInput	*pInput	= xmmplay->pInput;
  XMM_PluginSound	*pSound = xmmplay->pSound;
  int			read;

  do
  {
	if( xmmplay->seek >= 0 )
	{
		pInput->Seek( pInput, xmmplay->seek );
		pSound->Flush( pSound );
		xmmplay->seek = -1.0;
		continue;
	}

	if( xmmplay->playing )
	{
		read = pInput->AudioRead( pInput, 0, xmmplay->abuffer, xmmplay->abuffersize );
		if( read == XMM_RET_EOS )	xmmplay->playing = 0;
		if( read > XMM_RET_OK )	pSound->Write( pSound, xmmplay->abuffer, read );
	}

  }  while( !xmmplay->exit_player_thread );

  return XMM_RET_OK;
}

/*
 * Player thread ( video only )
 */
static int player_thread_video( void *arg )
{
  XMM_Play		*xmmplay = (XMM_Play *) arg;
  XMM_PluginInput	*pInput	= xmmplay->pInput;
  uint64_t		baseTime, currTime;
  uint32_t		videoPTS = 0, freq;
  int32_t		timeDiff;
  int			read;

  freq = xmmCPU_clockfreq() / 1000;	/* kHz */
  baseTime = xmmSYS_longcount() / freq;	/* ms */

  /* Main loop */
  while( xmmplay->exit_player_thread == 0 )
  {
	/* Seek */
	if( xmmplay->seek >= 0 )
	{
		pInput->Seek( pInput, xmmplay->seek );
		pInput->VideoPTS( pInput, 0, &videoPTS );
		baseTime = xmmSYS_longcount() / freq - (uint64_t)videoPTS;	/* ms */
		xmmplay->seek = -1.0;
	}

	/* Playback stopped */
	if( xmmplay->playing == 0 )
	{
		baseTime = xmmSYS_longcount() / freq - (uint64_t)videoPTS;	/* ms */
		continue;
	}
	/* Playback paused */
	if( xmmplay->paused == 1 )	continue;

	/*
	 * Video synchronization
	 */

	currTime = xmmSYS_longcount() / freq; /* ms */
	timeDiff = videoPTS - (uint32_t)( currTime - baseTime ); /* ms */

#ifdef DEBUG_SYNC
	xmm_logging( 1, "\t\t\tVideo: vPTS: %i realPTS: %i async: %i\r",
					videoPTS,
					(int32_t)( currTime - baseTime ),
					timeDiff );
#endif

	/* video to fast: sleep */
	if( timeDiff > 10 )	xmmSYS_usleep(( timeDiff * 1000 ) >> 1 );

	/* video to slow: drop frame */
	if( timeDiff < -150 )	pInput->VideoRead( pInput, 0, NULL );

	/* video very slow: drop one more frame */
	if( timeDiff < -230 )	pInput->VideoRead( pInput, 0, NULL );

	/* Decode frame */
	read = pInput->VideoRead( pInput, 0, xmmplay->image );
	if( read == XMM_RET_EOS )	xmmplay->playing = 0;
	if( read >= XMM_RET_OK )
	{
	    xmmplay->pGraph->Draw( xmmplay->pGraph, xmmplay->image, NULL, 0, 0, xmmplay->fi.vi[0].width, xmmplay->fi.vi[0].height, 0 );
	    xmmplay->pGraph->Blit( xmmplay->pGraph );
	}

	read = pInput->VideoPTS( pInput, 0, &videoPTS );
  }

  return XMM_RET_OK;
}
