/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * ac3.c
 * Codec Plugin for AC3 Audio Decoding
 */

#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpsound.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/error.h>
#include "libac3/ac3.h"

/*
 * Definitions
 */

#define	divUP( a, b )		(((a) + (b) - 1 ) / (b))

/*
 * Types
 */

struct priv_t
{
    XMM_BufferQueue		bq;
    XMM_SoundFormat		format;
    int				framesize;
    int				framesize_pcm;

    uint8_t			*pcm_buffer;
    uint32_t			pcm_buffer_pos;
};

/* for type casting */
typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

/*
 * Codec info
 */

#define	AC3_AF_NUM	2

static XMM_AudioFormat	ac3_af[AC3_AF_NUM] =
{
    { 0x2000, 2, 22050, 22050 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "22 kHz, 16 bit, stereo" },
    { 0x2000, 2, 44100, 44100 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "44 kHz, 16 bit, stereo" }
};

static XMM_CodecAudioInfo	ac3_cai =
{
    XMM_CODEC_ACF_DECODE,
    "AC3",				/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    AC3_AF_NUM,				/* Number of supported formats */
    ac3_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginCodecAudio	plugin_info;

static int			ac3_codec_initialized = 0;

/*
 * Initialize Plugin
 */
static XMM_PluginCodecAudio *ac3_Open( void *xmm, int mode, XMM_AudioFormat *af )
{
  XMM_PluginCodecAudio	*pCodec;
  struct priv_t		*priv;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check format ID */
  if( af->formatID != 0x2000 )	return (void *)XMM_RET_NOTSUPPORTED;

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* AC3 library may only be used once */
  if( ac3_codec_initialized )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(AC3) AC3 codec already in use." );
	return NULL;
  }

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(AC3) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* XMM sound format of decoded data */
  priv->format.samprate = af->samprate;
  priv->format.channels = 2;
  priv->format.format = XMM_SOUND_FMT_S16LE;

  /* Initialize data */
  priv->pcm_buffer_pos = 0;
  priv->framesize = 0;
  priv->framesize_pcm = 0;

  /* Init AC3 and buffer queue */
  ac3dec_init();
  xmmBQ_Init( &priv->bq );
  ac3_codec_initialized = 1;

  return pCodec;
}

/*
 * Free codec
 */
static int ac3_Close( XMM_PluginCodecAudio *codec )
{
  struct priv_t *priv = codec->sys.priv;

  /* free buffer queue */
  xmmBQ_Free( &priv->bq );
  ac3_codec_initialized = 0;

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int ac3_Control( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg )
{
  struct priv_t *priv = codec->sys.priv;
  XMM_SoundFormat *format;

  switch( cmd )
  {
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		if(( format->format == priv->format.format ) &&
		    ( format->samprate = priv->format.samprate ) &&
		    ( format->channels == priv->format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		format->channels = priv->format.channels;
		format->samprate = priv->format.samprate;
		format->format = priv->format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		if( priv->framesize == 0 )	*((uint32_t *)arg ) = 4096;
		else *((uint32_t *)arg ) = divUP( *((uint32_t *)arg ), priv->framesize_pcm ) * priv->framesize;
	        return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FORMAT_SIZE:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	case XMM_CTLGET_FORMAT_DATA:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No CODEC command */
}

/*
 * Decode data
 */
static int ac3_Decode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags )
{
  struct priv_t *priv = codec->sys.priv;
  int		decoded, os = 0, osize, needed;

  osize = *samples * priv->format.channels * 2;

  /*
   * First, we queue the source data
   */
  xmmBQ_Add( &priv->bq, src, isize );

  /*
   * Get unused PCM data
   */
  if( priv->pcm_buffer_pos > 0 )
  {
	/* Calculate needed amount of data */
	os = priv->framesize_pcm - priv->pcm_buffer_pos;
	if( os > osize )	os = osize;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer + priv->pcm_buffer_pos, os );
	dest += os;

	/* Updata buffer position */
	priv->pcm_buffer_pos += os;

	if( priv->pcm_buffer_pos == priv->framesize_pcm )
		priv->pcm_buffer_pos = 0;
  }

  /*
   * Decode data
   */
  while( os < osize )
  {
	/* Decode frame */
	decoded = ac3dec_decode_data( &priv->pcm_buffer, &priv->framesize, (bq_raw_read_t)xmmBQ_Read, (bq_raw_size_t)xmmBQ_Size, &priv->bq );
	if( decoded == 0 )	break;

	/* Set new frame size */
	priv->framesize_pcm = decoded;

	/* Calculate needed amount of data */
	needed = osize - os;
	if( needed > priv->framesize_pcm )	needed = priv->framesize_pcm;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer, needed );

	dest += needed;
	os += needed;

	/* Update buffer pos ( this is the last loop ) */
	if( needed < priv->framesize_pcm )	priv->pcm_buffer_pos = needed;
  }

  if( os == 0 )	return XMM_RET_ERROR;
  *samples = os / ( priv->format.channels * 2 );

  return isize;
}

/*
 * Encode data
 */
static int ac3_Encode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecAudioInfo *ac3_Info( void *xmm )
{
  return &ac3_cai;
}

/*
 * Plugin data
 */
XMM_PluginCodecAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_ACODEC,
				XMM_VERSION_NUM,
				"",
				"AC3",
				"Codec: AC3 ( Decoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				ac3_Open, ac3_Close, ac3_Control,
				ac3_Decode, ac3_Encode,
				ac3_Info };
