/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lame.c
 * Codec Plugin for MPEG Layer 3 Encoding
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpsound.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include "lame/lame.h"

/*
 * Definitions
 */

#define BITRATE			128

/*
 * Types
 */

struct priv_t
{
    XMM_SoundFormat		format;
    XMM_AudioFormat		af;
    lame_global_flags		gf;
};

/*
 * Codec info
 */

#define	LAME_AF_NUM	2

static XMM_AudioFormat	lame_af[LAME_AF_NUM] =
{
    { 0x55, 2, 22050, 22050 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "22 kHz, 16 bit, stereo" },
    { 0x55, 2, 44100, 44100 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "44 kHz, 16 bit, stereo" }
};

static XMM_CodecAudioInfo	lame_cai =
{
    XMM_CODEC_ACF_ENCODE,
    "MPEG Layer 3",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    LAME_AF_NUM,			/* Number of supported formats */
    lame_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginCodecAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginCodecAudio *lame_Open( void *xmm, int mode, XMM_AudioFormat *af )
{
  XMM_PluginCodecAudio	*pCodec;
  struct priv_t 	*priv;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_ENCODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check format ID */
  if( af->formatID != 0x55 )	return (void *)XMM_RET_NOTSUPPORTED;

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(LAME) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* XMM sound format of decoded data */
  priv->format.samprate = af->samprate;
  priv->format.channels = af->channels;
  priv->format.format = af->format;

  /* Output format */
  priv->af.formatID = 0x55;
  priv->af.channels = af->channels;
  priv->af.samprate = af->samprate;
  priv->af.bitrate = BITRATE * 1000;
  priv->af.format = XMM_SOUND_FMT_S16LE;
  priv->af.blockSize = 1;
  priv->af.extraSize = 0;
  strcpy( priv->af.desc, "" );

  /* Initialize lame */
  lame_init( &priv->gf );

  priv->gf.num_channels = priv->format.channels;
  priv->gf.in_samplerate = priv->format.samprate;
  priv->gf.brate = BITRATE;
  priv->gf.mode = 1;
  priv->gf.quality = 2;
  priv->gf.silent = 1;

  lame_init_params( &priv->gf );

  return pCodec;
}

/*
 * Free codec
 */
static int lame_Close( XMM_PluginCodecAudio *codec )
{
  free( codec );

  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int lame_Control( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg )
{
  struct priv_t		*priv = codec->sys.priv;
  XMM_SoundFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		if(( format->format == priv->format.format ) &&
		    ( format->samprate == priv->format.samprate ) &&
		    ( format->channels == priv->format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		format->channels = priv->format.channels;
		format->samprate = priv->format.samprate;
		format->format = priv->format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FORMAT_SIZE:
		*((uint32_t *)arg) = sizeof( XMM_AudioFormat );
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FORMAT_DATA:
		*((char **)arg) = (char *)&priv->af;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;			/* No CODEC command */
}

/*
 * Decode data
 */
static int lame_Decode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Encode data
 */
static int lame_Encode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags )
{
  struct priv_t *priv = codec->sys.priv;
  int		done;

  if( priv->gf.num_channels == 1 )
	done = lame_encode_buffer( &priv->gf, (short *)src, (short *)src, samples, (char *)dest, *osize );
  else
  	done = lame_encode_buffer_interleaved( &priv->gf, (short *)src, samples, (char *)dest, *osize );

  if( done < 0 )	return XMM_RET_ERROR;

  *osize = done;
  return samples;
}

/*
 * Codec Info
 */
static XMM_CodecAudioInfo *lame_Info( void *xmm )
{
  return &lame_cai;
}

/*
 * Plugin data
 */
XMM_PluginCodecAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_ACODEC,
				XMM_VERSION_NUM,
				"",
				"LAME",
				"Codec: MPEG Layer 3 ( Encoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				lame_Open, lame_Close, lame_Control,
				lame_Decode, lame_Encode,
				lame_Info };
