/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mpg123.c
 * Codec Plugin for MPEG 1.0 / 2.0 Layer 1/2/3 Decoding
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpsound.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/error.h>
#include "mpglib/libmpg123.h"
#include "avimp3.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		16384

#define	divUP( a, b )		(((a) + (b) - 1 ) / (b))

/*
 * Types
 */

struct priv_t
{
    mpg123_t			mpg123;
    mpg123_reader_t		reader;

    XMM_BufferQueue		bq;
    XMM_SoundFormat		format;

    int				framesize;
    int				framesize_pcm;

    uint8_t			*pcm_buffer;
    uint32_t			pcm_buffer_pos;
    uint32_t			pcm_buffer_size;
};

/* for type casting */
typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

/*
 * Codec info
 */

#define	MPG123_AF_NUM	2

static XMM_AudioFormat	mpg123_af[MPG123_AF_NUM] =
{
    { 0x55, 2, 22050, 22050 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "22 kHz, 16 bit, stereo" },
    { 0x55, 2, 44100, 44100 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "44 kHz, 16 bit, stereo" }
};

static XMM_CodecAudioInfo	mpg123_cai =
{
    XMM_CODEC_ACF_DECODE,
    "MPEG Layer 1/2/3",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    MPG123_AF_NUM,			/* Number of supported formats */
    mpg123_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginCodecAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginCodecAudio *mpg123_Open( void *xmm, int mode, XMM_AudioFormat *af )
{
  XMM_PluginCodecAudio	*pCodec;
  struct priv_t		*priv;
  wave_mp3format_t	*mp3extra;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check format ID */
  if( af->formatID != 0x55 )	return (void *)XMM_RET_NOTSUPPORTED;

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(MPG123) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* XMM sound format of decoded data */
  priv->format.samprate = af->samprate;
  priv->format.channels = af->channels;
  priv->format.format = XMM_SOUND_FMT_S16LE;

  /* Get frame size */
  if( af->extraSize == WAVE_MP3_EXTRA_SIZE )
  {
	mp3extra = (wave_mp3format_t *) &af[1];
	priv->framesize = mp3extra->nBlockSize / mp3extra->nFramesPerBlock;
	priv->framesize_pcm = 1152 * af->channels * 2;
  }
  else
  {
	xmm_logging( 1, "MPG123! %i bytes extra data found, %i bytes expected for format 0x%.4x\n", af->extraSize, WAVE_MP3_EXTRA_SIZE, af->formatID );

	/* Guess framesize, assumes MPEG 1.0/2.0 Layer 2 or MPEG 1.0 Layer 3 */
	priv->framesize = (int)(((double)af->bitrate / 8 ) / 125 * 144000 / af->samprate );
	priv->framesize_pcm = 1152 * af->channels * 2;
  }

  /* Allocate PCM buffer */
  priv->pcm_buffer_size = PCM_BUFFER_SIZE;
  priv->pcm_buffer_pos = 0;

  if(( priv->pcm_buffer = malloc( priv->pcm_buffer_size )) == NULL )
  {
	free( pCodec );
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(MPG123) PCM buffer" );
	return NULL;
  }

  /* Init reader */
  priv->reader.read = (bq_raw_read_t) xmmBQ_Read;

  /* Init MP3 and buffer queue */
  MPG123_Init( &priv->mpg123, 0 );
  xmmBQ_Init( &priv->bq );

  return pCodec;
}

/*
 * Free codec
 */
static int mpg123_Close( XMM_PluginCodecAudio *codec )
{
  struct priv_t *priv = codec->sys.priv;

  free( priv->pcm_buffer );

  /* Exit mp3 and free buffer queue */
  MPG123_Exit( &priv->mpg123 );
  xmmBQ_Free( &priv->bq );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int mpg123_Control( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg )
{
  struct priv_t *priv = codec->sys.priv;
  XMM_SoundFormat *format;

  switch( cmd )
  {
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		if(( format->format == priv->format.format ) &&
		    ( format->samprate = priv->format.samprate ) &&
		    ( format->channels == priv->format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		format->channels = priv->format.channels;
		format->samprate = priv->format.samprate;
		format->format = priv->format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		/* If data in PCM buffer, calculate less source data */
		if( priv->pcm_buffer_pos > 0 )	*((uint32_t *)arg ) = *((uint32_t *)arg ) - ( priv->framesize_pcm - priv->pcm_buffer_pos );
		*((uint32_t *)arg ) = divUP( *((uint32_t *)arg ), priv->framesize_pcm ) * priv->framesize;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FORMAT_SIZE:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	case XMM_CTLGET_FORMAT_DATA:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	case XMM_CTLSET_SEEK:
		priv->pcm_buffer_pos = 0;
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No CODEC command */
}

/*
 * Decode data
 */
static int mpg123_Decode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags )
{
  struct priv_t *priv = codec->sys.priv;
  int		decoded, ret, os = 0, osize, needed;

  osize = *samples * priv->format.channels * 2;

  /*
   * First, we queue the source data
   */
  xmmBQ_Add( &priv->bq, src, isize );

  /*
   * Get unused PCM data
   */
  if( priv->pcm_buffer_pos > 0 )
  {
	/* Calculate amount of data int PCM buffer */
	os = priv->framesize_pcm - priv->pcm_buffer_pos;
	if( os > osize )	os = osize;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer + priv->pcm_buffer_pos, os );
	dest += os;

	/* Updata buffer position */
	priv->pcm_buffer_pos += os;

	if( priv->pcm_buffer_pos == priv->framesize_pcm )
		priv->pcm_buffer_pos = 0;
  }

  /*
   * Decode data
   */
  while( os < osize )
  {
	/* Decode frame */
	decoded = priv->pcm_buffer_size;
	ret = MPG123_DecodeFrame( &priv->mpg123, priv->pcm_buffer, &decoded, &priv->reader, &priv->bq );
	if( ret != MP3_OK )	break;

	/* Set new frame size */
	priv->framesize_pcm = decoded;

	/* Calculate needed amount of data */
	needed = osize - os;
	if( needed > priv->framesize_pcm )	needed = priv->framesize_pcm;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer, needed );

	dest += needed;
	os += needed;

	/* Update buffer pos ( this is the last loop ) */
	if( needed < priv->framesize_pcm )	priv->pcm_buffer_pos = needed;
  }


  if( os == 0 )	return XMM_RET_ERROR;
  *samples = os / ( priv->format.channels * 2 );

  return isize;
}

/*
 * Encode data
 */
static int mpg123_Encode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecAudioInfo *mpg123_Info( void *xmm )
{
  return &mpg123_cai;
}

/*
 * Plugin data
 */
XMM_PluginCodecAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_ACODEC,
				XMM_VERSION_NUM,
				"",
				"MPG123",
				"Codec: MPEG Layer 1/2/3 ( Decoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				mpg123_Open, mpg123_Close, mpg123_Control,
				mpg123_Decode, mpg123_Encode,
				mpg123_Info };
