/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * opendivx.c
 * Codec Plugin for OpenDivX
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpgraph.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

#include <libdivxdecore/decore.h>

/*
 * Definitions
 */

	/* decore handle */
#define MY_APP_ID		0x12345
#define	DIVX_POSTPROC		0
#define	DIVX_BPP		32

/*
 * Types
 */

struct priv_t
{
};

/*
 * Codec info
 */

static XMM_CodecVideoInfo	opendivx_cvi =
{
    XMM_CODEC_VCF_DECODE,
    "OpenDivX",				/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    xmmFOURCC( 'd','v','x','1' ),	/* FOURCC */
};

/*
 * Global data
 */

extern XMM_PluginCodecVideo	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginCodecVideo *opendivx_Open( void *xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_PluginCodecVideo	*pCodec;
  struct priv_t		*priv;
  DEC_PARAM		dec_param;
  DEC_SET		dec_set;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check codec */
  switch( vf->codec )
  {
	case	xmmFOURCC( 'd','v','x','1' ):
			break;

	default:	return (void *)XMM_RET_NOTSUPPORTED;
  }

  /* Only query codec  */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(OpenDivX) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Initialize decoding */
  dec_param.x_dim = vf->width;
  dec_param.y_dim = vf->height;
  decore( MY_APP_ID, DEC_OPT_INIT, &dec_param, NULL );

  /* Postprocessing */
  dec_set.postproc_level = DIVX_POSTPROC;
  decore( MY_APP_ID, DEC_OPT_SETPP, &dec_set, NULL );

  return pCodec;
}

/*
 * Free codec
 */
static int opendivx_Close( XMM_PluginCodecVideo *codec )
{
  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int opendivx_Control( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  switch( cmd )
  {
    case XMM_CTLQUERY_GFORMAT:
	    if(((uint32_t)arg) == xmmFOURCC( 'Y','V','1','2' ))	return XMM_CTLRET_TRUE;
	    return XMM_CTLRET_FALSE;

    case XMM_CTLQUERY_YFLIP:
	    *((uint32_t *)arg) = 0;
	    return XMM_CTLRET_ARG;

    case XMM_CTLGET_GFORMAT:
	    *((uint32_t *)arg) = xmmFOURCC( 'Y','V','1','2' );
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLGET_FORMAT_SIZE:
	    return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLGET_FORMAT_DATA:
	    return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLSET_GFORMAT:
	    return XMM_CTLRET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No CODEC command */
}

/*
 * Decode data
 */
static int opendivx_Decode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  DEC_FRAME dec_frame;

  dec_frame.length	= isize;
  dec_frame.bitstream 	= src;
  dec_frame.bmp		= (void *) dest;
  dec_frame.render_flag	= 1;

  decore( MY_APP_ID, 0, &dec_frame, NULL );

  return isize;
}

/*
 * Encode data
 */
static int opendivx_Encode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecVideoInfo *opendivx_Info( void *xmm )
{
  return &opendivx_cvi;
}

/*
 * Plugin data
 */
XMM_PluginCodecVideo plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_VCODEC,
				XMM_VERSION_NUM,
				"",
				"OpenDivX",
				"Codec: OpenDivX ( Decoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				opendivx_Open, opendivx_Close, opendivx_Control,
				opendivx_Decode, opendivx_Encode,
				opendivx_Info };
