/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * w49gsm.c
 * Codec Plugin for WAVE GSM 6.10 Audio ( format 49 ) Decoding
 */

#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpsound.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include "gsm.h"

/*
 * Definitions
 */

#define	divUP( a, b )	(((a) + (b) - 1 ) / (b))

/*
 * Types
 */

struct priv_t
{
    gsm				gsm_handle;
    gsm_signal			gsm_dest[320];
    int				gsm_left;
    XMM_SoundFormat		format;
};

/*
 * Codec info
 */

#define GSM_AF_NUM	2

static XMM_AudioFormat	gsm_af[GSM_AF_NUM] =
{
    { 0x31, 2, 22050, 22050 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "22 kHz, 16 bit, stereo" },
    { 0x31, 2, 44100, 44100 * 4, XMM_SOUND_FMT_S16LE, 1, 0, "44 kHz, 16 bit, stereo" }
};

static XMM_CodecAudioInfo	gsm_cai =
{
    XMM_CODEC_ACF_DECODE,
    "Wave GSM 6.10",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    GSM_AF_NUM,				/* Number of supported formats */
    gsm_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginCodecAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginCodecAudio *gsm_Open( void *xmm, int mode, XMM_AudioFormat *af )
{
  XMM_PluginCodecAudio	*pCodec;
  struct priv_t		*priv;
  int			value = 1;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check format ID */
  if( af->formatID != 0x31 )	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check extra data */
  if( af->extraSize != 2 )
	xmm_logging( 1, "WARNING: WAV49 GSM with %i bytes extradata found. %i Expected.\n", af->extraSize, 2 );

#if 0
  if(((uint16_t *)&af[1])[0] != 320 )	/* WORD	wSamplesPerBlock */;
  {
	return xmm_SetError( xmm, XMM_ERR_UNKNOWN, "WAV49 GSM! 320 SamplesPerBlock expected. %i found.\n", ((uint16_t *)&af[1])[0] );
	return NULL;
  }
#endif

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(WAV49 GSM) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* XMM sound format of decoded data */
  priv->format.samprate = af->samprate;
  priv->format.channels = af->channels;
  priv->format.format = XMM_SOUND_FMT_S16LE;

  /* Init GSM stuff */
  if(( priv->gsm_handle = gsm_create()) == NULL )
  {
	free( pCodec );
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "WAV49 GSM! Cannot initialize gsm pass." );
	return NULL;
  }

  if( gsm_option( priv->gsm_handle, GSM_OPT_WAV49, &value ) == -1 )
  {
	free( pCodec );
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "WAV49 GSM! Cannot set WAV-style byte ordering option. Not enabled in gsm library ?" );
	return NULL;
  }

  priv->gsm_left = 0;

  return pCodec;
}

/*
 * Free codec
 */
static int gsm_Close( XMM_PluginCodecAudio *codec )
{
  struct priv_t *priv = codec->sys.priv;

  /* destroy gsm object */
  gsm_destroy( priv->gsm_handle );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int gsm_Control( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg )
{
  struct priv_t *priv = codec->sys.priv;
  XMM_SoundFormat *format;

  switch( cmd )
  {
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		if(( format->format == priv->format.format ) &&
		    ( format->samprate = priv->format.samprate ) &&
		    ( format->channels == priv->format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		format->channels = priv->format.channels;
		format->samprate = priv->format.samprate;
		format->format = priv->format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)arg ) = divUP( *((uint32_t *)arg ), 640 ) * 65;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FORMAT_SIZE:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	case XMM_CTLGET_FORMAT_DATA:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No CODEC command */
}

/*
 * Decode data
 */
static int gsm_Decode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags )
{
  struct priv_t *priv = codec->sys.priv;
  int		ipos = 0, bytes = 0, done, osize;

  memcpy( dest, &priv->gsm_dest[ 320 - priv->gsm_left ], priv->gsm_left * 2 );
  dest += priv->gsm_left * 2;
  done = priv->gsm_left;
  priv->gsm_left = 0;
  osize = *samples * priv->format.channels * 2;

  while( done < osize )
  {
	if(( ipos + 65 ) > isize )
		return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(WAV49 GSM) Only %i bytes left in source. Frame size = %i bytes\n", bytes, 65 );

	if( gsm_decode( priv->gsm_handle, src, priv->gsm_dest ) < 0 )
		return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(WAV49 GSM) Cannot decode" );

	if( gsm_decode( priv->gsm_handle, src + 33, priv->gsm_dest + 160 ) < 0 )
		return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(WAV49 GSM) Cannot decode" );

	bytes = ( osize - done );
	if( bytes < 320 )	priv->gsm_left = 320 - bytes;
	else	bytes = 320;

	memcpy( dest, priv->gsm_dest, bytes * 2 );
	dest += bytes * 2;
	done += bytes * 2;
	src += 65;
	ipos += 65;
  }

  if( done < osize )	osize = done;
  *samples = osize / ( priv->format.channels * 2 );

  return ipos;
}

/*
 * Encode data
 */
static int gsm_Encode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecAudioInfo *gsm_Info( void *xmm )
{
  return &gsm_cai;
}

/*
 * Plugin data
 */
XMM_PluginCodecAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_ACODEC,
				XMM_VERSION_NUM,
				"",
				"GSM610",
				"Codec: WAV GSM 6.10 ( Decoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				gsm_Open, gsm_Close, gsm_Control,
				gsm_Decode, gsm_Encode,
				gsm_Info };
