/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wacm.c
 * Codec Plugin for ACM (DLL) codecs
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpsound.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/error.h>
#include <libxmm/config.h>

#include "wine/windef.h"
#include "wine/driver.h"
#include "wine/msacm.h"

/* Change 'undef' in 'define' for verbose mode */
#ifndef VERBOSE
#define	VERBOSE
#endif

/*
 * Types
 */

struct priv_t
{
    XMM_SoundFormat		format;

    HACMSTREAM			acm_stream_handle;		/* acm stream handle */
    WAVEFORMATEX		*acm_format_source;		/* Input */
    WAVEFORMATEX		*acm_format_dest;		/* Output */
};

/*
 * Codec info
 */

static XMM_CodecAudioInfo	acm_cai =
{
    XMM_CODEC_ACF_DECODE,
    "WMA ( W32 DLL )",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginCodecAudio	plugin_info;

/*
 * Prototypes
 */

static int WMAudioCodecOpen( XMM_PluginCodecAudio *codec, XMM_AudioFormat *af, int query );
static int WMAudioCodecClose( XMM_PluginCodecAudio *codec  );
#ifdef VERBOSE
static void print_waveformatex( WAVEFORMATEX *wave_fmt_head );
#endif
/*
 * Initialize Plugin
 */
static XMM_PluginCodecAudio *acm_Open( void *xmm, int mode, XMM_AudioFormat *af )
{
  XMM_PluginCodecAudio	*pCodec;
  struct priv_t		*priv;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )
  {
	return (void *)xmm_AcCodecFile( xmm, af->formatID, XMM_CFG_CODEC_ACM, NULL );
  }

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(ACM) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Open codec */
  if( WMAudioCodecOpen( pCodec, af, 0 ) < 0 )
  {
	free( pCodec );
	return NULL;
  }

  /* XMM sound format of decoded data */
  priv->format.samprate = af->samprate;
  priv->format.channels = af->channels;
  priv->format.format = XMM_SOUND_FMT_S16LE;

  return pCodec;
}

/*
 * Free codec
 */
static int acm_Close( XMM_PluginCodecAudio *codec )
{
  /* Close WMA */
  WMAudioCodecClose( codec );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int acm_Control( XMM_PluginCodecAudio *codec, uint32_t cmd, void *arg )
{
  struct priv_t		*priv = codec->sys.priv;
  XMM_SoundFormat	*format;
  DWORD			srcsize;

  switch( cmd )
  {
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		if(( format->format == priv->format.format ) &&
		    ( format->samprate = priv->format.samprate ) &&
		    ( format->channels == priv->format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)arg;
		format->channels = priv->format.channels;
		format->samprate = priv->format.samprate;
		format->format = priv->format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		acmStreamSize( priv->acm_stream_handle, *((uint32_t *)arg ), &srcsize, ACM_STREAMSIZEF_DESTINATION );
		*((uint32_t *)arg ) = (uint32_t)srcsize;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FORMAT_SIZE:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	case XMM_CTLGET_FORMAT_DATA:
		return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No CODEC command */
}

/*
 * Decode data
 */
static int acm_Decode( XMM_PluginCodecAudio *codec, uint8_t *src, int isize, uint8_t *dest, uint32_t *samples, int *flags )
{
  struct priv_t		*priv = codec->sys.priv;
  HRESULT		h;
  ACMSTREAMHEADER	ash;
  DWORD			srcsize, used;
  int			osize;

  osize = *samples * priv->format.channels * 2;

  /* Get needed input data size */
  acmStreamSize( priv->acm_stream_handle, osize, &srcsize, ACM_STREAMSIZEF_DESTINATION );
  if( srcsize > isize )
	return xmm_SetError( codec->sys.xmm, XMM_ERR_MOREDATA, "(ACM) input data" );

  /*
   * Decode data
   */
  memset( &ash, 0, sizeof( ACMSTREAMHEADER ));
  ash.cbStruct = sizeof( ACMSTREAMHEADER );
  ash.fdwStatus = 0;
  ash.dwUser = 0; 
  ash.pbSrc = (BYTE *)src; 
  ash.cbSrcLength = srcsize;
  ash.pbDst = (BYTE *)dest;
  ash.cbDstLength = osize; 

  h = acmStreamPrepareHeader( priv->acm_stream_handle, &ash, 0 );
  if( h )	xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(ACM) acmStreamPrepareHeader() error %lx", h );

  h = acmStreamConvert( priv->acm_stream_handle, &ash, 0 );
  if( h )	xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(ACM) acmStreamConvert() error %lx", h );

  /* Set output */
  *samples = ash.cbDstLengthUsed / ( priv->format.channels * 2 );
  used = ash.cbSrcLengthUsed;

  h = acmStreamUnprepareHeader( priv->acm_stream_handle, &ash, 0 );
  if( h )	xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(ACM) acmStreamUnprepareHeader() error %lx", h );

  return used;
}

/*
 * Encode data
 */
static int acm_Encode( XMM_PluginCodecAudio *codec, uint8_t *src, uint32_t samples, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecAudioInfo *acm_Info( void *xmm )
{
  return &acm_cai;
}

/*
 * Plugin data
 */
XMM_PluginCodecAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_ACODEC,
				XMM_VERSION_NUM,
				"",
				"ACM",
				"Codec: ACM DLL Support ( Decoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				acm_Open, acm_Close, acm_Control,
				acm_Decode, acm_Encode,	acm_Info };

/*
 * Private code
 */

/*
 * Open WMA
 */
static int WMAudioCodecOpen( XMM_PluginCodecAudio *codec, XMM_AudioFormat *af, int query )
{
  struct priv_t		*priv = codec->sys.priv;
  MMRESULT		h;
  XMM_AcCodecProperties	cp;

  if( xmm_AcCodecFile( codec->sys.xmm, af->formatID, XMM_CFG_CODEC_ACM, &cp ) < XMM_RET_OK )
	return XMM_ERR_NOTSUPPORTED;

  xmm_logging( 3, "ACM! Using DLL: %s\n", cp.file );

  /* Set input data */
  if(( priv->acm_format_source = malloc( sizeof( WAVEFORMATEX ) + af->extraSize )) == NULL )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_ALLOC, "(ACM) Unable to allocate acm_format_source" );
  }

  priv->acm_format_source->wFormatTag = af->formatID;
  priv->acm_format_source->nChannels = af->channels;
  priv->acm_format_source->nSamplesPerSec = af->samprate;
  priv->acm_format_source->nAvgBytesPerSec = af->bitrate / 8;
  priv->acm_format_source->nBlockAlign = af->blockSize;
  priv->acm_format_source->wBitsPerSample = ( af->format & XMM_SOUND_MASK_SIZE );
  priv->acm_format_source->cbSize = af->extraSize;

  memcpy( (char *)priv->acm_format_source + 18, (void *)&af[1], af->extraSize );

  /* Set output data */
  if(( priv->acm_format_dest = malloc( sizeof( WAVEFORMATEX ))) == NULL )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_ALLOC, "(ACM) Unable to allocate acm_format_dest" );
  }

  priv->acm_format_dest->wFormatTag = WAVE_FORMAT_PCM;
  priv->acm_format_dest->nChannels = priv->acm_format_source->nChannels;
  priv->acm_format_dest->nSamplesPerSec = priv->acm_format_source->nSamplesPerSec;
  priv->acm_format_dest->nAvgBytesPerSec = 2 * priv->acm_format_dest->nSamplesPerSec * priv->acm_format_dest->nChannels;
  priv->acm_format_dest->nBlockAlign = 2 * priv->acm_format_source->nChannels;
  priv->acm_format_dest->wBitsPerSample = 16;
  priv->acm_format_dest->cbSize = 0;

#ifdef VERBOSE
  print_waveformatex( priv->acm_format_source );
  print_waveformatex( priv->acm_format_dest );
#endif

  /* Open ACM conversion stream */
  SetCodecDLL( cp.file );
  h = acmStreamOpen(	&priv->acm_stream_handle, NULL,
			priv->acm_format_source,
			priv->acm_format_dest,
			NULL, 0, 0, query ? ACM_STREAMOPENF_QUERY : 0 );

  if( h )	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(ACM) Cannot open decoder ( acmStreamOpen failed, error %i )", h );

  return XMM_RET_OK;
}

/*
 * Close WMA
 */
static int WMAudioCodecClose( XMM_PluginCodecAudio *codec )
{
  struct priv_t		*priv = codec->sys.priv;

  /* Close ACM */
  acmStreamClose( priv->acm_stream_handle, 0 );

  /* Free resources */
  free( priv->acm_format_source );
  free( priv->acm_format_dest );

  return XMM_RET_OK;
}

/*
 * Print WAVEFORMATEX
 */

#ifdef VERBOSE

static void print_waveformatex( WAVEFORMATEX *wave_fmt_head )
{
  xmm_logging( 1, "\twave fmt: FormatTag = %x\n", wave_fmt_head->wFormatTag );
  xmm_logging( 1, "\twave fmt: Channels = %i\n", wave_fmt_head->nChannels );
  xmm_logging( 1, "\twave fmt: Samples/Sec = %li\n",  wave_fmt_head->nSamplesPerSec );
  xmm_logging( 1, "\twave fmt: Avg Bytes/Sec = %li\n", wave_fmt_head->nAvgBytesPerSec );
  xmm_logging( 1, "\twave fmt: BlockAlign = %i\n", wave_fmt_head->nBlockAlign );
  xmm_logging( 1, "\twave fmt: Bits / Sample = %i\n", wave_fmt_head->wBitsPerSample );
  xmm_logging( 1, "\twave fmt: Extra data: %i bytes\n", wave_fmt_head->cbSize );
}

#endif
