/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wvfw.c
 * Codec Plugin for VFW (DLL) codecs
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpgraph.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/error.h>
#include <libxmm/config.h>

#include "wine/windef.h"
#include "wine/driver.h"
#include "wine/vfw.h"

/*
 * Definitions
 */

/*
 * Types
 */

struct priv_t
{
    HIC				hic;			/* Decompressor handle */
    BITMAPINFOHEADER		*src_bm_head;		/* Input */
    BITMAPINFO			*dest_bm_head;		/* Output */
    XMM_AcCodecProperties	cp;
};

/*
 * Codec info
 */

static XMM_CodecVideoInfo	vfw_cvi =
{
    XMM_CODEC_VCF_DECODE,
    "VFW ( W32 DLL )",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    xmmFOURCC( '?','?','?','?' ),	/* FOURCC. */
};

/*
 * Global data
 */

extern XMM_PluginCodecVideo	plugin_info;

/*
 * Prototypes
 */

static int WMVideoCodecOpen( XMM_PluginCodecVideo *codec, XMM_VideoFormat *vf, int query );
static int WMVideoSetFormat( XMM_PluginCodecVideo *codec, uint32_t format );
static int WMVideoQueryFormat( XMM_PluginCodecVideo *codec, uint32_t format );
static int WMVideoCodecClose( XMM_PluginCodecVideo *codec );
static void print_bitmapinfoheader( BITMAPINFOHEADER *bih, char *text );

/*
 * Initialize Plugin
 */
static XMM_PluginCodecVideo *vfw_Open( void *xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_PluginCodecVideo	*pCodec;
  struct priv_t		*priv;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )
  {
	return (void *)xmm_AcCodecFile( xmm, vf->codec, XMM_CFG_CODEC_VFW, NULL );
  }

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(VFW) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Open codec */
  if( WMVideoCodecOpen( pCodec, vf, 0 ) < 0 )
  {
	free( pCodec );
	return NULL;
  }

  return pCodec;
}

/*
 * Free codec
 */
static int vfw_Close( XMM_PluginCodecVideo *codec )
{
  /* Close WMV */
  WMVideoCodecClose( codec );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int vfw_Control( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  int		i;
  uint32_t	aFmt[] =
  {
	XMM_GRAPH_FMT_YUY2, XMM_GRAPH_FMT_YV12,
	XMM_GRAPH_FMT_RGB( 32 ), XMM_GRAPH_FMT_RGB( 24 ),
	XMM_GRAPH_FMT_RGB( 16 ), XMM_GRAPH_FMT_RGB( 15 ),
	0
  };

  switch( cmd )
  {
    case XMM_CTLQUERY_GFORMAT:
	    if( WMVideoQueryFormat( codec, (uint32_t) arg ) >= 0 )	return XMM_CTLRET_TRUE;
	    return XMM_CTLRET_FALSE;

    case XMM_CTLQUERY_YFLIP:
	    *((uint32_t *)arg) = 1;
	    return XMM_CTLRET_ARG;

    case XMM_CTLGET_GFORMAT:
	    for( i = 0; aFmt[i]; i++ )
	    {
		if( WMVideoQueryFormat( codec, aFmt[i] ) == 0 )
		{
		    *((uint32_t *)arg) = aFmt[i];
		    break;
		}
	    }

	    if( aFmt[i] == 0 )	return XMM_RET_NOTSUPPORTED;
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLGET_FORMAT_SIZE:
	    return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLGET_FORMAT_DATA:
	    return XMM_CTLRET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLSET_GFORMAT:
	    if( WMVideoSetFormat( codec, ( uint32_t) arg ) >= 0 )	return XMM_CTLRET_TRUE;
	    return XMM_CTLRET_ERROR;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_CODEC )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No CODEC command */
}

/*
 * Decode data
 */
static int vfw_Decode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  struct priv_t	*priv = codec->sys.priv;
  HRESULT	h;

  priv->src_bm_head->biSizeImage = isize;

  print_bitmapinfoheader( priv->src_bm_head, "-Before ICDecompress" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-Before ICDecompress" );

  h = ICDecompress( priv->hic, ICDECOMPRESS_NOTKEYFRAME, priv->src_bm_head,
				src, (BITMAPINFOHEADER *)priv->dest_bm_head, dest[0] );

  print_bitmapinfoheader( priv->src_bm_head, "+Before ICDecompress" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+Before ICDecompress" );

  if( h != ICERR_OK )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_ALLOC, "(VFW) Cannot decode frame ( ICDecompress, error = %li )", h );
  }

  return isize;
}

/*
 * Encode data
 */
static int vfw_Encode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecVideoInfo *vfw_Info( void *xmm )
{
  return &vfw_cvi;
}

/*
 * Plugin data
 */
XMM_PluginCodecVideo plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_VCODEC,
				XMM_VERSION_NUM,
				"",
				"VFW",
				"Codec: VFW DLL Support ( Decoding )",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				vfw_Open, vfw_Close, vfw_Control,
				vfw_Decode, vfw_Encode,	vfw_Info };

/*
 * Private code
 */

/*
 * Open WMV
 */
static int WMVideoCodecOpen( XMM_PluginCodecVideo *codec, XMM_VideoFormat *vf, int query )
{
  struct priv_t		*priv = codec->sys.priv;
  HRESULT		h;

  if( xmm_AcCodecFile( codec->sys.xmm, vf->codec, XMM_CFG_CODEC_VFW, &priv->cp ) < XMM_RET_OK )
	return XMM_ERR_NOTSUPPORTED;

  xmm_logging( 3, "VFW! Using DLL: %s\n", priv->cp.file );

  /* Set input data */
  if(( priv->src_bm_head = malloc( sizeof( BITMAPINFOHEADER ))) == NULL )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_ALLOC, "(VFW) Unable to allocated acm_format_source" );
  }

  priv->src_bm_head->biSize = 0x28;
  priv->src_bm_head->biWidth = vf->width;
  priv->src_bm_head->biHeight = vf->height;
  priv->src_bm_head->biPlanes = vf->planes;
  priv->src_bm_head->biBitCount = vf->bpp;
  priv->src_bm_head->biCompression = vf->codec;
  priv->src_bm_head->biSizeImage = vf->imgSize;
  priv->src_bm_head->biXPelsPerMeter = 0;
  priv->src_bm_head->biYPelsPerMeter = 0;
  priv->src_bm_head->biClrUsed = 0;
  priv->src_bm_head->biClrImportant = 0;

  /* Set output data, we need BITMAPINFO with 2 additional bmiColors */
  if(( priv->dest_bm_head = malloc( sizeof( BITMAPINFO ) + 2 * sizeof( int ))) == NULL )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_ALLOC, "(VFW) Unable to allocated acm_format_source" );
  }

  priv->dest_bm_head->bmiHeader.biSize = sizeof( BITMAPINFO ) + 2 * sizeof( int );

  /* Open VFW */
  SetCodecDLL( priv->cp.file );
  priv->hic = ICOpen( xmmFOURCC( 'v','i','d','c' ), priv->src_bm_head->biCompression, ICMODE_FASTDECOMPRESS );
  if( priv->hic == 0 )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) Cannot open decoder ( ICOpen failed )" );
  }

  print_bitmapinfoheader( priv->src_bm_head, "-ICDecompressGetFormat" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-ICDecompressGetFormat" );

  h = ICDecompressGetFormat( priv->hic, priv->src_bm_head, priv->dest_bm_head );
  if( h != ICERR_OK )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) Cannot handle format ( ICDecompressGetFormat failed, error = %li )", h );
  }

  print_bitmapinfoheader( priv->src_bm_head, "+ICDecompressGetFormat" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+ICDecompressGetFormat" );

  if( query == 1 )	WMVideoCodecClose( codec );

  return XMM_RET_OK;
}

/*
 * Close WMV
 */
static int WMVideoCodecClose( XMM_PluginCodecVideo *codec )
{
  struct priv_t	*priv = codec->sys.priv;
  HRESULT	h;

  /* Stop decoder */
  h = ICDecompressEnd( priv->hic );
  if( h != ICERR_OK )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) IC error ( ICDecompressEnd, error = %li )", h );
  }

  h = ICClose( priv->hic );
  if( h != ICERR_OK )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) IC error ( ICClose, error = %li )", h );
  }

  free( priv->src_bm_head );
  free( priv->dest_bm_head );

  return XMM_RET_OK;
}

/*
 * Set format
 */
static int WMVideoSetFormat( XMM_PluginCodecVideo *codec, uint32_t format )
{
  struct priv_t	*priv = codec->sys.priv;
  HRESULT	h;

  /* Check if format supported by codec */
  if( WMVideoQueryFormat( codec, format ) < 0 )	return XMM_RET_NOTSUPPORTED;

  /* Set destination bitmap */
  if( format == XMM_GRAPH_FMT_RGB( 32 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 32;
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 24 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 24;
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 16 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = 3; 	/* 3 == BI_BITFIELDS */
	priv->dest_bm_head->bmiColors[0] = 0xF800;
	priv->dest_bm_head->bmiColors[1] = 0x07E0;
	priv->dest_bm_head->bmiColors[2] = 0x001F;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 15 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = 3; 	/* 3 == BI_BITFIELDS */
	priv->dest_bm_head->bmiColors[0] = 0x7C00;
	priv->dest_bm_head->bmiColors[1] = 0x03E0;
	priv->dest_bm_head->bmiColors[2] = 0x001F;
  }
  else if( format == XMM_GRAPH_FMT_YV12 )
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
  }
  else if( format == XMM_GRAPH_FMT_YUY2 )
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
  }
  else return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) Unknown format '%s'.", xmm_FOURCC_string( format ));

  /* Set image size and flipping */
  priv->dest_bm_head->bmiHeader.biSizeImage = priv->src_bm_head->biWidth * priv->src_bm_head->biHeight * 4;
  priv->dest_bm_head->bmiHeader.biHeight = priv->src_bm_head->biHeight;

  print_bitmapinfoheader( priv->src_bm_head, "-After ICDecompressQuery" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-After ICDecompressQuery" );

  /* Query decoder */
  h = ICDecompressQuery( priv->hic, priv->src_bm_head, priv->dest_bm_head );
  if( h != ICERR_OK )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) Cannot Query decompress ( ICDecompressQuery, error = %li )", h );
  }

  print_bitmapinfoheader( priv->src_bm_head, "+After ICDecompressQuery" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+After ICDecompressQuery" );

  print_bitmapinfoheader( priv->src_bm_head, "-After ICDecompressBegin" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-After ICDecompressBegin" );

  /* Start decoder */
  h = ICDecompressBegin( priv->hic, priv->src_bm_head, priv->dest_bm_head );
  if( h != ICERR_OK )
  {
	return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) Cannot handle format ( ICDecompressBegin, error = %li )", h );
  }

  print_bitmapinfoheader( priv->src_bm_head, "+After ICDecompressBegin" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+After ICDecompressBegin" );

  /* mpeg4c32.dll and divxc32.dll does not accept these on query
     ( Thanks to Arpad Gereoffy ) */

  if(( format == XMM_GRAPH_FMT_YV12 ) || ( format == XMM_GRAPH_FMT_YUY2 ))
  {
	priv->dest_bm_head->bmiHeader.biCompression = format;
  }

  xmm_logging( 2, "VFW! Desired format %x [%s]\n", format, xmm_FOURCC_string( format ));
  xmm_logging( 2, "VFW! Decoding to %x [%s] biBitCount = %i\n", priv->dest_bm_head->bmiHeader.biCompression, xmm_FOURCC_string( priv->dest_bm_head->bmiHeader.biCompression ), priv->dest_bm_head->bmiHeader.biBitCount );

  return XMM_RET_OK;
}

/*
 * Query format
 */
static int WMVideoQueryFormat( XMM_PluginCodecVideo *codec, uint32_t format )
{
  struct priv_t	*priv = codec->sys.priv;
  char	*str;

  if( format == XMM_GRAPH_FMT_RGB( 32 ))	str = "RGB32";
  else if( format == XMM_GRAPH_FMT_RGB( 24 ))	str = "RGB24";
  else if( format == XMM_GRAPH_FMT_RGB( 16 ))	str = "RGB16";
  else if( format == XMM_GRAPH_FMT_RGB( 15 ))	str = "RGB15";
  else if( format == XMM_GRAPH_FMT_YV12 )	str = "YV12";
  else if( format == XMM_GRAPH_FMT_YUY2 )	str = "YUY2";
  else return xmm_SetError( codec->sys.xmm, XMM_ERR_UNKNOWN, "(VFW) Unknown format '%s'.", xmm_FOURCC_string( format ));

  if( strstr( priv->cp.ofmt, str ) != NULL )	return 0;

  return XMM_RET_NOTSUPPORTED;
}

/*
 * Print BITMAPINFOHEADER
 */
static void print_bitmapinfoheader( BITMAPINFOHEADER *bih, char *text )
{
#ifdef VERBOSE

  xmm_logging( 1, "--> %s\n", text );
  xmm_logging( 1, "\tbi_head: Size = %li\n", bih->biSize );
  xmm_logging( 1, "\tbi_head: Width = %li\n", bih->biWidth );
  xmm_logging( 1, "\tbi_head: Height = %li\n", bih->biHeight );
  xmm_logging( 1, "\tbi_head: Planes = %i\n", bih->biPlanes );
  xmm_logging( 1, "\tbi_head: BitCount = %i\n", bih->biBitCount );
  xmm_logging( 1, "\tbi_head: Compression = '%s' ( %lx )\n", xmm_FOURCC_string( bih->biCompression ), bih->biCompression );
  xmm_logging( 1, "\tbi_head: SizeImage = %li\n", bih->biSizeImage );
  xmm_logging( 1, "\tbi_head: XPels/m = %li\n", bih->biXPelsPerMeter );
  xmm_logging( 1, "\tbi_head: YPels/m = %li\n", bih->biYPelsPerMeter );
  xmm_logging( 1, "\tbi_head: ClrUsed = %li\n", bih->biClrUsed );
  xmm_logging( 1, "\tbi_head: ClrImportant = %li\n", bih->biClrImportant );
#endif
}
