/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * gc-mmx.c
 * MMX based Graph Conversion
 *
 * TODO:
 *  - Add conversions. This contains only RGB15 --> RGB16 without stretching.
 */

#include <stdlib.h>
#include <stdio.h>

#include <inttypes.h>

#include <libxmm/xmmp.h>
#include <libxmm/xmmctl.h>
#include <libxmm/version.h>
#include <libxmm/lpgraph.h>
#include <libxmm/lpfilterv.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/mmaccel.h>
#include "mm_mmx.h"

/*
 * Types
 */

struct priv_t
{
    int				swidth;		/* Source */
    int				sheight;
    int				dwidth;		/* Dest */
    int				dheight;
    int				wwidth;		/* Output ( screen size ) */
    int				wheight;
    int				gflags;		/* Flags ( e.g. Y-Flip ) */
    int (*Proc)( XMM_PluginFilterVideo *pfv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
    int (*ProcS)( XMM_PluginFilterVideo *pfv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
};

/*
 * Global data
 */

extern XMM_PluginFilterVideo	plugin_info;

/*
 * Prototypes
 */

static int rgb15_rgb16( XMM_PluginFilterVideo *filter, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
static int rgb15_rgb16_stretch( XMM_PluginFilterVideo *filter, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );

/*
 * Init
 */
static XMM_PluginFilterVideo *std_Open( void *_xmm, XMM_ImageFormat *sif, XMM_ImageFormat *dif )
{
  XMM_PluginFilterVideo	*pConv;
  struct priv_t		*priv;
  char			buffer[5];

  /* Check for MMX support */
  if( !( xmmCPU_mmsupport() & XMM_MMACCEL_X86_MMX ))
  {
	xmm_SetError( _xmm, XMM_ERR_UNKNOWN, "(GC-MMX) No MMX support." );
	return NULL;
  }

  /* Allocate plugin data */
  if(( pConv = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( _xmm, XMM_ERR_ALLOC, "(GC-MMX) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pConv[1];
  pConv->sys.priv = (void *) priv;
  pConv->sys.xmm = _xmm;

  /* Save data */
  priv->swidth = sif->width;
  priv->sheight = sif->height;
  priv->gflags = sif->flags;

  xmm_logging( 2, "GC-MMX! Source: %x [%s]\n", sif->codec, xmm_FOURCC_string( sif->codec ));
  xmm_logging( 2, "GC-MMX! Dest: %x [%s]\n", dif->codec, xmm_FOURCC_string( dif->codec ));

  /* Find converter */
  if( sif->codec == xmmFOURCC( 'R','G','B', 15 ))
  {
	if( dif->codec == xmmFOURCC( 'R','G','B', 16 ))
	{
	    priv->Proc = pConv->Proc = rgb15_rgb16;
	    priv->ProcS = rgb15_rgb16_stretch;
	    return pConv;
	}
  }

  free( pConv );

  strcpy( buffer, xmm_FOURCC_string( dif->codec ));
  xmm_SetError( pConv->sys.xmm, XMM_ERR_NOTSUPPORTED, "(GC-MMX) Needed conversion not supported. %x [%s] --> %x [%s]", sif->codec, xmm_FOURCC_string( sif->codec ), dif->codec, buffer );
  return NULL;
}

/*
 * Free converter
 */
static int std_Close( XMM_PluginFilterVideo *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Control function
 */
static int std_Control( XMM_PluginFilterVideo *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;
  XMM_ControlScale	*scale;

  switch( cmd )
  {
    case XMM_CTLSET_SCALE:
	    scale = (XMM_ControlScale *) data;
	    if(( priv->swidth - scale->width ) || ( priv->sheight - scale->height ))	filter->Proc = priv->ProcS;
	    else	filter->Proc = priv->Proc;
	    priv->dwidth = scale->width;
	    priv->dheight = scale->height;
	    priv->wwidth = scale->bwidth;
	    priv->wheight = scale->bheight;
	    return XMM_CTLRET_TRUE;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_VFILTER )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No VFILTER command */
}

/*
 * Plugin data
 */

XMM_PluginFilterVideo		plugin_info = {{
				NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_VFILTER,
				0,
				XMM_VERSION_NUM,
				"",
				"FVC-MMX",
				"MMX conversion filter",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				std_Open, std_Close, std_Control, NULL };

/*
 * Internal code 
 */

/*
 * RGB 15 Input
 */

/*
 * RGB 15 --> RGB 16
 */

/*
 * This is based on:
 * rgb15to16mmx.c ( MPlayer 0.11 )
 * Original ( MMX code ) by Strepto/Astral, ported to gcc & bugfixed by A'rpi
 */

static int rgb15_rgb16( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy )
{
  struct priv_t		*priv = conv->sys.priv;
  static uint64_t	mask_b  = 0x001F001F001F001FLL; /* 00000000 00011111  xxB */
  static uint64_t	mask_rg = 0x7FE07FE07FE07FE0LL; /* 01111111 11100000  RGx */
  int			swidth, dwidth;
  register char		*src, *dest;
  register int		pos;

  src = sdata[0];
  dest = (uint8_t *) ddata[0] + ( dy * priv->wwidth + dx ) * 2;

  swidth = (( stride == NULL ) ? priv->swidth << 1 : stride[0] );
  dwidth = priv->dwidth << 1;
  width <<= 1;

  if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
  {
	dest = (uint8_t *)ddata[0] + (( priv->dheight - dy - 1 ) * priv->wwidth + dx ) * 2;
	dwidth = -dwidth;
  }

  if(( swidth & 0x0007 ) || ( priv->dwidth & 0x0007 ))
  {
	xmm_logging( 1, "GC-MMX! Fatal ERROR: swidth or dwidth not multiple of 8\n" );
	return XMM_RET_ERROR;
  }

  movq_m2r( mask_b,  mm4 );
  movq_m2r( mask_rg, mm5 );

  for( ; height > 0; height--, src += swidth, dest += dwidth )
  {
	for( pos = 0; pos < width; pos += 16 )
	{
		movq_m2r( *(src + pos), mm0 );
		movq_r2r( mm0, mm1 );

		movq_m2r( *(src + 8 + pos), mm2 );
		movq_r2r( mm2, mm3 );
    
		pand_r2r( mm4, mm0 );
		pand_r2r( mm5, mm1 );
    
		psllq_i2r( 1, mm1 );
		pand_r2r( mm4, mm2 );

		pand_r2r( mm5, mm3 );
		por_r2r( mm1, mm0 );

		psllq_i2r( 1, mm3 );
		movq_r2m( mm0, *(dest + pos ));

		por_r2r( mm3, mm2 );
		movq_r2m( mm2, *(dest + 8 + pos ));
	}
  }

  emms();
  return XMM_RET_OK;
}

/*
 * RGB 15 --> RGB 16 ( Stretch )
 */

/*
 * TODO: MMX accelaration
 * Problem: A line may not be n * 16, n >= 1
 */

static int rgb15_rgb16_stretch( XMM_PluginFilterVideo *conv, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy )
{
  struct priv_t *priv = conv->sys.priv;
  int i, j, dstride_add, sstride;
  uint16_t *dest, *src;
  register uint16_t tmp;

  src = (uint16_t *) sdata[0];
  dest = (uint16_t *) ddata[0] + (( dy * priv->dheight / priv->sheight ) * priv->wwidth + ( dx * priv->dwidth / priv->swidth ));

  sstride = (( stride == NULL ) ? priv->swidth : stride[0] >> 1 );
  dstride_add = ( priv->wwidth - ( width * priv->dwidth / priv->swidth ));

  height = height * priv->dheight / priv->sheight;
  width = width * priv->dwidth / priv->swidth;

  if( priv->gflags & XMM_GRAPH_FLAG_YFLIP )
  {
	dest = (uint16_t *)ddata[0] + (( priv->dheight - ( dy * priv->dheight / priv->sheight ) - 1 ) * priv->wwidth + ( dx * priv->dwidth / priv->swidth ));
	dstride_add = - ( priv->wwidth + width );
  }

  for( i = 0; i < height; i++, dest += dstride_add )
  {
	for( j = 0; j < width; j++ )
	{
		tmp = *( src + (i * priv->sheight / priv->dheight) * sstride + (j * priv->swidth / priv->dwidth));
		*dest++ = ( tmp & 0x001F ) | (( tmp & 0x7FE0 ) << 1 );
	}
  }

  return XMM_RET_OK;
}

