/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * x11_dga2.c
 * XFree86 - Direct Graphic Access ( DGA ) 2.0
 */

#include <unistd.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>

#include <libxmm/xmmp.h>
#include <libxmm/util/utils.h>

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#define	DEBUG
#endif

/* Change 'undef' in 'define' to even more information */
#ifndef VERBOSE
#define	VERBOSE
#endif

/*
 * Definitions
 */

#define	DGA_MINMAJOR	2
#define	DGA_MINMINOR	0

/*
 * Global data
 */

static XDGAMode			*aVidMode = NULL;
static int			VidModes = 0;

/*
 * Prototypes
 */

#ifdef DEBUG
static void dga_PrintMode( XDGAMode *mode );
#endif

/*
 * ( initialize and ) start X11/XF86-DGA stuff
 */

char *xf86dga2_start( Display *display, int screen, int bpp, int *xsize, int *ysize, int *pwidth )
{
  int		MajorVersion, MinorVersion;
  int		EventBase, ErrorBase, n, i;
  int		cmode, cxsize, cysize;
  char		*MemBase = NULL;
  XDGADevice	*dga2_device;

  if( aVidMode )
  {
	xmm_logging( 1, "X11(DGA)! Already in fullscreen mode.\n" );
	return NULL;
  }

  /* Check for XFree86-DGA 2.0 */
  if( XDGAQueryExtension( display, &EventBase, &ErrorBase ) == 0 )
  {
	xmm_logging( 1, "X11(DGA2)! Cannot query DGA extension.\n" );
	return NULL;
  }

  /* Get XFree86-DGA 2.0 version */
  if( XDGAQueryVersion( display, &MajorVersion, &MinorVersion ) == 0 )
  {
	xmm_logging( 1, "X11(DGA2)! Cannot query DGA version.\n" );
	return NULL;
  }

  if( MajorVersion < DGA_MINMAJOR ||
	(( MajorVersion == DGA_MINMAJOR ) && ( MinorVersion < DGA_MINMINOR ))) 
  {
	xmm_logging( 1, "X11(DGA2)! XFree86-DGA version %i.%i found.\n",
						MajorVersion, MinorVersion );
	xmm_logging( 1, "X11(DGA2)! Minimum requiered XFree86-DGA version: %i.%i\n",
						DGA_MINMAJOR, DGA_MINMINOR );
	return NULL;
  }

  xmm_logging( 1, "X11(DGA2)! XFree86-DGA extension version %i.%i found.\n",
						MajorVersion, MinorVersion );

  /* Try to open frambuffer */
  if( XDGAOpenFramebuffer( display, screen ) == 0 )
  {
	xmm_logging( 1, "X11(DGA2)! Unable to open Frambuffer.\n" );
	return NULL;
  }

  XDGACloseFramebuffer( display, screen );

  /* Query modes */
  aVidMode = XDGAQueryModes( display, screen, &VidModes );

#ifdef VERBOSE
  for( i = 0; i < VidModes; i++ )	dga_PrintMode( &aVidMode[i] );
#endif

  /* Find mode */

  cmode = VidModes;
  cxsize = 0x7FFF;	/* Biggest positive value for signed short */
  cysize = 0x7FFF;

  for( n = 0; n < VidModes; n++ )
  {
	if( aVidMode[n].bitsPerPixel != bpp )	continue;

	if(( aVidMode[n].viewportWidth > *xsize ) && ( aVidMode[n].viewportHeight > *ysize ))
	{
		if(( aVidMode[n].viewportWidth < cxsize ) && ( aVidMode[n].viewportHeight < cysize ))
		{
			cxsize = aVidMode[n].viewportWidth;
			cysize = aVidMode[n].viewportHeight;
			cmode = n;
		}
	}
  }

  if( cmode == VidModes )
  {
	xmm_logging( 1, "X11(DGA2)! No matching mode found ( %ix%i ).\n", *xsize, *ysize );
	return NULL;
  }

  xmm_logging( 1, "X11(DGA2)! VidMode = %i ( %ix%i )\n",
						    aVidMode[cmode].num,
						    aVidMode[cmode].viewportWidth,
						    aVidMode[cmode].viewportHeight );

  *xsize = aVidMode[cmode].viewportWidth;
  *ysize = aVidMode[cmode].viewportHeight;

  cmode = aVidMode[cmode].num;

  /*
   * Initialize DGA
   */

  /* Open frame buffer */
  if( XDGAOpenFramebuffer( display, screen ) == 0 )
  {
	xmm_logging( 1, "X11(DGA2)! Unable to open Frambuffer.\n" );
	return NULL;
  }

  /* Initialize DGA */
  dga2_device = XDGASetMode( display, screen, cmode );

  /* Get mapped frame buffer memory */
  MemBase = dga2_device->data;
  *pwidth = dga2_device->mode.bytesPerScanline;
  XFree( dga2_device );

  /* Set Viewport */
  XDGASetViewport( display, screen, 0, 0, XDGAFlipRetrace );

  return MemBase;

#if 0
  /* Create Window */
  window = DefaultRootWindow( display );

  /* Map window */
  XMapWindow( display, window );

  /* Event handling */
  XGrabKeyboard( display, window, True, GrabModeAsync, GrabModeAsync, CurrentTime );
#endif
}

/*
 * Restore X11 settings and free memory
 */

void xf86dga2_exit( Display *display, int screen )
{
  /* Restore mode */
  XDGASetMode( display, screen, 0 );

  /* unmap framebuffer memory */
  XDGACloseFramebuffer( display, screen );

  /* Free video modes */
  XFree( aVidMode );
  aVidMode = NULL;
}

/*
 * Private code
 */

#ifdef DEBUG

static void dga_PrintMode( XDGAMode *mode )
{
  xmm_logging( 1, "X11(DGA2)! Info: Mode %i: %s ( %dx%d ) at %d(%d bpp) bpp ( refresh %f, pitch %d )\n",
	mode->num, mode->name, mode->viewportWidth, mode->viewportHeight,
	mode->depth, mode->bitsPerPixel,
	mode->verticalRefresh, mode->bytesPerScanline );

  xmm_logging( 1, "X11(DGA2)! Info: RGB: 0x%8.8x 0x%8.8x 0x%8.8x ( %d - %s )\n",
		mode->redMask, mode->greenMask, mode->blueMask,
		mode->visualClass,
		mode->visualClass == TrueColor ? "truecolor" :
		mode->visualClass == DirectColor ? "directcolor" :
		mode->visualClass == PseudoColor ? "pseudocolor" : "unknown" );

  xmm_logging( 1, "X11(DGA2)! Flags: " );

  if( mode->flags & XDGAConcurrentAccess )
	printf( " XDGAConcurrentAccess" );

  if( mode->flags & XDGASolidFillRect )
	printf( " XDGASolidFillRect" );

  if( mode->flags & XDGABlitRect )
	printf( " XDGABlitRect" );

  if( mode->flags & XDGABlitTransRect )
	printf( " XDGABlitTransRect" );

  if( mode->flags & XDGAPixmap )
	printf( " XDGAPixmap" );

  if( mode->flags & XDGAInterlaced )
	printf( " XDGAInterlaced" );

  if( mode->flags & XDGADoublescan )
	printf( " XDGADoublescan" );

  if( mode->viewportFlags & XDGAFlipRetrace )
	printf( " XDGAFlipRetrace" );

  if( mode->viewportFlags & XDGAFlipImmediate )
	printf( " XDGAFlipImmediate" );

  printf("\n");
}

#endif
