/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * plugin.c
 * AVI input plugin
 *
 * TODO:
 *  - current Sample in AudioInfo() is set to 0, as there is no way to get it
 *  - Only first video stream used. The AVI reader supports multiple streams
 *  - Control interface for first stream of each type usable
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include "reader.h"
#include "avifmt.h"

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Types
 */

/*
 * Private plugin data
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_FileInfo		*fi;

    struct
    {
	XMM_PluginCodecVideo	*pCodec;
	XMM_VideoFormat		*vf;

    }				vStream[MAX_STREAMS];

    struct
    {
	XMM_PluginCodecAudio	*pCodec;
	XMM_AudioInfo		ai;
	XMM_AudioFormat		*af;

    }				aStream[MAX_STREAMS];

    Reader_t			*reader;
    char 			*filename;
};

/*
 * Initialize
 */
static XMM_PluginInput *avi_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(AVI) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int avi_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr;
  int			i;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(AVI) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Open file */
  if(( priv->reader = ReaderOpen( input->sys.xmm, priv->pIO )) == NULL )
	return XMM_RET_ERROR;

  /* Get file info */
  priv->fi = priv->reader->FileInfo( priv->reader );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Initialize video */
  for( i = 0; i < 1 /* priv->fi->vstreams */; i++ )
  {
	priv->vStream[i].vf = priv->reader->VideoFormat( priv->reader, i );
	if( priv->vStream[i].vf == NULL )	return XMM_RET_ERROR;

	/* Initialize video codec */
	priv->vStream[i].pCodec = (XMM_PluginCodecVideo *)xmm_CodecVideoOpen( input->sys.xmm, XMM_CODEC_MODE_DECODE, priv->vStream[i].vf );
	if( priv->vStream[i].pCodec == NULL )	return XMM_RET_ERROR;
  }

  /* Initialize audio */
  for( i = 0; i < priv->fi->astreams; i++ )
  {
	priv->aStream[i].af = priv->reader->AudioFormat( priv->reader, i );
	if( priv->aStream[i].af == NULL )	return XMM_RET_ERROR;

	/* Initialize audio codec */
	priv->aStream[i].pCodec = (XMM_PluginCodecAudio *)xmm_CodecAudioOpen( input->sys.xmm, XMM_CODEC_MODE_DECODE, priv->aStream[i].af );
	if( priv->aStream[i].pCodec == NULL )	return XMM_RET_ERROR;

	/* Query codec output format */
	memcpy( &priv->aStream[i].ai, &priv->fi->ai[i], sizeof( XMM_AudioInfo ));
	priv->aStream[i].pCodec->Control( priv->aStream[i].pCodec, XMM_CTLGET_SFORMAT, &priv->aStream[i].ai.format );
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int avi_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;
  int		i;

  /* Close file and free resources */
  free( priv->filename );
  priv->pIO->Close( priv->pIO );

  /* Free video resources */
  for( i = 0; i < 1 /* priv->fi.vstreams */; i++ )
  {
	if( priv->vStream[i].vf )	free( priv->vStream[i].vf );

	if( priv->vStream[i].pCodec )
		priv->vStream[i].pCodec->Close( priv->vStream[i].pCodec );
  }

  /* Free audio resources */
  for( i = 0; i < priv->fi->astreams; i++ )
  {
	if( priv->aStream[i].af )	free( priv->aStream[i].af );

	if( priv->aStream[i].pCodec )
		priv->aStream[i].pCodec->Close( priv->aStream[i].pCodec );
  }

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int avi_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_SoundFormat	*format;

  switch( cmd )
  {
	/* Query graph format */
	case XMM_CTLQUERY_GFORMAT:
		return priv->vStream[0].pCodec->Control( priv->vStream[0].pCodec, XMM_CTLQUERY_GFORMAT, (void *)param );

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)data;
		if(( format->format == priv->aStream[0].ai.format.format ) &&
		    ( format->samprate = priv->aStream[0].ai.format.samprate ) &&
		    ( format->channels == priv->aStream[0].ai.format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	/* Query flip state */
	case XMM_CTLQUERY_YFLIP:
		return priv->vStream[0].pCodec->Control( priv->vStream[0].pCodec, XMM_CTLQUERY_YFLIP, data );

	/* Get graph format */
	case XMM_CTLGET_GFORMAT:
		return priv->vStream[0].pCodec->Control( priv->vStream[0].pCodec, XMM_CTLGET_GFORMAT, data );

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)data;
		format->channels = priv->aStream[0].ai.format.channels;
		format->samprate = priv->aStream[0].ai.format.samprate;
		format->format = priv->aStream[0].ai.format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_AUDIO |
					XMM_INPUT_CF_VIDEO;
		return XMM_CTLRET_ARG;

	/* Set graph format */
	case XMM_CTLSET_GFORMAT:
		return priv->vStream[0].pCodec->Control( priv->vStream[0].pCodec, XMM_CTLSET_GFORMAT, (void *)param );

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_CTLRET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int avi_Seek( XMM_PluginInput *input, double seek )
{
  struct priv_t	*priv = input->sys.priv;
  Reader_t	*reader = priv->reader;
  uint32_t	frame, samples;
  int		i;

  /* Calculate frame */
  frame = (uint32_t) ( seek * reader->TotalFrames( reader ));

  /* Set video stream position */
  frame = reader->VideoSeek( reader, 0, frame, 1 );

  for( i = 0; i < priv->fi->astreams; i++ )
  {
	/* Calculate audio stream position */
	samples = (uint32_t)(((double)frame / priv->fi->vi[0].framerate ) * priv->aStream[i].ai.format.samprate );

	input->AudioSeek( input, i, samples );
  }

  return XMM_RET_OK;
}

/*
 * Get Information
 */
static double avi_GetInfo( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	strncpy( ci->name, priv->filename, 64 );
	ci->name[63] = '\0';
	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = priv->pIO->Size( priv->pIO );
	ci->playtime = (double)priv->reader->TotalTime( priv->reader );
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )		*seekval = priv->reader->CurrentTime( priv->reader ) / priv->reader->TotalTime( priv->reader );

  return priv->reader->CurrentTime( priv->reader );
}

/*
 * Get audio stream number
 */
static int avi_AudioStreams( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  return priv->fi->astreams;
}

/*
 * Get audio stream information
 */
static int avi_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->fi->ai[stream], sizeof( XMM_AudioInfo ));
  if( cSample )	*cSample = 0;

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int avi_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	ssize, decoded, ret, bps;
  char		*src;

  /* Calculate bytes / sample */
  bps = priv->aStream[stream].ai.format.channels * ( priv->aStream[stream].ai.format.format & XMM_SOUND_MASK_SIZE ) / 8;

  /* Get needed size of source data */
  decoded = ssize = samples * bps;
  priv->aStream[stream].pCodec->Control( priv->aStream[stream].pCodec, XMM_CTLGET_DATA_SSIZE, &ssize );

  /* Read data */
  ret = priv->reader->AudioRead( priv->reader, 0, &src, &ssize );
  if( ret != XMM_RET_OK )	return ret;

  /* Decode data */
  decoded = samples;
  priv->aStream[stream].pCodec->Decode( priv->aStream[stream].pCodec, src, ssize, buffer, &decoded, NULL );

  return decoded;
}

/*
 * Seek to position in audio stream
 */
static int avi_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	pos, bps;

  /* Seek notify */
  priv->aStream[stream].pCodec->Control( priv->aStream[stream].pCodec, XMM_CTLSET_SEEK, 0 );

  /* Calculate bytes / sample */
  bps = priv->aStream[stream].ai.format.channels * ( priv->aStream[stream].ai.format.format & XMM_SOUND_MASK_SIZE ) / 8;

  /* Get needed size of source data */
  pos = sample * bps;
  priv->aStream[stream].pCodec->Control( priv->aStream[stream].pCodec, XMM_CTLGET_DATA_SSIZE, &pos );

  return priv->reader->AudioSeek( priv->reader, stream, pos );
}

/*
 * Get video stream number
 */
static int avi_VideoStreams( XMM_PluginInput *input )
{
#if 0
  struct priv_t	*priv = input->sys.priv;

  return priv->fi->vstreams;
#else
  return 1;
#endif
}

/*
 * Get video stream information
 */
static int avi_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  struct priv_t	*priv = input->sys.priv;

  if( vi )	memcpy( vi, &priv->fi->vi[stream], sizeof( XMM_VideoInfo ));
  if( cFrame )	*cFrame = priv->reader->CurrentFrame( priv->reader, stream );

  return XMM_RET_OK;
}

/*
 * Read video data
 */
static int avi_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	ssize = 0, dsize, ret;
  char		*src;

  if( buffer )
  {
	ret = priv->reader->VideoRead( priv->reader, 0, &src, &ssize );
	if( ret != XMM_RET_OK )	return ret;

	dsize = priv->vStream[stream].vf->imgSize;
	priv->vStream[stream].pCodec->Decode( priv->vStream[stream].pCodec, src, ssize, buffer, &dsize, 0 );
  }

  return XMM_RET_OK;
}

/*
 * Video PTS
 */
static int avi_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  struct priv_t	*priv = input->sys.priv;

  /* Update timestamps */
  if( videoPTS )
  {
	*videoPTS = priv->reader->CurrentFrame( priv->reader, stream ) * 1000 /
					priv->fi->vi[stream].framerate; /* ms */

  }

  return XMM_RET_OK;
}

/*
 * Seek to position in video stream
 */
static int avi_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  struct priv_t	*priv = input->sys.priv;

  return priv->reader->VideoSeek( priv->reader, stream, frame, 0 );
}

/*
 * Check if AVI file
 */
static int avi_CheckFile( void *xmm, char *filename )
{
  uint32_t		dwSize, dwName, dwTemp;
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  pIO->Read( pIO, &dwName, 4, 1 );
  pIO->Read( pIO, &dwSize, 4, 1 );
  pIO->Read( pIO, &dwTemp, 4, 1 );

  pIO->Close( pIO );

  if(( dwName == RIFF_ID_RIFF ) && ( dwTemp == RIFF_ID_AVI ))	return 1;
  else	return 0;
}

/*
 * Get file info
 */
static int avi_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"AVI",
				"Input: AVI",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				avi_Init, avi_Open, avi_Close,
				avi_Control, avi_Seek, avi_GetInfo, 
				avi_AudioStreams, avi_AudioInfo,
				avi_AudioRead, avi_AudioSeek,
				avi_VideoStreams, avi_VideoInfo,
				avi_VideoRead, avi_VideoPTS, avi_VideoSeek,
				avi_CheckFile, avi_FileInfo };
