/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * reader_avi.c
 * AVI file reader
 *
 * TODO:
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <libxmm/xmmp.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include "reader.h"
#include "avifmt.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif

/* Change 'undef' in 'define' to force idx1 ckecking, will check streamID and chunk size  */
#ifndef	AVI_CHECK_IDX1_ENTRIES
#undef	AVI_CHECK_IDX1_ENTRIES
#endif

#define	min( a, b )	((a) < (b) ? (a) : (b))
#define	SwapWORD( a )	((( a & 0xFF ) << 8 ) | (( a >> 8 ) & 0xFF ))

/*
 * Types
 */

struct reader_priv_t
{
    void			*xmm;
    XMM_FileInfo		fi;
    XMM_PluginIO		*pIO;

    /* AVI Header */
    avi_header_t		avi_head;

    /* Misc stuff */
    int				seekable;
    uint32_t			data_offset;
    uint32_t			data_size;

    /* Stream stuff */
    int				nStreams;
    int				asIDX[MAX_STREAMS];
    int				vsIDX[MAX_STREAMS];

    /* Video / Audio streams */
    struct stream_t
    {
	avi_stream_header_t	avi_stream_header;	/* stream header */
	char			*asf_extra;		/* extra data */

	union
	{
	    WAVEFORMATEX	wfhead;
	    BITMAPINFOHEADER	bmhead;
	} asf;

	/* Index stuff */
	avi_index_entry_t	*idx1_chunk;		/* idx1 entries */
	uint32_t		*idx1_total;		/* audio bytes, including this chunk */
	uint32_t		idx1_entries;		/* number of idx1 entries */
	uint32_t		position;		/* Current position */

	/* Misc */
	uint32_t		bposition;		/* (byte) position in current chunk */
	uint32_t		total_size;

	/* buffering */
	XMM_BufferQueue		bq;
	char			*rbuffer;
	int			rbuffersize;

    } 				aStream[MAX_STREAMS];

#if 0
//    uint32_t			cSample;
//    int				bps, bpp;
//    unsigned char		*avi_fmt_extra;
#endif

};

/*
 * Prototypes
 */

static int	Read_avih( XMM_PluginIO *pIO, avi_header_t *avi_head, uint32_t size );
static int	Read_strh( XMM_PluginIO *pIO, avi_stream_header_t *avi_stream_head, uint32_t size );
static int	Read_strf_vids( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *avi_stream_head, struct reader_priv_t *priv );
static int	Read_strf_auds( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *avi_stream_head, struct reader_priv_t *priv );
static int	Read_strd( XMM_PluginIO *pIO, unsigned char *data, uint32_t size );
static int	Read_idx1( XMM_PluginIO *pIO, uint32_t size, struct reader_priv_t *priv );

static uint8_t	READ8( XMM_PluginIO *pIO );
static uint16_t	READ16( XMM_PluginIO *pIO );
static uint32_t	READ32( XMM_PluginIO *pIO );
static uint32_t	READn( XMM_PluginIO *pIO, void *data, uint32_t size );
static int	SEEK( XMM_PluginIO *pIO, long offset, int seekable );
static int	buffer_realloc( char **buffer, int *oldsize, int newsize );

/*
 * Reader API prototypes
 */

static void		Close( Reader_t *reader );
static XMM_FileInfo	*FileInfo( Reader_t *reader );

static XMM_VideoFormat	*VideoFormat( Reader_t *reader, int stream );
static uint32_t		VideoRead_raw( Reader_t *reader, int stream, char **src, uint32_t *ssize );
static uint32_t		VideoRead( Reader_t *reader, int stream, char **src, uint32_t *ssize );
static uint32_t		VideoSeek_nop( Reader_t *reader, int stream, uint32_t frame, int keyframe );
static uint32_t		VideoSeek( Reader_t *reader, int stream, uint32_t frame, int keyframe );

static XMM_AudioFormat	*AudioFormat( Reader_t *reader, int stream );
static uint32_t		AudioRead_raw( Reader_t *reader, int stream, char **src, uint32_t *ssize );
static uint32_t		AudioRead( Reader_t *reader, int stream, char **src, uint32_t *ssize );
static uint32_t		AudioSeek_nop( Reader_t *reader, int stream, uint32_t bytes );
static uint32_t		AudioSeek( Reader_t *reader, int stream, uint32_t bytes );

static uint32_t		CurrentFrame( Reader_t *reader, int stream );
static uint32_t		TotalFrames( Reader_t *reader );
static double		CurrentTime( Reader_t *reader );
static double		TotalTime( Reader_t *reader );

/*
 * Global data
 */

static Reader_t	reader_info = { NULL, Close, FileInfo,
				VideoFormat, VideoRead_raw, VideoSeek_nop,
				AudioFormat, AudioRead_raw, AudioSeek_nop,
				CurrentFrame, TotalFrames,
				CurrentTime, TotalTime };

/*
 * Initialize Reader / Open AVI file
 */

Reader_t *ReaderOpen_avi( void *xmm, XMM_PluginIO *pIO )
{
  Reader_t		*reader;
  struct reader_priv_t	*priv;
  struct stream_t	*ps;
  uint32_t		dwSize, dwName, dwTemp, last_LIST_size = 0, stream;
  int			loop = 1, i, j, idx1_build = 1, idx1_found = 0, total;
  avi_stream_header_t	avi_stream_head;
  unsigned char		avi_strd_head[100];
  char			*endptr;
  avi_index_entry_t	*avi_index_entry = NULL;
  uint32_t		avi_index_entries = 0, data_done = 0;
  uint32_t		db = xmmFOURCC( 'd','b', 0, 0 );
  uint32_t		dc = xmmFOURCC( 'd','c', 0, 0 );
  uint32_t		wb = xmmFOURCC( 'w','b', 0, 0 );
  
  /* Allocate / Duplicate reader */
  if(( reader = xmm_memdup_x( &reader_info, sizeof( Reader_t ), sizeof( struct reader_priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(AVI) Unable to duplicate reader_info" );
	return NULL;
  }

  /* Initialize data */
  reader->priv = priv = (struct reader_priv_t *) &reader[1];
  priv->pIO = pIO;
  priv->xmm = xmm;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
  xmm_logging( 2, "AVI! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );

  /* Parse file */
  while( loop )
  {
	dwName = READ32( priv->pIO );
	if( priv->pIO->Eof( priv->pIO ))	break;

#ifdef VERBPOSE
	xmm_logging( 2, "AVI! Chunk ID = '%s' ( %lx ) at %x\n", xmm_FOURCC_string( dwName ), dwName, priv->pIO->Tell( priv->pIO ) - 4 );
#endif

	/*
	 * Handle list chunks
	 */
	switch( dwName )
	{
		case RIFF_ID_hdrl:
		case RIFF_ID_strl:
		case RIFF_ID_odml:
			dwName = 0;
			break;

		case RIFF_ID_movi:
			priv->data_offset = priv->pIO->Tell( priv->pIO );
			priv->data_size = last_LIST_size;
			xmm_logging( 3, "AVI! data_offset = %li ( %lx )\n", priv->data_offset, priv->data_offset );

			if( priv->seekable == 0 )	loop = 0;
			else	SEEK( priv->pIO, last_LIST_size - 4, priv->seekable );

			dwName = 0;
			break;

		/* Non-list chunk: read size */
		default:
			dwSize = READ32( priv->pIO );
#ifdef VERBOSE
			xmm_logging( 2, "AVI! Chunk Size = %li ( %lx )\n", dwSize, dwSize );
#endif
			break;
	}

	if( dwName == 0 )	continue;

	switch( dwName )
	{
		case RIFF_ID_RIFF:
			dwTemp = READ32( priv->pIO );
			if( dwTemp != RIFF_ID_AVI )
			{
			    priv->pIO->Close( priv->pIO );
			    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(AVI) No AVI RIFF form ( '%s' )\n", xmm_FOURCC_string( dwTemp ));
			    return NULL;
			}
			xmm_logging( 2, "AVI! RIFF Type = '%s'\n", xmm_FOURCC_string( dwTemp ));
			break;

		case RIFF_ID_LIST:
			last_LIST_size = dwSize;
			break;

		/* Header chunks */
		case RIFF_ID_avih:
			Read_avih( priv->pIO, &priv->avi_head, dwSize );
			break;

		case RIFF_ID_strh:
			Read_strh( priv->pIO, &avi_stream_head, dwSize );
			break;

		case RIFF_ID_strf:
			switch( avi_stream_head.fccType )
			{
			    case RIFF_STREAM_VIDEO:
				    Read_strf_vids( priv->pIO, dwSize, &avi_stream_head, priv );
				    break;

			    case RIFF_STREAM_AUDIO:
				    Read_strf_auds( priv->pIO, dwSize, &avi_stream_head, priv );
				    break;

			    default:
				priv->pIO->Close( priv->pIO );
				xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(AVI) Unsupported stream type '%s'", xmm_FOURCC_string( avi_stream_head.fccType ));
				return NULL;
			}
			break;

		case RIFF_ID_strd:
			Read_strd( priv->pIO, avi_strd_head, dwSize );
			break;

		/* 'movi' chunk index */
		case RIFF_ID_idx1:
			Read_idx1( priv->pIO, dwSize, priv );
			idx1_found = 1;
			break;

		/* Chunks to ignore */
		case RIFF_ID_JUNK:
			xmm_logging( 2, "AVI! Ignoring chunk '%s' (%lx) [%li bytes]\n", xmm_FOURCC_string(dwName), dwName, dwSize );
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;

		/* X-tended AVI */
		case RIFF_ID_dmlh:
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;

		/* Unknown chunks */
		default:
			xmm_logging( 2, "AVI! Ignoring chunk '%s' (%lx) [%li bytes]\n", xmm_FOURCC_string(dwName), dwName, dwSize );
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;
	}
  }

  /* Set default reader proc */
  reader->VideoRead = VideoRead_raw;
  reader->VideoSeek = VideoSeek_nop;
  reader->AudioRead = AudioRead_raw;
  reader->AudioSeek = AudioSeek_nop;

  /* Seek to the beginning of data */
  if( priv->seekable )
  {
	/* Seek to data start */
	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

	/* Use idx1 chunk for reading */
	if( idx1_found == 1 )
	{
	    reader->VideoRead = VideoRead;
	    reader->VideoSeek = VideoSeek;
	    reader->AudioRead = AudioRead;
	    reader->AudioSeek = AudioSeek;
	}

	/* Rebuild index */
	if(( idx1_found == 0 ) && idx1_build )
	{
	    xmm_logging( 1, "AVI! Reconstructing IDX1 chunk" );

	    /* Rebuild idx1 chunk */
	    while( 1 )
	    {
		dwName = READ32( priv->pIO );
		dwSize = READ32( priv->pIO );

		if( priv->pIO->Eof( priv->pIO ))
		{
		    xmm_logging( 1, "AVI! End of File. Corrupted file ?\n" );
		    break;
		}

		/**/
		if((( dwName & 0xFFFF ) == db ) || (( dwName & 0xFFFF ) == dc )
						|| (( dwName & 0xFFFF ) == wb ))
		{
		    avi_index_entry[avi_index_entries].dwChunkID = dwName;
		    avi_index_entry[avi_index_entries].dwFlags = 0;
		    avi_index_entry[avi_index_entries].dwChunkOffset = priv->pIO->Tell( priv->pIO ) - 8 - priv->data_offset;
		    avi_index_entry[avi_index_entries].dwChunkLength = dwSize;
#ifdef DEBUG
		    fprintf( stderr, "%6i %s at %x ( size = %i )\n",
			avi_index_entries,
			xmm_FOURCC_string( avi_index_entry[avi_index_entries].dwChunkID ),
			avi_index_entry[avi_index_entries].dwChunkOffset,
			avi_index_entry[avi_index_entries].dwChunkLength );
#endif
		    avi_index_entries++;
		}
		else	break;

		/* AVI is WORD aligned, NOTE: paddding byte must NOT be used */
		if( dwSize & 1 )	dwSize++;
		/* Skip data */
		SEEK( priv->pIO, dwSize, priv->seekable );

		/* Output state */
		data_done += dwSize;
		if( data_done >= ( priv->data_size / 10 ))
		{
		    xmm_logging( 1, "." );
		    data_done -= ( priv->data_size / 10 );
		}
	    }
	    xmm_logging( 1, "\n" );

	    /* Allocate memory for each stream */
	    for( stream = 0; stream < priv->avi_head.dwStreams; stream++ )
	    {
		i = min( priv->aStream[stream].avi_stream_header.dwLength, avi_index_entries );
		if(( priv->aStream[stream].idx1_chunk = malloc( i * sizeof( avi_index_entry_t ))) == NULL )
		{
		    xmm_logging( 1, "AVI! Error allocating memory..\n" );
		    return NULL; 
		}
	    }

	    /* Split idx1 chunk into a idx1 table for each stream */
	    for( i = 0; i < avi_index_entries; i++ )
	    {
		stream = strtol( xmm_FOURCC_string( avi_index_entry[i].dwChunkID & 0xFFFF ), &endptr, 16 );

		memcpy( &priv->aStream[stream].idx1_chunk[priv->aStream[stream].idx1_entries], &avi_index_entry[i], sizeof( avi_index_entry_t ));
		priv->aStream[stream].idx1_entries++;
	    }

	    /* Seek to start of 'movi' chunk */
	    priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

	    /* */
	    reader->VideoRead = VideoRead;
	    reader->VideoSeek = VideoSeek;
	    reader->AudioRead = AudioRead;
	    reader->AudioSeek = AudioSeek;
	}
  }

  if( reader->VideoRead == VideoRead_raw )
	xmm_logging( 2, "AVI! Using RAW read mode ( No IDX1 Chunk / Not Seekable ? )\n" );
  else	xmm_logging( 2, "AVI! Using IDX1 read mode\n" );

  /*
   * Initialize XMM_FileInfo struct / Build audio total bytes table
   */

  for( i = 0; i < priv->nStreams; i++ )
  {
	ps = &priv->aStream[i];

	/* Initialize data */
	ps->rbuffersize = -1;
	ps->rbuffer = NULL;

	/* Initialize buffer queue */
	xmmBQ_Init( &ps->bq );

	/* Initialize stream specific data and audio/video index table */
	switch( ps->avi_stream_header.fccType )
	{
	    case RIFF_STREAM_VIDEO:
		    priv->fi.vi[priv->fi.vstreams].width = priv->avi_head.dwWidth;
		    priv->fi.vi[priv->fi.vstreams].height = priv->avi_head.dwHeight;
		    priv->fi.vi[priv->fi.vstreams].tSize = priv->avi_head.dwLength;
		    priv->fi.vi[priv->fi.vstreams].tFrames = (double)priv->avi_head.dwTotalFrames;
		    priv->fi.vi[priv->fi.vstreams].framerate = (double)1000000 / priv->avi_head.dwMicroSecPerFrame;

		    priv->vsIDX[priv->fi.vstreams++] = i;
		    break;

	    case RIFF_STREAM_AUDIO:
		    priv->fi.ai[priv->fi.astreams].format.samprate = ps->asf.wfhead.nSamplesPerSec;
		    priv->fi.ai[priv->fi.astreams].format.channels = ps->asf.wfhead.nChannels;
		    priv->fi.ai[priv->fi.astreams].format.format = (( ps->asf.wfhead.wBitsPerSample == 8 ) ?
				XMM_SOUND_FMT_U8 : XMM_SOUND_FMT_S16LE );
		    priv->fi.ai[priv->fi.astreams].bitrate = ps->asf.wfhead.nAvgBytesPerSec * 8;
		    priv->fi.ai[priv->fi.astreams].tSamples = ps->avi_stream_header.dwLength;

		    priv->asIDX[priv->fi.astreams++] = i;

		    /* Only needed if idx1 mode */
		    if( reader->AudioRead == AudioRead )
		    {
			/* Build audio bytes table */
			if(( ps->idx1_total = malloc( sizeof( uint32_t ) * ps->idx1_entries )) == NULL )
			{
			    xmm_SetError( xmm, XMM_ERR_ALLOC, "(AVI) audio bytes table" );
			    return NULL;
			}

			for( j = 0, total = 0; j < ps->idx1_entries; j++ )
			{
			    ps->idx1_total[j] = total;
			
			    total += ps->idx1_chunk[j].dwChunkLength;
			}

			ps->total_size = total;
		    }

		    break;

	    default:
		    priv->pIO->Close( priv->pIO );
		    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(AVI) Unsupported stream type '%s'", xmm_FOURCC_string( avi_stream_head.fccType ));
		    return NULL;
	}
  }

  return reader;
}

/*
 * Internal code
 */

/*
 * Reader API calls
 */

/*
 * Close reader
 */

void Close( Reader_t *reader )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  int			i;

  for( i = 0; i < priv->nStreams; i++ )
      if( priv->aStream[i].rbuffer )	free( priv->aStream[i].rbuffer );

  free( reader );
}

/*
 * Get file info
 */
static XMM_FileInfo *FileInfo( Reader_t *reader )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;

  return &priv->fi;
}

/*
 * Get audio format
 */
static XMM_AudioFormat *AudioFormat( Reader_t *reader, int stream )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps;
  XMM_AudioFormat	*af;

  /* Lookup stream */
  if( stream >= priv->fi.astreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i audio streams.", stream, priv->fi.astreams );
	return NULL;
  }

  stream = priv->asIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return NULL;
  }
  ps = &priv->aStream[ stream ];

  /* Allocate memory */
  if(( af = malloc( sizeof( XMM_AudioFormat ) + ps->asf.wfhead.cbSize )) == NULL )
  {
	xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) Unable to allocate AudioFormat" );
	return NULL;
  }

  /* Fill audio format struct */
  af->formatID = ps->asf.wfhead.wFormatTag;
  af->channels = ps->asf.wfhead.nChannels;
  af->samprate = ps->asf.wfhead.nSamplesPerSec;
  af->bitrate = ps->asf.wfhead.nAvgBytesPerSec * 8;
  switch( ps->asf.wfhead.wBitsPerSample )
  {
	case 4:
		af->format = XMM_SOUND_FMT_U4;
		break;

	case 8:
		af->format = XMM_SOUND_FMT_U8;
		break;

	case 16:
		af->format = XMM_SOUND_FMT_S16LE;
		break;

	case 0:
		break;

	default:
		af->format = XMM_SOUND_FMT_S16LE;
		xmm_logging( 1, "AVI! WARNING: Unknown wBitsPerSample value (%i). Setting to 16.\n", ps->asf.wfhead.wBitsPerSample );
		break;
  }
  af->blockSize = ps->asf.wfhead.nBlockAlign;
  af->extraSize = ps->asf.wfhead.cbSize;
  af->desc[0] = '\0';
  if( ps->asf_extra )	memcpy( &af[1], ps->asf_extra, af->extraSize );

  return af;
}

/*
 * Read audio data ( sequential )
 */
static uint32_t AudioRead_raw( Reader_t *reader, int stream, char **dest, uint32_t *dsize )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps, *psb;
  uint32_t		read, demuxed = 0, proc, ret, i;
  uint32_t		dwSize, dwName, nextChunk;
  char			*ptr, *rbuffer, *endptr;

  /* Check for end of file */
  if( priv->pIO->Eof( priv->pIO ))
  {
	xmm_logging( 1, "AVI! End of File. Corrupted file ? Returning XMM_RET_EOS\n" );
	return XMM_RET_EOS;
  }

  /* Lookup stream */
  if( stream >= priv->fi.astreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i audio streams.", stream, priv->fi.astreams );
	return XMM_RET_INVALID_ARG;
  }
  stream = priv->asIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return XMM_RET_INVALID_ARG;
  }
  ps = &priv->aStream[ stream ];

  /* Amount of data to decode */
  read = *dsize;

  /* Resize buffer if needed */
  if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, read  ) < 0 )
  {
	xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) read buffer" );
	return XMM_RET_ERROR;
  }

  /* Set rbuffer pointer */
  rbuffer = ps->rbuffer;

  /* Get reamaining audio chunk bytes */
  if( xmmBQ_Size( &ps->bq ) > 0 )
  {
	proc = xmmBQ_Size( &ps->bq );
	if( proc > read )	proc = read;

	ret = xmmBQ_Read( &ps->bq, rbuffer, proc );
	if( ret != proc )
	{
	    xmm_logging( 1, "AVI! Unable to get requested number of bytes from buffer. Must NOT happen!!!\n" );
	}

	demuxed += ret;
	rbuffer += ret;
  }

  /* Get data from next audio chunks */
  while( demuxed < read )
  {
	dwName = READ32( priv->pIO );
	dwSize = READ32( priv->pIO );

	if( priv->pIO->Eof( priv->pIO ))
	{
	    xmm_logging( 1, "AVI! End of File. Corrupted file ?\n" );
	    break;
	}

#ifdef DEBUG
	xmm_logging( 3, "AVI! Chunk ID = '%s' ( %lx ), File pos = %lx, Size = %li ( %lx )\n", xmm_FOURCC_string( dwName ), dwName, priv->pIO->Tell( priv->pIO ) - 4, dwSize, dwSize );
#endif

	/* AVI is WORD aligned, NOTE: paddding byte must NOT be used */
	nextChunk = dwSize + (( dwSize & 1 ) ? 1 : 0 );

	/* Lookup stream */
	i = strtol( xmm_FOURCC_string( dwName & 0xFFFF ), &endptr, 16 );
	if( i >= priv->nStreams )	return	XMM_RET_INVALID_ARG;
	psb = &priv->aStream[i];

	/* Desired stream */
	if( psb == ps )
	{
	    /* Read some byte into the buffer */
	    proc = dwSize;
	    if( proc > ( read - demuxed ))	proc = read - demuxed;

	    ret = priv->pIO->Read( priv->pIO, rbuffer, 1, proc );
	    if( ret != proc )
	    {
		xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
	    }

	    demuxed += ret;
	    rbuffer += ret;
	    nextChunk -= ret;

	    /* Buffer remaining chunk data */
	    if( ret < dwSize )
	    {
		proc = dwSize - ret;	/* bytes to buffer */

		if(( ptr = malloc( proc )) == NULL )
		{
		    xmm_logging( 1, "AVI! Error allocation memory\n" );
		    return XMM_RET_ERROR;
		}

		ret = priv->pIO->Read( priv->pIO, ptr, 1, proc );
		if( ret != proc )
		{
		    xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
		}

		nextChunk -= ret;
		xmmBQ_Add( &priv->aStream[stream].bq, ptr, ret );
		free( ptr );
	    }
	}
	else
	{
	    ptr = xmmBQ_Add( &psb->bq, NULL, dwSize );

	    ret = priv->pIO->Read( priv->pIO, ptr, 1, dwSize );
	    if( ret != dwSize )
	    {
		xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes ( %i of %i ). Corrupted file ?\n", ret, dwSize );
	    }

	    nextChunk -= ret;
	}

	SEEK( priv->pIO, nextChunk, priv->seekable );
  }

  /* Return data */
  *dest = priv->aStream[stream].rbuffer;
  *dsize = demuxed;

  return XMM_RET_OK;
}

/*
 * Read audio data ( idx1 )
 */
static uint32_t	AudioRead( Reader_t *reader, int stream, char **dest, uint32_t *dsize )
{
  struct reader_priv_t		*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t		*ps;
  struct avi_index_entry_s	*idx;
  int				ret;
  uint32_t			read, demuxed = 0, proc, left;
  char				*rbuffer;
#ifdef AVI_CHECK_IDX1_ENTRIES
  uint32_t			dwName, dwSize;
#endif

  /* Lookup stream */
  if( stream >= priv->fi.astreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i audio streams.", stream, priv->fi.astreams );
	return XMM_RET_INVALID_ARG;
  }
  stream = priv->asIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return XMM_RET_INVALID_ARG;
  }
  ps = &priv->aStream[ stream ];

  /* Amount of data to decode */
  read = *dsize;

  /* Read data */
  if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, read  ) < 0 )
  {
	xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) read buffer" );
	return XMM_RET_ERROR;
  }

  /* Set rbuffer pointer */
  rbuffer = ps->rbuffer;

  /* Get data from next audio chunks */
  while( demuxed < read )
  {
	left = ps->idx1_chunk[ps->position].dwChunkLength - ps->bposition;

	/* No more data left in chunk */
	if( left == 0 )
	{
	    /* Check for EOS */
	    if( ps->position >= ps->idx1_entries )
	    {
		xmm_logging( 1, "AVI! End of File. Corrupted file ? Returning XMM_RET_EOS\n" );
		return XMM_RET_EOS;
	    }

	    /* Next chunk */
	    ps->position++;
	    ps->bposition = 0;
	}

	/* Get idx1 entry */
	idx = &ps->idx1_chunk[ ps->position ];

#ifdef AVI_CHECK_IDX1_ENTRIES

	/* Seek to position */
	priv->pIO->Seek( priv->pIO, idx->dwChunkOffset, XMM_SEEK_SET );

	/* Get chunk ID and size */
	dwName = READ32( priv->pIO );
	dwSize = READ32( priv->pIO );

	/* Check for conflicts */
	if( dwName != idx->dwChunkID )
	{
	    xmm_logging( 1, "AVI! ChunkID conflict! IDX1 = %x RAW = %x\n", idx->dwChunkID, dwName );
	}

	if(( dwSize != idx->dwChunkLength ) && (( dwSize + 1 ) != idx->dwChunkLength ))
	{
	    xmm_logging( 1, "AVI! ChunkSize conflict! IDX1 = %x RAW = %x\n", idx->dwChunkLength, dwSize );
	}

#endif

	/* Seek to position */
	priv->pIO->Seek( priv->pIO, idx->dwChunkOffset + 2 * sizeof( uint32_t ) + ps->bposition, XMM_SEEK_SET );

	/* Read some bytes into the buffer */
	proc = left;
	if( proc > ( read - demuxed ))	proc = read - demuxed;

	ret = priv->pIO->Read( priv->pIO, rbuffer, 1, proc );
	if( ret != proc )
	{
	    xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
	}

	demuxed += ret;
	rbuffer += ret;
	ps->bposition += ret;
  }

  /* Return data */
  *dest = ps->rbuffer;
  *dsize = demuxed;

  return XMM_RET_OK;
}

/*
 * Seek Audio ( dummy function: no seeking supported )
 */
static uint32_t AudioSeek_nop( Reader_t *reader, int stream, uint32_t bytes )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;

  xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) RAW mode. No seeking supported.\n" );

  return XMM_RET_NOTSUPPORTED;
}

/*
 * Seek Audio
 */
static uint32_t AudioSeek( Reader_t *reader, int stream, uint32_t bytes )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps;
  int			n, nl, nr;
#if 0
  int			ret;
  char			*dest;
  uint32_t		dsize;
#endif

  /* Lookup stream */
  if( stream >= priv->fi.astreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i video streams.", stream, priv->fi.astreams );
	return XMM_RET_INVALID_ARG;
  }
  stream = priv->asIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return XMM_RET_INVALID_ARG;
  }
  ps = &priv->aStream[ stream ];

  /* Block alignments */
  bytes -= ( bytes % ps->asf.wfhead.nBlockAlign );

  /* Find new audio chunk */
  nl = 0;
  nr = ps->idx1_entries;

  while( nl < ( nr - 1 ))
  {
	n = ( nl + nr ) >> 1;  				/* */

	if( ps->idx1_total[n] > bytes )	nr = n;		/* Skip right part */
	else	nl = n;					/* Skip left part */
  }

  ps->position = nl;

  /* Free buffered data */
  xmmBQ_Free( &ps->bq );

  /* Read audio data ( complete chunk ) */
#if 0
  dsize = ps->idx1_chunk[nl].dwChunkLength;
  ret = AudioRead( reader, stream, &dest, &dsize );
  if( ret != XMM_RET_OK )	return	ret;
#endif

  /* Skip some data */
  ps->bposition = bytes - ps->idx1_total[nl];

  return XMM_RET_OK;
}

/*
 * Get video format
 */
static XMM_VideoFormat *VideoFormat( Reader_t *reader, int stream )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps;
  XMM_VideoFormat	*vf;

  /* Lookup stream */
  if( stream >= priv->fi.vstreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i video streams.", stream, priv->fi.vstreams );
	return NULL;
  }
  stream = priv->vsIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return NULL;
  }
  ps = &priv->aStream[ stream ];

  /* Allocate memory */
  if(( vf = malloc( sizeof( XMM_VideoFormat ))) == NULL )
  {
	xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) Unable to allocate VideoFormat" );
	return NULL;
  }

  /* Fill video format struct */
  vf->codec = ps->asf.bmhead.biCompression;
  vf->width = ps->asf.bmhead.biWidth;
  vf->height = ps->asf.bmhead.biHeight;
  vf->planes = ps->asf.bmhead.biPlanes;
  vf->bpp = ps->asf.bmhead.biBitCount;
  vf->imgSize = ps->asf.bmhead.biSizeImage;
  vf->extraSize = 0;
  vf->desc[0] = '\0';

  return vf;
}

/*
 * Read video data ( sequential )
 */
static uint32_t	VideoRead_raw( Reader_t *reader, int stream, char **dest, uint32_t *dsize )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps, *psb;
  int			ret;
  uint32_t		dwSize, dwName, nextChunk, i;
  char			*ptr, *endptr;

  /* Check for end of file */
  if( priv->pIO->Eof( priv->pIO ))
  {
	xmm_logging( 1, "AVI! End of File. Corrupted file ? Returning XMM_RET_EOS\n" );
	return XMM_RET_EOS;
  }

  /* Lookup stream */
  if( stream >= priv->fi.vstreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i video streams.", stream, priv->fi.vstreams );
	return XMM_RET_INVALID_ARG;
  }
  stream = priv->vsIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return XMM_RET_INVALID_ARG;
  }
  ps = &priv->aStream[ stream ];

  /* No buffered frame */
  *dsize = 0;

  /* Get buffered frame */
  if( xmmBQ_Size( &ps->bq ) > 0 )
  {
	ret = xmmBQ_HeadSize( &ps->bq );

	if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, ret ) < 0 )
	{
		xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) read buffer" );
		return XMM_RET_ERROR;
	}

	*dsize = xmmBQ_HeadRead( &ps->bq, ps->rbuffer, ps->rbuffersize );
	if( *dsize < 0 )
	    xmm_logging( 1, "AVI! Unable to get buffer. Must NOT happen!!!\n" );
  }

  /* Get frame from next video chunks */
  while( *dsize == 0 )
  {
	dwName = READ32( priv->pIO );
	dwSize = READ32( priv->pIO );

	if( priv->pIO->Eof( priv->pIO ))
	{
	    xmm_logging( 1, "AVI! End of File. Corrupted file ?\n" );
	    break;
	}

#ifdef DEBUG
	xmm_logging( 3, "AVI! Chunk ID = '%s' ( %lx ), File pos = %lx, Size = %li ( %lx )\n", xmm_FOURCC_string( dwName ), dwName, priv->pIO->Tell( priv->pIO ) - 4, dwSize, dwSize );
#endif

	/* AVI is WORD aligned, NOTE: paddding byte must NOT be used */
	nextChunk = dwSize + (( dwSize & 1 ) ? 1 : 0 );

	/* Lookup stream */
	i = strtol( xmm_FOURCC_string( dwName & 0xFFFF ), &endptr, 16 );
	if( i >= priv->nStreams )	return	XMM_RET_INVALID_ARG;
	psb = &priv->aStream[i];

	/* Desired stream */
	if( psb == ps )
	{
	    if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, dwSize  ) < 0 )
	    {
		xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) read buffer" );
		return XMM_RET_ERROR;
	    }

	    ret = priv->pIO->Read( priv->pIO, ps->rbuffer, 1, dwSize );
	    nextChunk -= ret;
	    *dsize = ret;

	}
	else
	{
	    ptr = xmmBQ_Add( &psb->bq, NULL, dwSize );

	    ret = priv->pIO->Read( priv->pIO, ptr, 1, dwSize );
	    if( ret != dwSize )
	    {
		xmm_logging( 1, "AVI! I/O Read didn't return the requested number of bytes. Corrupted file ?\n" );
	    }

	    nextChunk -= ret;
	}

	SEEK( priv->pIO, nextChunk, priv->seekable );
  }

  /* Update frame number */
  ps->position++;

  /* Return data */
  *dest = ps->rbuffer;

  return XMM_RET_OK;
}

/*
 * Read video data ( idx1 )
 */
static uint32_t	VideoRead( Reader_t *reader, int stream, char **dest, uint32_t *dsize )
{
  struct reader_priv_t		*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t		*ps;
  struct avi_index_entry_s	*idx;
  int				ret;
  uint32_t			dwName, dwSize;

  /* Lookup stream */
  if( stream >= priv->fi.vstreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i video streams.", stream, priv->fi.vstreams );
	return XMM_RET_INVALID_ARG;
  }
  stream = priv->vsIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return XMM_RET_INVALID_ARG;
  }
  ps = &priv->aStream[ stream ];

  /* Check for EOS */
  if( ps->position >= ps->idx1_entries )
  {
	xmm_logging( 1, "AVI! End of Stream. Returning XMM_RET_EOS\n" );
	return XMM_RET_EOS;
  }

  /* Get idx1 entry */
  idx = &ps->idx1_chunk[ ps->position++ ];

  /* Reallocate buffer */
  if( buffer_realloc( &ps->rbuffer, &ps->rbuffersize, idx->dwChunkLength  ) < 0 )
  {
	xmm_SetError( priv->xmm, XMM_ERR_ALLOC, "(AVI) read buffer" );
	return XMM_RET_ERROR;
  }

  /* Seek to position */
  priv->pIO->Seek( priv->pIO, idx->dwChunkOffset, XMM_SEEK_SET );

  /* Get chunk ID and size */
  dwName = READ32( priv->pIO );
  dwSize = READ32( priv->pIO );

  /* Check for conflicts */
  if( dwName != idx->dwChunkID )
  {
	xmm_logging( 1, "AVI! ChunkID conflict! IDX1 = %x RAW = %x\n", idx->dwChunkID, dwName );
  }

  if(( dwSize != idx->dwChunkLength ) && (( dwSize + 1 ) != idx->dwChunkLength ))
  {
	xmm_logging( 1, "AVI! ChunkSize conflict! IDX1 = %x RAW = %x\n", idx->dwChunkLength, dwSize );
  }

  /* Read data */
  ret = priv->pIO->Read( priv->pIO, ps->rbuffer, 1, dwSize );

  /* Return data */
  *dsize = ret;
  *dest = ps->rbuffer;

  return XMM_RET_OK;
}

/*
 * Seek video ( dummy function: no seeking supported )
 */
static uint32_t VideoSeek_nop( Reader_t *reader, int stream, uint32_t frame, int keyframe )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;

  xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) RAW mode. No seeking supported.\n" );

  return XMM_RET_NOTSUPPORTED;
}

/*
 * Seek video
 */
static uint32_t VideoSeek( Reader_t *reader, int stream, uint32_t frame, int keyframe )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps;

  /* Lookup stream */
  if( stream >= priv->fi.vstreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, File has only %i video streams.", stream, priv->fi.vstreams );
	return XMM_RET_INVALID_ARG;
  }
  stream = priv->vsIDX[stream];

  if( stream >= priv->nStreams )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Wrong stream number: %i, Internal error (?).", stream );
	return XMM_RET_INVALID_ARG;
  }
  ps = &priv->aStream[ stream ];

  /* Check for EOS */
  if( frame >= ps->idx1_entries )
  {
	xmm_SetError( priv->xmm, XMM_ERR_UNKNOWN, "(AVI) Invalid frame number: %i, Stream contains only %i frames", frame, ps->idx1_entries );
	return XMM_RET_INVALID_ARG;
  }

  if( keyframe )
  {
	/* Find next keyframe */
	while((( ps->idx1_chunk[frame].dwFlags & AVIIF_KEYFRAME ) == 0 ) &&
		( frame < ps->idx1_entries ))	frame++;
  }

  /* End of stream, rewind stream */
  if( frame >= ps->idx1_entries )	frame = 0;

  /* Set frame number */
  ps->position = frame;

  return ps->position;
}


/*
 * Get current frame
 */
static uint32_t CurrentFrame( Reader_t *reader, int stream )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;

  return priv->aStream[ priv->vsIDX[stream]].position;
}

/*
 * Get total frames
 */
static uint32_t TotalFrames( Reader_t *reader )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;

  return priv->aStream[ priv->vsIDX[0]].avi_stream_header.dwLength;
}

/*
 * Get current time
 */
static double CurrentTime( Reader_t *reader )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps;

  /**/
  ps = &priv->aStream[ priv->vsIDX[0]];

  return ((double)ps->position * ps->avi_stream_header.dwScale / ps->avi_stream_header.dwRate );
}

/*
 * Get total time
 */
static double TotalTime( Reader_t *reader )
{
  struct reader_priv_t	*priv = (struct reader_priv_t *) reader->priv;
  struct stream_t	*ps;

  ps = &priv->aStream[ priv->vsIDX[0]];

  return ((double)ps->avi_stream_header.dwLength * ps->avi_stream_header.dwScale / ps->avi_stream_header.dwRate );
}

/*
 * Misc functions for file I/O
 */

static uint32_t READn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Read( pIO, data, size, 1 );
}

static uint32_t READ32( XMM_PluginIO *pIO )
{
  uint32_t tmp;

  pIO->Read( pIO, &tmp, 4, 1 );

  return tmp;
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return tmp;
}

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static int SEEK( XMM_PluginIO *pIO, long offset, int seekable )
{
  if( seekable )	return pIO->Seek( pIO, offset, XMM_SEEK_CUR );
  else
  {
	unsigned char tmp;
	int i;

	for( i = 0; i < offset; i++ )	pIO->Read( pIO, &tmp, 1, 1 );
	return i;
  }
}

/*
 * Read AVI headers
 */

/*
 * Main AVI header
 */

static int Read_avih( XMM_PluginIO *pIO, avi_header_t *avi_head, uint32_t size )
{
  if( size != 0x38 )
  {
	xmm_logging( 1, "AVI! avih: FILE ERROR: Size not 56 bytes ( is %i )\n", size );
	return 0;
  }

  avi_head->dwMicroSecPerFrame		= READ32( pIO );
  avi_head->dwMaxBytesPerSec		= READ32( pIO );
  avi_head->dwReserved1			= READ32( pIO );
  avi_head->dwFlags			= READ32( pIO );
  avi_head->dwTotalFrames		= READ32( pIO );
  avi_head->dwInitialFrames		= READ32( pIO );
  avi_head->dwStreams			= READ32( pIO );
  avi_head->dwSuggestedBufferSize	= READ32( pIO );
  avi_head->dwWidth			= READ32( pIO );
  avi_head->dwHeight			= READ32( pIO );
  avi_head->dwScale			= READ32( pIO );
  avi_head->dwRate			= READ32( pIO );
  avi_head->dwStart			= READ32( pIO );
  avi_head->dwLength			= READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 2, "AVI! avih: MicroSecs/Frame = %li\n", avi_head->dwMicroSecPerFrame );
  xmm_logging( 2, "AVI! avih: MaxBytes/Sec = %li\n", avi_head->dwMaxBytesPerSec );
  xmm_logging( 2, "AVI! avih: Reserved = %li\n", avi_head->dwReserved1 );
  xmm_logging( 2, "AVI! avih: Flags = %lx\n", avi_head->dwFlags );
  xmm_logging( 2, "AVI! avih: TotalFrames = %li\n", avi_head->dwTotalFrames );
  xmm_logging( 2, "AVI! avih: InitialFrames = %li\n", avi_head->dwInitialFrames );
  xmm_logging( 2, "AVI! avih: Streams = %li\n", avi_head->dwStreams );
  xmm_logging( 2, "AVI! avih: Suggested buffer size = %li\n", avi_head->dwSuggestedBufferSize	);
  xmm_logging( 2, "AVI! avih: Width = %li\n", avi_head->dwWidth );
  xmm_logging( 2, "AVI! avih: Height = %li\n", avi_head->dwHeight );
  xmm_logging( 2, "AVI! avih: Scale = %li\n", avi_head->dwScale );
  xmm_logging( 2, "AVI! avih: Rate = %li\n", avi_head->dwRate );
  xmm_logging( 2, "AVI! avih: Start = %li\n", avi_head->dwStart );
  xmm_logging( 2, "AVI! avih: Length = %li\n", avi_head->dwLength );
#endif

  return 1;
}

/*
 * Stream header
 */

static int Read_strh( XMM_PluginIO *pIO, avi_stream_header_t *avi_stream_head, uint32_t size )
{
  int tsize;

  if( size < 0x30 )
  {
	xmm_logging( 1, "AVI! avih: FILE ERROR: Size < 48 bytes ( is %i )\n", size );
	return 0;
  }

  avi_stream_head->fccType		= READ32( pIO );
  avi_stream_head->fccHandler		= READ32( pIO );
  avi_stream_head->dwFlags		= READ32( pIO );
  avi_stream_head->dwReserved1		= READ32( pIO );
  avi_stream_head->dwInitialFrames	= READ32( pIO );
  avi_stream_head->dwScale		= READ32( pIO );
  avi_stream_head->dwRate		= READ32( pIO );
  avi_stream_head->dwStart		= READ32( pIO );
  avi_stream_head->dwLength		= READ32( pIO );
  avi_stream_head->dwSuggestedBufferSize= READ32( pIO );
  avi_stream_head->dwQuality		= READ32( pIO );
  avi_stream_head->dwSampleSize		= READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 2, "AVI! strh: Type = '%s' ( %lx )\n", xmm_FOURCC_string( avi_stream_head->fccType ), avi_stream_head->fccType);
  xmm_logging( 2, "AVI! strh: Handler = '%s' ( %lx )\n", xmm_FOURCC_string( avi_stream_head->fccHandler ), avi_stream_head->fccHandler);
  xmm_logging( 2, "AVI! strh: Flags = %lx\n", avi_stream_head->dwFlags );
  xmm_logging( 2, "AVI! strh: Reserved = %li\n", avi_stream_head->dwReserved1 );
  xmm_logging( 2, "AVI! strh: InitialFrames = %li\n", avi_stream_head->dwInitialFrames );
  xmm_logging( 2, "AVI! strh: Scale = %li\n", avi_stream_head->dwScale );
  xmm_logging( 2, "AVI! strh: Rate = %li\n", avi_stream_head->dwRate );
  xmm_logging( 2, "AVI! strh: Start = %li\n", avi_stream_head->dwStart );
  xmm_logging( 2, "AVI! strh: Length = %li\n", avi_stream_head->dwLength );
  xmm_logging( 2, "AVI! strh: Suggested buffer size = %li\n", avi_stream_head->dwSuggestedBufferSize	);
  xmm_logging( 2, "AVI! strh: Quality = %li\n", avi_stream_head->dwQuality );
  xmm_logging( 2, "AVI! strh: Sample size = %li\n", avi_stream_head->dwSampleSize );
#endif

  if( size & 1 )	size++;
  for( tsize = 48; tsize < size; tsize++ )	READ8( pIO );

  return 1;
}

/*
 * Read stream format: video
 */

static int Read_strf_vids( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *avi_stream_head, struct reader_priv_t *priv )
{
  BITMAPINFOHEADER	*avi_bm_head = &priv->aStream[priv->nStreams].asf.bmhead;
  int tsize;

  avi_bm_head->biSize		= READ32( pIO );
  avi_bm_head->biWidth		= READ32( pIO );
  avi_bm_head->biHeight		= READ32( pIO );
  avi_bm_head->biPlanes		= READ16( pIO );
  avi_bm_head->biBitCount	= READ16( pIO );
  avi_bm_head->biCompression	= READ32( pIO );
  avi_bm_head->biSizeImage	= READ32( pIO );
  avi_bm_head->biXPelsPerMeter	= READ32( pIO );
  avi_bm_head->biYPelsPerMeter	= READ32( pIO );
  avi_bm_head->biClrUsed	= READ32( pIO );
  avi_bm_head->biClrImportant	= READ32( pIO );

#ifdef VERBOSE
  xmm_logging( 2, "AVI! strf_vid: Size = %li\n", avi_bm_head->biSize );
  xmm_logging( 2, "AVI! strf_vid: Width = %li\n", avi_bm_head->biWidth );
  xmm_logging( 2, "AVI! strf_vid: Height = %li\n", avi_bm_head->biHeight );
  xmm_logging( 2, "AVI! strf_vid: Planes = %i\n", avi_bm_head->biPlanes );
  xmm_logging( 2, "AVI! strf_vid: BitCount = %i\n", avi_bm_head->biBitCount );
  xmm_logging( 2, "AVI! strf_vid: Compression = '%s' ( %lx )\n", xmm_FOURCC_string( avi_bm_head->biCompression ), avi_bm_head->biCompression );
  xmm_logging( 2, "AVI! strf_vid: SizeImage = %li\n", avi_bm_head->biSizeImage );
  xmm_logging( 2, "AVI! strf_vid: XPels/m = %li\n", avi_bm_head->biXPelsPerMeter );
  xmm_logging( 2, "AVI! strf_vid: YPels/m = %li\n", avi_bm_head->biYPelsPerMeter );
  xmm_logging( 2, "AVI! strf_vid: ClrUsed = %li\n", avi_bm_head->biClrUsed );
  xmm_logging( 2, "AVI! strf_vid: ClrImportant = %li\n", avi_bm_head->biClrImportant );
#endif

  if( size & 1 )	size++;
  for( tsize = 40; tsize < size; tsize++ )	READ8( pIO );

  memcpy( &priv->aStream[priv->nStreams].avi_stream_header, avi_stream_head, sizeof( avi_stream_header_t ));
  priv->nStreams++;

  return 1;
}

/*
 * Read stream format: audio
 */

static int Read_strf_auds( XMM_PluginIO *pIO, uint32_t size, avi_stream_header_t *avi_stream_head, struct reader_priv_t *priv )
{
  WAVEFORMATEX		*avi_wf_head = &priv->aStream[priv->nStreams].asf.wfhead;

  avi_wf_head->wFormatTag	= READ16( pIO );
  avi_wf_head->nChannels	= READ16( pIO );
  avi_wf_head->nSamplesPerSec	= READ32( pIO );
  avi_wf_head->nAvgBytesPerSec	= READ32( pIO );
  avi_wf_head->nBlockAlign	= READ16( pIO );
  avi_wf_head->wBitsPerSample	= READ16( pIO );
  avi_wf_head->cbSize		= 0;
  priv->aStream[priv->nStreams].asf_extra	= NULL;

  if( avi_wf_head->wFormatTag != 0x01 )
  {
	avi_wf_head->cbSize		= READ16( pIO );

	if( size > ( 0x12 + avi_wf_head->cbSize ))
		xmm_logging( 1, "AVI! fmt: WARNING: Extra size < chunk size - 18 ( fixing, diff = %li bytes )\n", size - ( 0x12 + avi_wf_head->cbSize ));

	size -= 2;

	priv->aStream[priv->nStreams].asf_extra = malloc( size - 16 );
	if( priv->aStream[priv->nStreams].asf_extra == NULL )
	{
	    xmm_logging( 1, "AVI! fmt: ERROR allocating memory for extra data ( %i bytes )\n", size - 16 );
	    return 0;
	}
  
	READn( pIO, priv->aStream[priv->nStreams].asf_extra, size - 16 );
  }

#ifdef VERBOSE
  xmm_logging( 2, "AVI! strf_aud: Format = %i\n", avi_wf_head->wFormatTag );
  xmm_logging( 2, "AVI! strf_aud: Channels = %i\n", avi_wf_head->nChannels );
  xmm_logging( 2, "AVI! strf_aud: Samples / sec = %li\n", avi_wf_head->nSamplesPerSec );
  xmm_logging( 2, "AVI! strf_aud: Bytes / sec = %li\n", avi_wf_head->nAvgBytesPerSec );
  xmm_logging( 2, "AVI! strf_aud: BlockAlign = %i\n", avi_wf_head->nBlockAlign );
  xmm_logging( 2, "AVI! strf_aud: Bits / Sample = %i\n", avi_wf_head->wBitsPerSample );
  xmm_logging( 2, "AVI! strf_aud: Extra size = %i\n", avi_wf_head->cbSize );
#endif

  memcpy( &priv->aStream[priv->nStreams].avi_stream_header, avi_stream_head, sizeof( avi_stream_header_t ));
  priv->nStreams++;

  return 1;
}

/*
 * Read stream data
 */

static int Read_strd( XMM_PluginIO *pIO, unsigned char *data, uint32_t size )
{
  READn( pIO, data, size );
  return 1;
}

/*
 * Read idx1 chunk
 */

static int Read_idx1( XMM_PluginIO *pIO, uint32_t size, struct reader_priv_t *priv )
{
  uint32_t		minoff, count, i, str, offset;
  avi_index_entry_t	*avi_index_entry;
  char			*endptr;

  /* Calculate number of entries ( 16 bytes each ) */
  count = size >> 4;
#ifdef VERBOSE
  xmm_logging( 2, "AVI! idx1: %li entries\n", count );
#endif

  /* Allocate memory */
  if(( avi_index_entry = malloc( count * sizeof( avi_index_entry_t ))) == NULL )
  {
	xmm_logging( 1, "AVI! Error allocating memory..\n" );
	return 0; 
  }

  /* Read idx1 chunk */
  READn( pIO, avi_index_entry, count * sizeof( avi_index_entry_t ));

  /*
   * Note: minoff is needed to check, whether offset is from beginn of file
   * or relative to the movi list. ( this is the way xanim handles it )
   */
  for( i = 0, minoff = 0xFFFFFFFF; i < count; i++ )
  {
#if 0
	xmm_logging( 2, "AVI! idx1: %i: ID = '%s' ( %x )\n", i, xmm_FOURCC_string( avi_index_entry[i].dwChunkID ), avi_index_entry[i].dwChunkID );
	xmm_logging( 2, "AVI! idx1: %i: Flags = %x\n", i, avi_index_entry[i].dwFlags );
	xmm_logging( 2, "AVI! idx1: %i: Offset = %i\n", i, avi_index_entry[i].dwChunkOffset );
	xmm_logging( 2, "AVI! idx1: %i: Length = %i\n", i, avi_index_entry[i].dwChunkLength );
#endif

	if( avi_index_entry[i].dwChunkOffset < minoff )
			minoff = avi_index_entry[i].dwChunkOffset;
  }
  if( minoff < priv->data_offset )	offset = priv->data_offset - 4;
  else	offset = 0;

  /* Allocate memory for each stream */
  for( str = 0; str < priv->avi_head.dwStreams; str++ )
  {
	i = min( priv->aStream[str].avi_stream_header.dwLength, count );
	if(( priv->aStream[str].idx1_chunk = malloc( i * sizeof( avi_index_entry_t ))) == NULL )
	{
	    xmm_logging( 1, "AVI! Error allocating memory..\n" );
	    return 0; 
	}
  }

  /* Split idx1 chunk into a idx1 table for each stream */
  for( i = 0; i < count; i++ )
  {
	if( !isdigit( ((char *)&avi_index_entry[i].dwChunkID)[0] ) ||
	    !isdigit( ((char *)&avi_index_entry[i].dwChunkID)[1] ))	continue;

	str = strtol( xmm_FOURCC_string( avi_index_entry[i].dwChunkID & 0xFFFF ), &endptr, 16 );
	if(( str < 0 ) || ( str >= MAX_STREAMS ))	continue;

	memcpy( &priv->aStream[str].idx1_chunk[priv->aStream[str].idx1_entries], &avi_index_entry[i], sizeof( avi_index_entry_t ));
	priv->aStream[str].idx1_chunk[priv->aStream[str].idx1_entries++].dwChunkOffset += offset;
  }

  size -= ( count << 4 );
  while( size-- )	READ8( pIO );

  free( avi_index_entry );
  return 1;
}

/*
 *
 */

/*
 * Misc stuff
 */

/*
 * Realloc buffer
 */

static int buffer_realloc( char **buffer, int *oldsize, int newsize )
{
  /* Check, if rbuffer needs to be (re)allocated */
  if(( *oldsize < newsize ) && *buffer )
  {
	free( *buffer );
	*buffer = NULL;
  }

  /* Allocate buffer */
  if( *buffer == NULL )
  {
	*oldsize = newsize;
	if(( *buffer = malloc( *oldsize )) == NULL )
		return -1;
  }

  return *oldsize;
}
