/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * cdda.c
 * CD Digital Audio Plugin
 *
 * TODO:
 *  - in read mode, playback does not stop exactly at end of track
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/event.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/config.h>

#include "cdda.h"

/*
 * Definitions ( compile control )
 */

/* if defined, cdda read mode will be compiled
This enables reading as from any other plugin */
#define	XMM_CDDA_READ		0

/*
 * Definitions
 */

#define	DEFAULT_DEVICE		"/dev/cdrom"
#define	DEFAULT_CDINDEX_SERVER	"www.cdindex.org"

/*
 * Types
 */

struct priv_t
{
    int				FirstTrack, LastTrack, cTrack;
				/* CDDA can hold up to 99 audio tracks */
    struct cdr_tocentry		Track[100];
    struct cdr_discdata		Data;
    int				hasData;
    char			display[XMM_MAXPATHLEN];

    int 			fd;
    int				paused, stopped;

#ifdef XMM_CDDA_READ
    XMM_AudioInfo		ai;
    int				frames_cpos, frames_spos, frames_epos;
#endif

};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Config
 */

static char	cdr_device[XMM_MAXPATHLEN] = DEFAULT_DEVICE;
static int	cdr_volmode = 0;
static char 	cdindex_server[XMM_MAXPATHLEN] = DEFAULT_CDINDEX_SERVER;
static int	cdindex_enable = 0;
static char	display_format[XMM_MAXPATHLEN] = "%1 - %4";
#ifdef XMM_CDDA_READ
static int	cdr_readmode = 0;
#endif

static XMM_ConfigBlock cdda_config[] =
{
    { &cdr_device, "cdda_device", XMM_CFG_TYPE_STRING },
    { &cdr_volmode, "cdda_cdrvolume", XMM_CFG_TYPE_BOOL },
    { &cdindex_enable, "cdindex_enable", XMM_CFG_TYPE_BOOL },
    { &cdindex_server, "cdindex_server", XMM_CFG_TYPE_STRING },
    { &display_format, "display_format", XMM_CFG_TYPE_STRING },
#ifdef XMM_CDDA_READ
    { &cdr_readmode, "cdda_readmode", XMM_CFG_TYPE_BOOL },
#endif
    { NULL, "", 0 }
};

/*
 * Prototypes
 */

static void cdda_create_display( char *dest, struct cdr_discdata *Data, int track );
static int cdda_GetVolume( XMM_PluginInput *input, uint32_t *volume );
static int cdda_SetVolume( XMM_PluginInput *input, uint32_t volume );

#ifdef XMM_CDDA_READ
static int cdda_read_Close( XMM_PluginInput *input );
static int cdda_read_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data );
static int cdda_read_Seek( XMM_PluginInput *input, double seek );
static double cdda_read_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval );
static int cdda_read_AudioStreams( XMM_PluginInput *input );
static int cdda_read_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample );
static int cdda_read_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples );
static int cdda_read_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample );
#endif

#ifdef PROVIDE_GTK_WINDOWS
static void cdda_gtk_about( void );
static int cdda_gtk_configure( void );
#endif

/*
 * Init
 */

static XMM_PluginInput *cdda_Init( void *xmm )
{
  XMM_PluginInput *input;
  struct priv_t *priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(CDDA) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */

static int cdda_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_tochdr	th;
  struct cdrom_tocentry	te;
  char 			*ptr, device[256];
  XMM_Event_TOC		etoc;
  int			i;

#ifndef XMM_CDDA_READ
  /* Only direct out mode supported */
  if( flags & XMM_INPUT_CF_MODE_READ )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) Only 'direct out' mode compiled." );
#endif

  /* Read configuration */
  xmmCfg_BlockLoad( "config", "input-cdda", cdda_config );

  /* Check filename string */
  if( !strncasecmp( filename, "cdda://", 7 ))	ptr = filename + 7;
  else	if( !strncasecmp( filename, "cdda:", 5 ))	ptr = filename + 5;
  else	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) No CDDA file pattern: '%s'", filename );

  if( !isdigit( *ptr ))
  {
	memset( device, 0, 256 );
	memccpy( device, ptr, ':', 255 );
	if( device[ strlen( device ) - 1 ] == ':' )	device[ strlen( device ) - 1 ] = '\0';
	ptr += strlen( device );
	if(( ptr[0] == ':' ) && ( ptr[1] != '\0' ))	priv->cTrack = atoi( ++ptr );
	else	priv->cTrack = -1;
  }
  else
  {
	strcpy( device, cdr_device );
	priv->cTrack = atoi( ptr );
  }

  /* Open cdrom device */
  if(( priv->fd = open( device, O_RDONLY | O_NONBLOCK )) == -1 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) Error opening device %s: %s", device, strerror( errno ));

  /* Read table of content */
  if( ioctl( priv->fd, CDROMREADTOCHDR, &th ))
  {
	close( priv->fd );
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) ioctl ( CDROMREADTOCHDR ) failed" );
  }

  priv->FirstTrack = th.cdth_trk0;
  priv->LastTrack = th.cdth_trk1;

  if( priv->cTrack == -1 )	/* Get Track number */
  {
	etoc.type = XMM_EVENT_TOC;
	etoc.entries = priv->LastTrack - priv->FirstTrack + 1;
	if(( etoc.entry = malloc( 256 * etoc.entries )) == NULL )
		return xmm_SetError( input->sys.xmm, XMM_ERR_ALLOC, "(CDDA) Unable to allocate data for XMM_Event_TOC struct" );

	for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )
		sprintf( etoc.entry[i - priv->FirstTrack], "Track %.2i", i - priv->FirstTrack + 1 );

	i = xmm_PushEvent( input->sys.xmm, (XMM_Event *) &etoc, 0 );
	free( etoc.entry );

	if( i >= 0 )	priv->cTrack = etoc.selected + priv->FirstTrack;
	else	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) Wrong CDDA file pattern '%s' or no Track specified", filename );
  }

  xmm_logging( 3, "CDDA! Using device %s Track %i\n", device, priv->cTrack );

  for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )
  {
	te.cdte_track = i;
	te.cdte_format = CDROM_MSF;
	if( ioctl( priv->fd, CDROMREADTOCENTRY, &te ))
	{
	    close( priv->fd );
	    return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) ioctl ( CDROMREADTOCENTRY ) failed" );
	}
	priv->Track[ i - priv->FirstTrack ].type = ( te.cdte_ctrl & CDROM_DATA_TRACK );
	priv->Track[ i - priv->FirstTrack ].minute = te.cdte_addr.msf.minute;
	priv->Track[ i - priv->FirstTrack ].second = te.cdte_addr.msf.second;
	priv->Track[ i - priv->FirstTrack ].frame = te.cdte_addr.msf.frame;

	xmm_logging( 4, "CDDA! TE(%2i): track = %i\n", i, te.cdte_track );
	xmm_logging( 4, "CDDA! TE(%2i): format = %i\n", i, te.cdte_format );
	xmm_logging( 4, "CDDA! TE(%2i): ctrl = %i\n", i, te.cdte_ctrl );
	xmm_logging( 4, "CDDA! TE(%2i): addr = %i ( %i:%i:%i )\n", i, te.cdte_addr.lba, te.cdte_addr.msf.minute, te.cdte_addr.msf.second, te.cdte_addr.msf.frame );
	xmm_logging( 4, "CDDA! TE(%2i): adr = %i\n", i, te.cdte_adr );
	xmm_logging( 4, "CDDA! TE(%2i): datamode = %i\n", i, te.cdte_datamode );
  }

  if(( priv->cTrack < priv->FirstTrack ) || ( priv->cTrack > priv->LastTrack ) || priv->Track[ priv->cTrack - priv->FirstTrack ].type )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) Invalid track number ( %i )", priv->cTrack );

  te.cdte_track = CDROM_LEADOUT;
  te.cdte_format = CDROM_MSF;
  if( ioctl( priv->fd, CDROMREADTOCENTRY, &te ))
  {
	close( priv->fd );
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(CDDA) ioctl ( CDROMREADTOCENTRY ) failed" );
  }
  priv->Track[ i - priv->FirstTrack ].type = ( te.cdte_ctrl & CDROM_DATA_TRACK );
  priv->Track[ i - priv->FirstTrack ].minute = te.cdte_addr.msf.minute;
  priv->Track[ i - priv->FirstTrack ].second = te.cdte_addr.msf.second;
  priv->Track[ i - priv->FirstTrack ].frame = te.cdte_addr.msf.frame;

  xmm_logging( 4, "CDDA! TE(LO): track = %i\n", i, te.cdte_track );
  xmm_logging( 4, "CDDA! TE(LO): format = %i\n", i, te.cdte_format );
  xmm_logging( 4, "CDDA! TE(LO): ctrl = %i\n", i, te.cdte_ctrl );
  xmm_logging( 4, "CDDA! TE(LO): addr = %i ( %i:%i:%i )\n", i, te.cdte_addr.lba, te.cdte_addr.msf.minute, te.cdte_addr.msf.second, te.cdte_addr.msf.frame );
  xmm_logging( 4, "CDDA! TE(LO): adr = %i\n", i, te.cdte_adr );
  xmm_logging( 4, "CDDA! TE(LO): datamode = %i\n", i, te.cdte_datamode );

  for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )
  {
	xmm_logging( 3, "CDDA! Track %.2i Start = %.2i:%.2i.%.2i Length = %4.2f seconds\n",
					i - priv->FirstTrack + 1,
					priv->Track[ i - priv->FirstTrack ].minute,
					priv->Track[ i - priv->FirstTrack ].second,
					priv->Track[ i - priv->FirstTrack ].frame,
	(float)( LBA( priv->Track[i - priv->FirstTrack + 1] ) - LBA( priv->Track[i - priv->FirstTrack] )) / 75 );
  }

  /* Get cdindex information */
  if( cdindex_enable )
  {
	priv->hasData = cdindex_get_info( input->sys.xmm, cdindex_server, priv->FirstTrack, priv->LastTrack, priv->Track, &priv->Data );
	if( priv->hasData )
	{
	    xmm_logging( 3, "CDDA! Album: %s ( %s )\n", priv->Data.Title, ( priv->Data.ArtistType == CDDA_ARTIST_SINGLE ) ? "Single Artist" : "Multiple Artist" );
	    for( i = priv->FirstTrack; i <= priv->LastTrack; i++ )	xmm_logging( 3, "CDDA! Track %i: %s - %s\n", i, priv->Data.Track[i].Artist, priv->Data.Track[i].Title );
	}
  }

  /* Create title string */
  if( priv->hasData == 0 )	sprintf( priv->display, "CD Audio Track %.2i", priv->cTrack );
  else	cdda_create_display( priv->display, &priv->Data, priv->cTrack );

#ifdef XMM_CDDA_READ

  /* Now we are ready ( for direct output ) */
  if( flags & XMM_INPUT_CF_MODE_DOUT )	return 0;

  if(( flags & XMM_INPUT_CF_MODE_READ ) && !cdr_readmode )	return 0;

  /* Read mode desired, intitialize it */

  priv->frames_spos = LBA( priv->Track[ priv->cTrack - priv->FirstTrack ] );
  priv->frames_epos = LBA( priv->Track[ priv->cTrack - priv->FirstTrack + 1 ] );
  priv->frames_cpos = priv->frames_spos;

  priv->ai.format.samprate = 44100;
  priv->ai.format.channels = 2;
  priv->ai.format.format = XMM_SOUND_FMT_S16LE;
  priv->ai.bitrate = 44100 * 2 * 2 * 8;
  priv->ai.tSamples = ( priv->frames_epos - priv->frames_spos ) *
							CD_FRAMESIZE_RAW / 4;

  input->Close = cdda_read_Close;
  input->Control = cdda_read_Control;
  input->Seek = cdda_read_Seek;
  input->Info = cdda_read_Info;
  input->AudioStreams = cdda_read_AudioStreams;
  input->AudioInfo = cdda_read_AudioInfo;
  input->AudioRead = cdda_read_AudioRead;
  input->AudioSeek = cdda_read_AudioSeek;

  xmm_logging( 2, "CDDA! PCM Read Mode initialized\n" );
#endif

  return XMM_RET_OK;
}

/*
 * CDDA tracks do not contain video data
 */
static int cdda_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int cdda_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int cdda_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if CDDA
 */
static int cdda_Check( void *xmm, char *filename )
{
  return !( strncasecmp( filename, "cdda://", 7 ) && strncasecmp( filename, "cdda:", 5 ));
}

/*
 * Get file info
 */
static int cdda_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/* ========================================================================== *
 * * * * * * * * * * * * * * * * * * * CD ROM * * * * * * * * * * * * * * * * *
 * ========================================================================== */

/*
 * Close
 */
static int cdda_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  /* Stop playback */
  if( ioctl( priv->fd, CDROMSTOP ))

  /* Close file descriptor */
  close( priv->fd );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int cdda_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t	*priv = input->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_VOLUME:
		cdda_GetVolume( input, (uint32_t *) data );
		return XMM_CTLRET_ARG;		/* Result in arg */

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_DOUT |
#ifdef XMM_CDDA_READ
					XMM_INPUT_CF_MODE_READ |
#endif
					XMM_INPUT_CF_VOLUME |
					XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		cdda_SetVolume( input, param );
		return XMM_CTLRET_TRUE;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
		priv->paused = 0;
		priv->stopped = 0;
		input->Seek( input, 0.0 );
		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_STOP:
		priv->paused = 0;
		priv->stopped = 1;
		if( ioctl( priv->fd, CDROMSTOP ))
		{
		    xmm_logging( 1, "CDDA! ioctl ( CDROMSTOP ) failed\n" );
		    return XMM_CTLRET_ERROR;
		}
		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_PAUSE:
		{
		    int cmd;

		    if( priv->paused )	cmd = CDROMRESUME;
		    else	cmd = CDROMPAUSE;

		    if( ioctl( priv->fd, cmd ))
		    {
			xmm_logging( 1, "CDDA! ioctl ( CDROMPAUSE ) failed\n");
			return XMM_CTLRET_ERROR;
		    }

		    priv->paused = !priv->paused;

		    return XMM_CTLRET_TRUE;
		}

	case XMM_CTLINP_STATUS:
		{
		    struct cdrom_subchnl subchan;

		    subchan.cdsc_format = CDROM_MSF;
		    if( ioctl( priv->fd, CDROMSUBCHNL, &subchan ))
		    {
			xmm_logging( 1, "CDDA! ioctl ( CDROMSUBCHNL ) failed\n");
			return XMM_CTLRET_ERROR;
		    }

		    switch( subchan.cdsc_audiostatus )
		    {
			case CDROM_AUDIO_PLAY:
				*((uint32_t *) data ) = XMM_PBS_PLAYING;
				return XMM_CTLRET_ARG;

			case CDROM_AUDIO_PAUSED:
			case CDROM_AUDIO_COMPLETED:
				if( priv->paused )
					*((uint32_t *) data ) = XMM_PBS_PAUSED;
				else	*((uint32_t *) data ) = XMM_PBS_STOPPED;
				priv->stopped = 1;
				return XMM_CTLRET_ARG;

			default:
				if( priv->stopped )
					*((uint32_t *) data ) = XMM_PBS_STOPPED;
				else	*((uint32_t *) data ) = XMM_PBS_ERROR;
				return XMM_CTLRET_ARG;
		    }
		}

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
#ifdef PROVIDE_GTK_WINDOWS
		if(( param == XMM_GTKDLG_ABOUT ) ||
		    ( param == XMM_GTKDLG_CONFIG ))	return XMM_CTLRET_TRUE;
#endif
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		switch( param )
		{
#ifdef PROVIDE_GTK_WINDOWS
		    case XMM_GTKDLG_ABOUT:
			    cdda_gtk_about();
			    break;

		    case XMM_GTKDLG_CONFIG:
			    cdda_gtk_configure();
			    break;
#endif
		    default:
			    return XMM_CTLRET_NOTSUPPORTED;
		}

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int cdda_Seek( XMM_PluginInput *input, double seek )
{
  struct priv_t	*priv = input->sys.priv;
  struct cdrom_msf msf;
  int frame;

  frame = (int) ( seek * (double)( 
      LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) - 
      LBA( priv->Track[priv->cTrack - priv->FirstTrack] ))) +
      LBA( priv->Track[priv->cTrack - priv->FirstTrack] );


  msf.cdmsf_min0 = frame / 60 / 75;
  msf.cdmsf_sec0 = ( frame - msf.cdmsf_min0 * 60 * 75 ) / 75;
  msf.cdmsf_frame0 = frame % 75;
  msf.cdmsf_min1 = priv->Track[ priv->cTrack - priv->FirstTrack + 1 ].minute;
  msf.cdmsf_sec1 = priv->Track[ priv->cTrack - priv->FirstTrack + 1 ].second;
  msf.cdmsf_frame1 = priv->Track[ priv->cTrack - priv->FirstTrack + 1 ].frame;

  if( ioctl( priv->fd, CDROMPLAYMSF, &msf ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMPLAYMSF ) failed\n");
	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Get CDDA Track information
 */
static double cdda_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_subchnl	subchan;
  int			frame;

  /* Get current playtime */
  subchan.cdsc_format = CDROM_MSF;
  if( ioctl( priv->fd, CDROMSUBCHNL, &subchan ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMSUBCHNL ) failed\n");
	return (double)XMM_RET_OK;
  }

  frame = LBA( subchan.cdsc_absaddr.msf ) - LBA( priv->Track[priv->cTrack - priv->FirstTrack ] );

  if( ci )
  {
	strncpy( ci->name, priv->display, 64 );
	ci->name[63] = '\0';
	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = 0;
	ci->playtime = (double) (
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) -
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack] )) / 75;
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )
  {
	*seekval = (double)frame / ( 
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack + 1] ) -
	    LBA( priv->Track[priv->cTrack - priv->FirstTrack] ));
	if( *seekval < 0 )	*seekval = 0.0;
  }

  return (double)frame / 75;
}

/*
 * Get audio stream number
 * In DOUT mode CDDA plugin does not support AudioRead, so we return 0.
 */
static int cdda_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int cdda_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t		*priv = input->sys.priv;
  struct cdrom_subchnl	subchan;
  int			frame;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));
  if( cSample )
  {
	/* Get current playtime */
	subchan.cdsc_format = CDROM_MSF;
	if( ioctl( priv->fd, CDROMSUBCHNL, &subchan ))
	{
	    xmm_logging( 1, "CDDA! ioctl ( CDROMSUBCHNL ) failed\n");
	    return XMM_RET_OK;
	}

	frame = LBA( subchan.cdsc_absaddr.msf ) - LBA( priv->Track[priv->cTrack - priv->FirstTrack ] );

	*cSample = frame * CD_FRAMESIZE_RAW / 4;
  }

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int cdda_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Seek to position in audio stream
 */
static int cdda_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  return XMM_RET_NOTSUPPORTED;
}

/* ========================================================================== *
 * * * * * * * * * * * * * * * * * * Read Mode  * * * * * * * * * * * * * * * *
 * ========================================================================== */

#ifdef XMM_CDDA_READ

/*
 * Close
 */
static int cdda_read_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  /* Close file descriptor */
  close( priv->fd );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int cdda_read_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_SoundFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)data;
		if(( format->format == priv->ai.format.format ) &&
		    ( format->samprate = priv->ai.format.samprate ) &&
		    ( format->channels == priv->ai.format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)data;
		format->channels = priv->ai.format.channels;
		format->samprate = priv->ai.format.samprate;
		format->format = priv->ai.format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) =	XMM_INPUT_CF_MODE_DOUT |
					XMM_INPUT_CF_MODE_READ |
					XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
#ifdef PROVIDE_GTK_WINDOWS
		if(( param == XMM_GTKDLG_ABOUT ) ||
		    ( param == XMM_GTKDLG_CONFIG ))	return XMM_CTLRET_TRUE;
#endif
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		switch( param )
		{
#ifdef PROVIDE_GTK_WINDOWS
		    case XMM_GTKDLG_ABOUT:
			    cdda_gtk_about();
			    break;

		    case XMM_GTKDLG_CONFIG:
			    cdda_gtk_configure();
			    break;
#endif
		    default:
			    return XMM_CTLRET_NOTSUPPORTED;
		}

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int cdda_read_Seek( XMM_PluginInput *input, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  priv->frames_cpos = priv->frames_spos + (int) ( seek * (double)( priv->frames_epos - priv->frames_spos ));
  if( priv->frames_cpos >= priv->frames_epos )	priv->frames_cpos = priv->frames_epos;

  return XMM_RET_OK;
}

/*
 * Get Information
 */
static double cdda_read_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	strncpy( ci->name, priv->display, 64 );
	ci->name[63] = '\0';
	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = 0;
	ci->playtime = (double)( priv->frames_epos - priv->frames_spos ) / 75;
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )
  {
	*seekval = (double)( priv->frames_cpos - priv->frames_spos ) /
			(double)( priv->frames_epos - priv->frames_spos );
	if( *seekval < 0 )	*seekval = 0.0;
  }

  return (double)( priv->frames_cpos - priv->frames_spos ) / 75;
}

/*
 * Get audio stream number
 */
static int cdda_read_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int cdda_read_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));
  if( cSample )	*cSample = ( priv->frames_cpos - priv->frames_spos ) *
							CD_FRAMESIZE_RAW / 4;

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int cdda_read_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  struct priv_t			*priv = input->sys.priv;
  struct cdrom_read_audio	ra;
  int				frames;

  frames = samples * 4 / CD_FRAMESIZE_RAW;
  if( frames > ( priv->frames_epos - priv->frames_cpos ))
  {
	frames = priv->frames_epos - priv->frames_cpos;
	if( frames == 0 )	return XMM_RET_EOS;
	xmm_logging( 2, "CDDA! Only %i frames left\n", frames );
  }

  ra.addr.lba = priv->frames_cpos;
  ra.addr_format = CDROM_LBA;
  ra.nframes = frames;
  ra.buf = (void *)buffer;

  if( ioctl( priv->fd, CDROMREADAUDIO, &ra ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMREADAUDIO ) failed\n");
	return XMM_RET_ERROR;
  }

  priv->frames_cpos += frames;

  return frames * CD_FRAMESIZE_RAW / 4;
}

/*
 * Seek to position in audio stream
 */
static int cdda_read_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  struct priv_t	*priv = input->sys.priv;

  /* Calculate frame number */
  priv->frames_cpos = priv->frames_spos + ( sample * 4 ) / CD_FRAMESIZE_RAW;

  /* Only to be sure, we are in current track */
  if( priv->frames_cpos >= priv->frames_epos )
	priv->frames_cpos = priv->frames_epos;

  return XMM_RET_OK;
}

#endif

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = { { NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"CDDA",
				"Input: CD Digital Audio",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				cdda_Init, cdda_Open, cdda_Close,
				cdda_Control, cdda_Seek, cdda_Info,
				cdda_AudioStreams, cdda_AudioInfo,
				cdda_AudioRead, cdda_AudioSeek,
				cdda_VideoStreams, cdda_VideoInfo,
				cdda_VideoRead, cdda_VideoPTS, cdda_VideoSeek,
				cdda_Check, cdda_FileInfo };

/*
 * Internal code
 */

/*
 * Build song title info
 */

static void cdda_create_display( char *dest, struct cdr_discdata *Data, int track )
{
  char	ch, *format = display_format;
  int	idx;

  for( idx = 0; format[idx]; idx++ )
  {
	if( format[idx] != '%' )	*dest++ = format[idx];
	else
	{
	    ch = format[++idx];
	    switch( ch )
	    {
		    case 0:	idx--;
				*dest++ = '%';
				break;

		    case '%':	*dest++ = '%';
				break;

		    case '1':	strcpy( dest, Data->Track[track].Artist );
				dest += strlen( dest );
				break;

		    case '2':	strcpy( dest, Data->Title );
				dest += strlen( dest );
				break;

		    case '3':	sprintf( dest, "%.2i", track );
				dest += strlen( dest );
				break;

		    case '4':	strcpy( dest, Data->Track[track].Title );
				dest += strlen( dest );
				break;

		    default:	*dest++ = ch;	/* not any defined format */
				break;
	    }
	}
  }
}

/*
 * Set Volume
 */

static int cdda_SetVolume( XMM_PluginInput *input, uint32_t volume )
{
  struct priv_t	*priv = input->sys.priv;
  struct cdrom_volctrl vol;

  vol.channel0 = vol.channel2 = (( volume & 0x0000FFFF ) * 255 ) / 100;
  vol.channel1 = vol.channel3 = ((( volume & 0xFFFF0000 ) >> 16) * 255 ) / 100;
  if( ioctl( priv->fd, CDROMVOLCTRL, &vol ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMVOLCTRL ) failed\n");
	return 0;
  }
  return 1;
}

/*
 * Get Volume
 */

static int cdda_GetVolume( XMM_PluginInput *input, uint32_t *volume )
{
  struct priv_t	*priv = input->sys.priv;
  struct cdrom_volctrl vol;

  if( ioctl( priv->fd, CDROMVOLREAD, &vol ))
  {
	xmm_logging( 1, "CDDA! ioctl ( CDROMVOLREAD ) failed\n");
	return 0;
  }

  *volume = ( 100 * vol.channel0 / 255 ) | (( 100 * vol.channel1 / 255 ) << 16 );

  return 1;
}

/* ========================================================================== *
 * * * * * * * * * * * * * * * * * * Gtk Stuff  * * * * * * * * * * * * * * * *
 * ========================================================================== */

#ifdef PROVIDE_GTK_WINDOWS

#include <gtk/gtk.h>
#include "gtk/interface.h"
#include "gtk/support.h"

/*
 * About
 */

static void cdda_gtk_about( void )
{
  GtkWidget *about_window;

  about_window = create_cdda_about_window();
  gtk_widget_show( about_window );
}

/*
 * Configure
 */

/*
 * Callback for settings OK button
 */

void on_cdda_pref_ok_clicked( GtkWidget *pref_window, gpointer user_data )
{
#ifdef XMM_CDDA_READ
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_gen_mode_read_rbutton" ))->active )	cdr_readmode = 1;
  else cdr_readmode = 0;
#endif
	
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_gen_volume_cdr_rbutton" ))->active )	cdr_volmode = 1;
  else	cdr_volmode = 0;

  strcpy( cdr_device, gtk_entry_get_text( GTK_ENTRY( lookup_widget( pref_window,
			    "pref_gen_device_entry" ))));

  cdindex_enable = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(
	lookup_widget( pref_window, "pref_cdindex_enable_checkbutton" )));

  strcpy( cdindex_server, gtk_entry_get_text( GTK_ENTRY( lookup_widget( pref_window,
			    "pref_cdindex_entry" ))));

  strcpy( display_format, gtk_entry_get_text( GTK_ENTRY( lookup_widget( pref_window,
			    "pref_format_entry" ))));

  xmmCfg_BlockSave( "config", "input-cdda", cdda_config );

  gtk_widget_destroy( pref_window );
}

/*
 * Create window
 */

static int cdda_gtk_configure( void )
{
  GtkWidget	*pref_window;

  /* Create window */
  pref_window = create_cdda_pref_window();

  /* Load config */
  xmmCfg_BlockLoad( "config", "input-cdda", cdda_config );

  /*
   * Set widgets
   */

#ifdef XMM_CDDA_READ
  if( cdr_readmode == 1 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_gen_mode_read_rbutton" )), TRUE );
  else	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_gen_mode_cdr_rbutton" )), TRUE );
#endif

  if( cdr_volmode == 1 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_gen_volume_cdr_rbutton" )), TRUE );
  else	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_gen_volume_read_rbutton" )), TRUE );

  gtk_entry_set_text( GTK_ENTRY( lookup_widget( pref_window,
			    "pref_gen_device_entry" )), cdr_device );

  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_cdindex_enable_checkbutton" )), cdindex_enable );

  gtk_entry_set_text( GTK_ENTRY( lookup_widget( pref_window,
			    "pref_cdindex_entry" )), cdindex_server );

  gtk_entry_set_text( GTK_ENTRY( lookup_widget( pref_window,
			    "pref_format_entry" )), display_format );

  gtk_widget_show( pref_window );

  return 0;
}

#endif
