/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * CD Index support
 * ID Generation based on cdindex 1.2.0 ( www.cdindex.org )
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "sha.h"
#include "base64.h"
#include "cdda.h"

#include <libxmm/xmmp.h>
#include <libxmm/lpio.h>
#include <libxmm/util/utils.h>

/*
 * Definitions
 */

#define	CDINDEX_URL	"http://%s/cgi-bin/cdi/xget.pl?id=%s"

#define	TYPE_STRING	1
#define	TYPE_INT	2

/*
 * Prototypes
 */

static int GetEntry( char *src, char *tag, void *ptr, int type );
static int GetOption( char *src, char *tag, char *option, void *ptr, int type );

/*
 * Public code
 */

int cdindex_get_info( void *xmm, char *cdindex_server, int FirstTrack, int LastTrack, struct cdr_tocentry *Track, struct cdr_discdata *Data )
{
  SHA_INFO		sha;
  unsigned char		digest[20], *base64, *url, buffer[256];
  unsigned long		size;
  char			temp[9], discID[33], *ptr;
  int			i, ntracks, type = -1, track = -1;
  XMM_PluginIO		*pIO;

  /* Calculate ID */
  sha_init( &sha );

  sprintf( temp, "%02X", FirstTrack );		/* First Track: 1 byte */
  sha_update( &sha, (unsigned char *) temp, strlen( temp ));

  sprintf( temp, "%02X", LastTrack );		/* Last Track: 1 byte */
  sha_update( &sha, (unsigned char *) temp, strlen( temp ));

  sprintf( temp, "%08X", LBA( Track[LastTrack] ));	/* Leadout track: 4 byte */
  sha_update( &sha, (unsigned char *) temp, strlen( temp ));

  for( i = 1; i < 100; i++ )				/* Track n: 4 byte */
  {
 	sprintf( temp, "%08X", LBA( Track[i - 1] ) );
	if( i == ( LastTrack + 1 ))	sprintf( temp, "%08X", 0 );
	sha_update( &sha, (unsigned char *) temp, strlen( temp ));
  }

  sha_final( digest, &sha );

  base64 = rfc822_binary( digest, 20, &size );
  memcpy( discID, base64, size );
  discID[size] = 0;
  free( base64 );

  /* Query server */
  url = malloc( strlen( CDINDEX_URL ) + strlen( cdindex_server ) + strlen( discID ));
  if( url == NULL )	return 0;

  sprintf( url, CDINDEX_URL, cdindex_server, discID );
  xmm_logging( 2, "CDDA! CD Index request: %s\n", url );

//  pIO = xmm_IOOpen( xmm, "placebo-wyin.xml", XMM_IO_READ );
  pIO = xmm_IOOpen( xmm, url, XMM_IO_READ );
  free( url );
  if( pIO == NULL )	return 0;

  Data->ArtistType = -1;

  while( 1 ) 
  {
	for( ptr = buffer; ((int)ptr < ( (int)buffer + 256 )); ptr++ )
	{
		pIO->Read( pIO, ptr, 1, 1 );

		if( *ptr == '\0' || ( *ptr == 10 ))	break;
		if( pIO->Eof( pIO ))		break;
	};

	if( pIO->Eof( pIO ))	break;
	*ptr = '\0';
	ptr = buffer;

	if( ptr[ strlen( ptr ) - 1 ] == 10 )	ptr[ strlen( ptr ) - 1 ] = 0;
	while( isspace( *ptr ))	ptr++;

	if( *ptr != '<' )	continue;
    
	if( !strncmp( ptr, "<Title>", 7 ))
		GetEntry( ptr, "<Title>", (char *)Data->Title, TYPE_STRING );

	if( !strncmp( ptr, "<NumTracks>", 11 ))
		GetEntry( ptr, "<NumTracks>", (int *)&ntracks, TYPE_INT );
    
	if( !strncmp( ptr, "<SingleArtistCD>", 16 ) && ( Data->ArtistType == -1 ))
		Data->ArtistType = type = CDDA_ARTIST_SINGLE;

	if( !strncmp( ptr, "<MultipleArtistCD>", 18 ) && ( Data->ArtistType == -1 ))
		Data->ArtistType = type = CDDA_ARTIST_MULTIPLE;

	if( !strncmp( ptr, "<Artist>", 8 ))
	{
	    if( type == CDDA_ARTIST_SINGLE )
	    {
		GetEntry( ptr, "<Artist>", (char *)Data->Track[0].Artist, TYPE_STRING );
	    }
	    else if( type == CDDA_ARTIST_MULTIPLE )
	    {
		if( track != -1 )
		    GetEntry( ptr, "<Artist>", (char *)Data->Track[track].Artist, TYPE_STRING );
		else
		    xmm_logging( 1, "CDDA! CD Index: Error in format... unexpected '<Artist>' outside '<Track>'.\n" );
		    
	    }
	    else
	    {
		xmm_logging( 1, "CDDA! CD Index: Error in format... unexpected '<Artist>' outside '<xxxArtistCD>'.\n" );
	    }
	}

	if( !strncmp( ptr, "<Track ", 7 ))
		GetOption( ptr, "<Track>", "Num", (int *)&track, TYPE_INT );

	if( !strncmp( ptr, "<Name>", 6 ))
	{
		if( track != -1 )
		    GetEntry( ptr, "<Name>", (char *)Data->Track[track].Title, TYPE_STRING );
		else
		    xmm_logging( 1, "CDDA! CD Index: Error in format... unexpected '<Name>' outside '<Track>'.\n" );
	}

	if( !strncmp( ptr, "</Track>", 6 ))	track = -1;

	if( !strncmp( ptr, "</SingleArtistCD>", 16 ))	type = -1;

	if( !strncmp( ptr, "</MultipleArtistCD>", 18 ))	type = -1;
  }

  if( Data->ArtistType == CDDA_ARTIST_SINGLE )
  {
	for( i = 1; i <= ntracks; i++ )	strcpy( Data->Track[i].Artist, Data->Track[0].Artist );
  }

  pIO->Close( pIO );
  return 1;
}

/*
 * Private code
 */

static int GetEntry( char *src, char *tag, void *ptr, int type )
{
  char end_tag[256], *end;

  sprintf( end_tag, "</%s", tag + 1 );

  src += strlen( tag );
  end = strstr( src, end_tag );
  if( end == NULL )	return 0;

  *end = '\0';

  switch( type )
  {
	case TYPE_STRING:
		strcpy( (char *)ptr, src );
		break;

	case TYPE_INT:
		*((int *)ptr) = atoi( src );
		break;

	default:
		return 0;
  }

  return 1;
}

static int GetOption( char *src, char *tag, char *option, void *ptr, int type )
{
  src += strlen( tag ) - 1;

  while( *src )
  {
	if( !strncmp( src, option, strlen( option )))	break;
	while( *src && !isspace( *src ))	src++;
	src++;
  }

  if( *src == '\0' )	return 0;
  src += strlen( option ) + 1;
  if( *src != '"' )	return 0;
  src++;

  switch( type )
  {
	case TYPE_STRING:
		strcpy( (char *)ptr, src );
		break;

	case TYPE_INT:
		*((int *)ptr) = atoi( src );
		break;

	default:
		return 0;
  }

  return 1;
}
