/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mikmod.c
 * MikMod input
 *
 * TODO:
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <mikmod.h>

#include <libxmm/xmmp.h>
#include <libxmm/config.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/thread.h>
#include <libxmm/util/config.h>
#include "priv.h"

/*
 * Global data
 */

extern XMM_PluginInput		plugin_info;

extern MIKMODAPI		MDRIVER drv_xmm;

struct priv_t 			*mikmod_priv = NULL;

/*
 * Config
 */

static int	cfg_rate = 44100, cfg_size = 16, cfg_chan = 2;
static int	cfg_ipolate = 1, cfg_surround = 0;

static XMM_ConfigBlock mikmod_config[] =
{
    { &cfg_rate, "mikmod_rate", XMM_CFG_TYPE_INT },
    { &cfg_size, "mikmod_size", XMM_CFG_TYPE_INT },
    { &cfg_chan, "mikmod_chan", XMM_CFG_TYPE_INT },
    { &cfg_ipolate, "mikmod_interpolate", XMM_CFG_TYPE_BOOL },
    { &cfg_surround, "mikmod_surround", XMM_CFG_TYPE_BOOL },
    { NULL, "", 0 }
};


/*
 * Prototypes
 */

static int	mikmod_loop_thread( void *arg );

static int wrap_Seek( struct MREADER *_mreader, long offset, int whence );
static long wrap_Tell( struct MREADER *_mreader );
static int wrap_Read( struct MREADER *_mreader, void *ptr, size_t size );
static int wrap_Get( struct MREADER *_mreader );
static int wrap_Eof( struct MREADER *_mreader );

#ifdef PROVIDE_GTK_WINDOWS
static void mikmod_gtk_about( void );
static int mikmod_gtk_configure( void );
#endif

/*
 * Initialize
 */
static XMM_PluginInput *mikmod_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  /* Check for multiple use. This plugin ( and MikMod ) may only be used once */
  if( mikmod_priv )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(MikMod) MikMod already in use." );
	return NULL;
  }

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(MikMod) Unable to duplicate plugin_info" );
	return NULL;
  }

  /*
   * These have to be initialized
   */
  mikmod_priv = priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  /* Initialize sound */
  priv->pSound = xmm_SoundOpen( xmm, (char *)xmm_AcSoundPluginName( xmm, NULL ));
  if( priv->pSound == NULL )
  {
	free( input );
	return (void *)NULL;
  }

  return input;
}

/*
 * Open
 */
static int mikmod_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr, cmd[100], *tmpfile;

  /* Only direct out mode supported */
  if( flags & XMM_INPUT_CF_MODE_READ )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(MikMod) Only 'direct out' mode supported." );

  /* Read configuration */
  xmmCfg_BlockLoad( "config", "input-mikmod", mikmod_config );

  /* Initialize MikMod */
  MikMod_RegisterDriver( &drv_xmm );
  MikMod_RegisterAllLoaders();

  md_mixfreq = cfg_rate;

  md_mode = DMODE_SOFT_MUSIC | DMODE_SOFT_SNDFX;
  if( cfg_chan == 2 )	md_mode |= DMODE_STEREO;
  if( cfg_size == 16 )	md_mode |= DMODE_16BITS;
  if( cfg_surround )	md_mode |= DMODE_SURROUND;
  if( cfg_ipolate )	md_mode |= DMODE_INTERP;

  if( MikMod_Init( "" ))
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(MikMod) Cannot initialize mikmod: %s", MikMod_strerror( MikMod_errno ));

  /* Check filename */
  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(MikMod) Invalid file type. Should never happen." );

  /* Get uncompressed tracker module format file */
  tmpfile = xmm_gethome_filename( "tmp/mikmod_zip" );

  if( !strcasecmp( ptr, ".gz" ))
  {
	sprintf( cmd, "gzip -dc %s >>%s", filename, tmpfile );
	system( cmd );
	strcpy( priv->mikmod_path, tmpfile );
  }
  else if( !strcasecmp( ptr, ".bz2" ))
  {
	sprintf( cmd, "bzip2 -dc %s >>%s", filename, tmpfile );
	system( cmd );
	strcpy( priv->mikmod_path, tmpfile );
  }
  else	strcpy( priv->mikmod_path, filename );

  free( tmpfile );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->mikmod_filename = strdup( ptr + 1 );
  else	priv->mikmod_filename = strdup( buffer );
  free( buffer );

  /* Open file */
  priv->mreader.pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->mreader.pIO == NULL )	return XMM_RET_ERROR;

  priv->mreader.sys.Seek = wrap_Seek;
  priv->mreader.sys.Tell = wrap_Tell;
  priv->mreader.sys.Read = wrap_Read;
  priv->mreader.sys.Get = wrap_Get;
  priv->mreader.sys.Eof = wrap_Eof;
  /* Load file */
#if 0
  priv->module = Player_Load( priv->mikmod_path, 64, 0 );
#else
  priv->module = Player_LoadGeneric( &priv->mreader.sys, 64, 0 );
#endif
  if( priv->module == NULL )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(MikMod) Error loading file: %s\nFilename: %s", MikMod_strerror( MikMod_errno ), priv->mikmod_filename );

  /* Save audio info */
  priv->ai.format.samprate = md_mixfreq;
  priv->ai.format.channels = 2;
  priv->ai.format.format = XMM_SOUND_FMT_S16LE;
  priv->ai.bitrate = priv->module->bpm * 1000;
  priv->ai.tSamples = 0;
  priv->bps = priv->ai.format.channels * ( priv->ai.format.format & XMM_SOUND_MASK_SIZE ) / 8;

  /* Start sound */
  if(( priv->abuffersize = priv->pSound->Start( priv->pSound, &priv->ai.format, 8192, 4 )) < 0 )
  	return XMM_RET_ERROR;

  /* Allocate buffer size */
  if(( priv->abuffer = malloc( priv->abuffersize * priv->bps )) == NULL )
  {
	priv->pSound->Close( priv->pSound );
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(MikMod) audio buffer" );
  }

  /* Create thread */
  priv->mikmod_thread = xmmThread_Create( mikmod_loop_thread, (void *)priv );

  return XMM_RET_OK;
}

/*
 * Close
 */

static int mikmod_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  Player_Stop();
  Player_Free( priv->module );
  MikMod_Exit();

  priv->exit_mikmod_thread = 1;
  xmmThread_Wait( priv->mikmod_thread );

  free( priv->mikmod_filename );
  priv->mreader.pIO->Close( priv->mreader.pIO );

  if( priv->abuffer )	free( priv->abuffer );
  priv->abuffer = NULL;
  priv->pSound->Close( priv->pSound );

  mikmod_priv = NULL;
  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int mikmod_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t	*priv = input->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_DOUT | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
		Player_Start( priv->module );
		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_STOP:
		Player_Stop();
		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_PAUSE:
		Player_TogglePause();
		return XMM_CTLRET_TRUE;

	case XMM_CTLINP_STATUS:
		if( Player_Active())	*((uint32_t *) data ) = XMM_PBS_PLAYING;
		else	*((uint32_t *) data ) = XMM_PBS_STOPPED;
		return XMM_CTLRET_ARG;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
#ifdef PROVIDE_GTK_WINDOWS
		if(( param == XMM_GTKDLG_ABOUT ) ||
		    ( param == XMM_GTKDLG_CONFIG ))	return XMM_CTLRET_TRUE;
#endif
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		switch( param )
		{
#ifdef PROVIDE_GTK_WINDOWS
		    case XMM_GTKDLG_ABOUT:
			    mikmod_gtk_about();
			    break;

		    case XMM_GTKDLG_CONFIG:
			    mikmod_gtk_configure();
			    break;
#endif
		    default:
			    return XMM_CTLRET_NOTSUPPORTED;
		}

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int mikmod_Seek( XMM_PluginInput *input, double seek )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Get Information
 */
static double mikmod_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	strncpy( ci->name, Player_LoadTitle( priv->mikmod_path ), 64 );
	ci->name[63] = '\0';
	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = priv->mreader.pIO->Size( priv->mreader.pIO );
	ci->playtime = (double)0.0;
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )		*seekval = (double)priv->module->sngpos / priv->module->numpos;

  return (double)priv->module->sngtime / 1024;
}

/*
 * Get audio stream number
 * MikMod plugin does not support AudioRead. We still returned 1,
 * so information can be obtained using AudioInfo()
 */
static int mikmod_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int mikmod_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));
  if( cSample )	*cSample = 0;

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int mikmod_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Seek to position in audio stream
 */
static int mikmod_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * MikMod files do not contain video data
 */
static int mikmod_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int mikmod_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mikmod_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mikmod_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mikmod_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if MikMod file
 */
static int mikmod_Check( void *xmm, char *filename )
{
  char *ptr;
  
  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return -1;

  if( !strcasecmp( ptr, ".gz" ) || !strcasecmp( ptr, ".bz2" ))
  {
	for( ptr--; ( *ptr != '.' ) && ( ptr != filename ); ptr-- );
	if( ptr == filename )	return XMM_RET_ERROR;
  }

  if( !strncasecmp( ptr, ".669", 4 ))	return 1;	/* 669 and Extended-669 (by Tran/Renaissance) */
  if( !strncasecmp( ptr, ".amf", 4 ))	return 1;	/* DMP Advanced Module Format (by Otto Chrons) */
  if( !strncasecmp( ptr, ".dsm", 4 ))	return 1;	/* DSIK internal module format */
  if( !strncasecmp( ptr, ".far", 4 ))	return 1;	/* Farandole Composer (by Daniel Potter) */
  if( !strncasecmp( ptr, ".gdm", 4 ))	return 1;	/* General DigiMusic (by Edward Schlunder) */
  if( !strncasecmp( ptr, ".it", 3 ))	return 1;	/* Impulse Tracker (by Jeffrey Lim) */
  if( !strncasecmp( ptr, ".imf", 4 ))	return 1;	/* Imago Orpheus (by Lutz Roeder) */
  if( !strncasecmp( ptr, ".med", 4 ))	return 1;	/* Amiga MED modules (by Teijo Kinnunen) */
  if( !strncasecmp( ptr, ".m15", 4 ))	return 1;	/* Soundtracker 15-instrument */
  if( !strncasecmp( ptr, ".mod", 4 ))	return 1;	/* Standard 31-instrument Module loader */
  if( !strncasecmp( ptr, ".mtm", 4 ))	return 1;	/* Multi-Tracker Module (by Renaissance) */
  if( !strncasecmp( ptr, ".okt", 4 ))	return 1;	/* Amiga Oktalyzer */
  if( !strncasecmp( ptr, ".stm", 4 ))	return 1;	/* ScreamTracker 2 (by Future Crew) */
  if( !strncasecmp( ptr, ".stx", 4 ))	return 1;	/* STMIK 0.2 (by Future Crew) */
  if( !strncasecmp( ptr, ".s3m", 4 ))	return 1;	/* ScreamTracker 3 (by Future Crew) */
  if( !strncasecmp( ptr, ".ult", 4 ))	return 1;	/* UltraTracker (by MAS) */
  if( !strncasecmp( ptr, ".uni", 4 ))	return 1;	/* MikMod and APlayer internal module format */
  if( !strncasecmp( ptr, ".xm", 3 )) 	return 1;	/* FastTracker 2 (by Triton) */

  /* I have some mod's with this name pattern */
  if( !strncasecmp( filename, "mod.", 4 ))	return 1;
  
  return 0;
}

/*
 * Get file info
 */
static int mikmod_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}


/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"MikMod",
				"Input: MikMod",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				mikmod_Init, mikmod_Open, mikmod_Close,
				mikmod_Control, mikmod_Seek, mikmod_Info, 
				mikmod_AudioStreams, mikmod_AudioInfo,
				mikmod_AudioRead, mikmod_AudioSeek,
				mikmod_VideoStreams, mikmod_VideoInfo,
				mikmod_VideoRead, mikmod_VideoPTS,
				mikmod_VideoSeek,
				mikmod_Check, mikmod_FileInfo };

/*
 * Internal code
 */

/*
 * Player thread
 */

static int mikmod_loop_thread( void *arg )
{
  struct priv_t	*priv = arg;

  priv->exit_mikmod_thread = 0;

  do
  {
	if( Player_Active())	drv_xmm.Update();
	else	xmmSYS_usleep( 10000 );

  }  while( !priv->exit_mikmod_thread );

  return XMM_RET_OK;
}

/*
 * MREADER wrapper functions
 */

static int wrap_Seek( struct MREADER *_mreader, long offset, int whence )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  int			ret;

  ret = mreader->pIO->Seek( mreader->pIO, offset, whence );
  if( ret < XMM_RET_OK )	ret = -1;

  return ret;
}

static long wrap_Tell( struct MREADER *_mreader )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  long			ret;

  ret = mreader->pIO->Tell( mreader->pIO );
  if( ret < XMM_RET_OK )	ret = -1;

  return ret;
}

static int wrap_Read( struct MREADER *_mreader, void *ptr, size_t size )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  int			ret;

  ret = mreader->pIO->Read( mreader->pIO, ptr, size, 1 );
  if( ret < XMM_RET_OK )	ret = 0;

  return ret;
}

static int wrap_Get( struct MREADER *_mreader )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  char			tmp;

  if( mreader->pIO->Read( mreader->pIO, &tmp, 1, 1 ) < XMM_RET_OK )	tmp = -1;

  return (int) tmp;
}

static int wrap_Eof( struct MREADER *_mreader )
{
  struct mikmod_reader	*mreader = (struct mikmod_reader *)_mreader;
  int			ret;

  ret = mreader->pIO->Eof( mreader->pIO );
  if( ret < XMM_RET_OK )	ret = -1;

  return ret;
}

/* ========================================================================== *
 * * * * * * * * * * * * * * * * * * Gtk Stuff  * * * * * * * * * * * * * * * *
 * ========================================================================== */

#ifdef PROVIDE_GTK_WINDOWS

#include <gtk/gtk.h>
#include "gtk/interface.h"
#include "gtk/support.h"

/*
 * About
 */

static void mikmod_gtk_about( void )
{
  GtkWidget *about_window;

  about_window = create_mikmod_about_window();
  gtk_widget_show( about_window );
}

/*
 * Configure
 */

/*
 * Callback for settings OK button
 */

void on_mikmod_pref_ok_clicked( GtkWidget *pref_window, gpointer user_data )
{
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_44khz" ))->active )	cfg_rate = 44100;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_22khz" ))->active )	cfg_rate = 22050;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_11khz" ))->active )	cfg_rate = 11025;

  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
 		    "pref_quality_stereo" ))->active )	cfg_chan = 2;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_mono" ))->active )	cfg_chan = 1;

  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_16bit" ))->active )	cfg_size = 16;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_8bit" ))->active )	cfg_size = 8;

  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_ipolate" ))->active )		cfg_ipolate = 1;
  else	cfg_ipolate = 0;

  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_surround" ))->active )	cfg_surround = 1;
  else	cfg_surround = 0;

  xmmCfg_BlockSave( "config", "input-mikmod", mikmod_config );

  gtk_widget_destroy( pref_window );
}

/*
 * Create window
 */

static int mikmod_gtk_configure( void )
{
  GtkWidget	*pref_window;

  /* Create window */
  pref_window = create_mikmod_pref_window();

  /* Load config */
  xmmCfg_BlockLoad( "config", "input-mikmod", mikmod_config );

  /*
   * Set widgets
   */

  if( cfg_rate == 44100 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_44khz" )), TRUE );
  if( cfg_rate == 22050 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_22khz" )), TRUE );
  if( cfg_rate == 11025 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_11khz" )), TRUE );

  if( cfg_chan == 2 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_stereo" )), TRUE );
  if( cfg_chan == 1 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_mono" )), TRUE );

  if( cfg_size == 8 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_8bit" )), TRUE );
  if( cfg_size == 16 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_16bit" )), TRUE );

  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_ipolate" )), cfg_ipolate );

  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_surround" )), cfg_surround );

  gtk_widget_show( pref_window );

  return 0;
}

#endif
