/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mpg123.c
 * MPEG 1.0/2.0/2.5 Layer 1/2/3 input
 *
 * TODO:
 *  - AudioSeek() only seeks to sample positions multiple of the frame size
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/util/config.h>

#include "mpglib/libmpg123.h"

#include "libmpg123_info.h"
#include "libmpg123_vbr.h"
#include "id3v1.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		16384
 
#define alignUP( a,b )		((((a) + (b) - 1 ) / (b)) * (b))
#define	divUP( a, b )		(((a) + (b) - 1 ) / (b))

/*
 * Types
 */

typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;

    uint32_t			cSample, data_offset, tFrames, tSize;
    int				seekable, bps;
    char 			*filename;

    char			*rbuffer;
    int				rbuffersize;

    /* MP3 codec stuff */
    mpg123_t			mpg123;
    mpg123_reader_t		reader;
    XMM_BufferQueue		bq;
    MPG123_Info			mpg123_info;
    int				framesize_pcm;

    uint8_t			*pcm_buffer;
    uint32_t			pcm_buffer_pos;
    uint32_t			pcm_buffer_size;

    /* VBR header stuff */
    VBRHeader			vbrhead;
    int				vbrhead_found;

    /* File info and ID3 */
    char			mpg123_path[XMM_MAXPATHLEN];
    char			mpg123_song_title[XMM_MAXPATHLEN];
    ID3v1tag			id3v1tag;
    int				mpg123_has_id3v1tag;
};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Config
 */

static int	cfg_rate = 44100, cfg_size = 16, cfg_chan = 2;
static int	cfg_disable_id3tag = 0;
static char	cfg_display[XMM_MAXPATHLEN] = "%1 - %2";

static XMM_ConfigBlock mpg123_config[] =
{
    { &cfg_rate, "mpg123_rate", XMM_CFG_TYPE_INT },
    { &cfg_size, "mpg123_size", XMM_CFG_TYPE_INT },
    { &cfg_chan, "mpg123_chan", XMM_CFG_TYPE_INT },
    { &cfg_disable_id3tag, "mpg123_disable_id3tag", XMM_CFG_TYPE_BOOL },
    { cfg_display, "mpg123_display", XMM_CFG_TYPE_STRING },
    { NULL, "", 0 }
};

/*
 * Prototypes
 */
 
static void create_song_title( struct priv_t *priv );

#ifdef PROVIDE_GTK_WINDOWS

static void mpg123_gtk_about( void );
static int mpg123_gtk_configure( void );
static void mpg123_gtk_fileinfo( struct priv_t *priv );

#endif

/*
 * Initialize
 */
static XMM_PluginInput *mpg123_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(MPG123) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int mpg123_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  uint32_t		read;
  char			*buffer, *ptr, *vbr_buffer;
  XMM_BufferQueue	tmp_bq;
  int			done, ret;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(MPG123) Only 'read' mode supported" );

  /* Read configuration */
  xmmCfg_BlockLoad( "config", "input-mpg123", mpg123_config );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
  xmm_logging( 2, "MPG123! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Save filepath */
  strcpy( priv->mpg123_path, filename );

  /* Get ID3 Tag */
  for( priv->id3v1tag.Genre = 0; tab_genre[priv->id3v1tag.Genre][0] != '\0'; priv->id3v1tag.Genre++ );
  strcpy( priv->id3v1tag.Title, priv->filename );
  priv->id3v1tag.Artist[0] = '\0';
  priv->id3v1tag.Album[0] = '\0';
  priv->id3v1tag.Comment[0] = '\0';
  priv->id3v1tag.Year[0] = '\0';
  priv->mpg123_has_id3v1tag = id3v1tag_read( priv->mpg123_path, &priv->id3v1tag );

  create_song_title( priv );

  /* Initialize MPEG Layer 3 library */
  MPG123_Init( &priv->mpg123, 0 /*MMACCEL_CPU_586 | MMACCEL_X86_MMX*/ );

  /*
   * Decode one frame to get file info
   */

  read = 4096;		/* We read and buffer 4k */
  if(( buffer = malloc( read * 2 )) == NULL )	return XMM_RET_ERROR;

  /* Read data */
  read = priv->pIO->Read( priv->pIO, buffer, 1, read );

  /* Initialize buffer queue and buffer data for later, save buffer pointer */
  xmmBQ_Init( &priv->bq );
  vbr_buffer = xmmBQ_Add( &priv->bq, buffer, read );

  /* buffer data for now */
  xmmBQ_Init( &tmp_bq );
  xmmBQ_Add( &tmp_bq, buffer, read );

  /* Init reader */
  priv->reader.read = (bq_raw_read_t) xmmBQ_Read;

  /* Decode data ( buffer is bufferd in the queue, we decode data into this ) */
  ret = MPG123_DecodeFrame( &priv->mpg123, buffer, &done, &priv->reader, &tmp_bq );
  if( ret != MPG123_OK )	return XMM_RET_ERROR;

  /* Check for VBR header */
  priv->vbrhead_found = MPG123_GetVBRHeader( &priv->mpg123, vbr_buffer, &priv->vbrhead );

  if( priv->vbrhead_found )
  {
	int i;

	xmm_logging( 2, "MPG123! VBR Header found...\n" );
	xmm_logging( 3, "MPG123! VBR Header: Flags = 0x%lx\n", priv->vbrhead.flags );
	xmm_logging( 3, "MPG123! VBR Header: Frames = %li\n", priv->vbrhead.frames );
	xmm_logging( 3, "MPG123! VBR Header: Bytes = %li\n", priv->vbrhead.bytes );
	xmm_logging( 3, "MPG123! VBR Header: Scale = %li", priv->vbrhead.scale );

	for( i = 0; i < VBR_TOC_ENTRIES; i++ )
	{
	    if(( i % 10 ) == 0 )	xmm_logging( 4, "\nMPG123! VBR Header: TOC (%2i): ", i );
	    xmm_logging( 4, "%3i ", priv->vbrhead.toc[i] );
	}
	xmm_logging( 4, "\n" );
  }

  /* PCM size of a frame */
  priv->framesize_pcm = done;

  free( buffer );
  xmmBQ_Free( &tmp_bq );


  /* Allocate PCM buffer */
  priv->pcm_buffer_size = PCM_BUFFER_SIZE;
  priv->pcm_buffer_pos = 0;

  if(( priv->pcm_buffer = malloc( priv->pcm_buffer_size )) == NULL )
  {
	free( input );
	return xmm_SetError( input->sys.xmm, XMM_ERR_ALLOC, "(MPG123) PCM buffer" );
  }

  /* Get Info */
  MPG123_GetInfo( &priv->mpg123, &priv->mpg123_info );

  /* Query codec output format */
  priv->ai.format.channels = priv->mpg123_info.channels;
  priv->ai.format.samprate = priv->mpg123_info.sampRate;
  priv->ai.format.format = XMM_SOUND_FMT_S16LE;

  /* Initialize data */
  priv->rbuffersize = -1;
  priv->cSample = 0;
  priv->bps = priv->ai.format.channels * 2;

  if( priv->vbrhead_found )
  {
	priv->tSize = priv->vbrhead.bytes;
	priv->tFrames = priv->vbrhead.frames;
	priv->ai.bitrate = -((int32_t)(( priv->vbrhead.bytes / 
			((double) priv->tFrames * priv->mpg123_info.tpf )) *
			8 ));
  }
  else
  {
	priv->tSize = priv->pIO->Size( priv->pIO );
	priv->tFrames = priv->tSize / priv->mpg123_info.bpf;
	priv->ai.bitrate = priv->mpg123_info.bitRate * 1000;
  }

  priv->ai.tSamples = (uint32_t)((double) priv->tFrames *
			priv->mpg123_info.tpf *
			priv->ai.format.samprate );

  return XMM_RET_OK;
}

/*
 * Close
 */
static int mpg123_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  free( priv->filename );
  priv->pIO->Close( priv->pIO );

  if( priv->rbuffer )	free( priv->rbuffer );

  /* Exit MP3 */
  MPG123_Exit( &priv->mpg123 );
  xmmBQ_Free( &priv->bq );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int mpg123_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_SoundFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)data;
		if(( format->format == priv->ai.format.format ) &&
		    ( format->samprate = priv->ai.format.samprate ) &&
		    ( format->channels == priv->ai.format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)data;
		format->channels = priv->ai.format.channels;
		format->samprate = priv->ai.format.samprate;
		format->format = priv->ai.format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
#ifdef PROVIDE_GTK_WINDOWS
		if(( param == XMM_GTKDLG_ABOUT ) ||
		    ( param == XMM_GTKDLG_CONFIG ) ||
		    ( param == XMM_GTKDLG_INFO ))	return XMM_CTLRET_TRUE;
#endif
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		switch( param )
		{
#ifdef PROVIDE_GTK_WINDOWS
		    case XMM_GTKDLG_ABOUT:
			    mpg123_gtk_about();
			    break;

		    case XMM_GTKDLG_CONFIG:
			    mpg123_gtk_configure();
			    break;

		    case XMM_GTKDLG_INFO:
			    mpg123_gtk_fileinfo( priv );
			    break;
#endif
		    default:
			    return XMM_CTLRET_NOTSUPPORTED;
		}

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int mpg123_Seek( XMM_PluginInput *input, double seek )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	bytes, p1, sp1, sp2;
  double	sp;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_NOTSUPPORTED, "(MPG123) Stream is not seekable." );

  /* Clip seek value */
  if( seek < 0.0 )	seek = 0.0;
  if( seek > 100.0 )	seek = 100.0;

  /* Calculate new position */
  if( priv->vbrhead_found )
  {
	/* Get left seek point */
	p1 = (int)( seek * 100 );
	if( p1 > 99 )	p1 = 99;
	sp1 = priv->vbrhead.toc[ p1 ];

	/* Get right seek point */
	sp2 = priv->vbrhead.toc[ ( p1 < 99 ) ? ( p1 + 1 ) : 99 ];

	/* Interpolate seek position */
	sp = sp1 + ( sp2 - sp1 ) * ( seek * 100 - p1 );

	/* Get new position in bytes */
	bytes = (uint32_t)(( sp / 256.0 ) * priv->vbrhead.bytes );

	/* Update cSample, actually we guess it... because of the VBR */
	priv->cSample = (uint32_t)( seek * priv->ai.tSamples );
  }
  else
  {
	/* Get new position in bytes */
	bytes = (uint32_t)(( seek * priv->tFrames ) * priv->mpg123_info.bpf );

	/* Update cSample */
	priv->cSample = (uint32_t )((double) bytes / priv->mpg123_info.bpf *
			priv->mpg123_info.tpf * priv->ai.format.samprate );
  }

  /* Remove data from queue */
  xmmBQ_Free( &priv->bq );

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + bytes, XMM_SEEK_SET );
}

/*
 * Get Information
 */
static double mpg123_GetInfo( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	strncpy( ci->name, priv->mpg123_song_title, 64 );
	ci->name[63] = '\0';
	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = priv->pIO->Size( priv->pIO );
	ci->playtime = (double)priv->ai.tSamples / priv->ai.format.samprate;
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )		*seekval = (double)priv->cSample / priv->ai.tSamples;

  return (double)priv->cSample / priv->ai.format.samprate;
}

/*
 * Get audio stream number
 */
static int mpg123_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int mpg123_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));
  if( cSample )	*cSample = priv->cSample;

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int mpg123_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read, osize, buffered;
  int		decoded, ret, os = 0, needed;

  osize = read = samples * priv->bps;

  /* Get needed source size */
  read = divUP( read, priv->framesize_pcm ) * ( priv->mpg123.fsizeold + 4 );

  if(( priv->rbuffersize < read ) && priv->rbuffer )
  {
	free( priv->rbuffer );
	priv->rbuffer = NULL;
  }

  if( priv->rbuffer == NULL )
  {
	priv->rbuffersize = read;
	if(( priv->rbuffer = malloc( priv->rbuffersize )) == NULL )
	{
		return xmm_SetError( input->sys.xmm, XMM_ERR_ALLOC, "(MPG123) audio buffer." );
	}
  }

  /* Get buffered bytes */
  buffered = xmmBQ_Size( &priv->bq );
  if( buffered > read )	buffered = read;

  /* Read from buffer queue */
  if( buffered )	read -= buffered;

  /* Read from file */
  if( read )
  {
	read = priv->pIO->Read( priv->pIO, priv->rbuffer, 1, read );
	if(( read == 0 ) && priv->pIO->Tell( priv->pIO ))	return XMM_RET_EOS;
  }

  /*
   * First, we queue the source data
   */
  if( read )	xmmBQ_Add( &priv->bq, priv->rbuffer, read );

  /*
   * Get unused PCM data
   */
  if( priv->pcm_buffer_pos > 0 )
  {
	/* Calculate needed amount of data */
	os = priv->framesize_pcm - priv->pcm_buffer_pos;
	if( os > osize )	os = osize;

	/* Copy data */
	memcpy( buffer, priv->pcm_buffer + priv->pcm_buffer_pos, os );
	buffer += os;

	/* Updata buffer position */
	priv->pcm_buffer_pos += os;

	if( priv->pcm_buffer_pos == priv->framesize_pcm )
		priv->pcm_buffer_pos = 0;
  }

  /*
   * Decode data
   */
  while( os < osize )
  {
	/* Decode frame */
	decoded = priv->pcm_buffer_size;
	ret = MPG123_DecodeFrame( &priv->mpg123, priv->pcm_buffer, &decoded, &priv->reader, &priv->bq );
	if( ret == MPG123_MOREDATA )
	{
		/* Read from file */
		read = priv->pIO->Read( priv->pIO, priv->rbuffer, 1, priv->rbuffersize );
		if(( read == 0 ) && priv->pIO->Tell( priv->pIO ))	return XMM_RET_EOS;

		/* First, we queue the source data */
		if( read )	xmmBQ_Add( &priv->bq, priv->rbuffer, read );
		continue;
	}
	if( ret != MPG123_OK )	break;

	/* Set new frame size */
	priv->framesize_pcm = decoded;

	/* Calculate needed amount of data */
	needed = osize - os;
	if( needed > priv->framesize_pcm )	needed = priv->framesize_pcm;

	/* Copy data */
	memcpy( buffer, priv->pcm_buffer, needed );

	buffer += needed;
	os += needed;

	/* Update buffer pos ( this is the last loop ) */
	if( needed < priv->framesize_pcm )	priv->pcm_buffer_pos = needed;
  }

  priv->cSample += ( os / priv->bps );
  os /= priv->bps;

  return os;
}

/*
 * Seek to position in audio stream
 */
static int mpg123_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_NOTSUPPORTED, "(MPG123) Stream is not seekable." );

  /* Calculate / update cSample */
  priv->cSample = alignUP( sample * priv->bps, priv->mpg123_info.bpf ) / priv->bps;

  /* Remove data from queue */
  xmmBQ_Free( &priv->bq );

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + priv->cSample * priv->bps, XMM_SEEK_SET );
}

/*
 * MPG123 files do not contain video data
 */
static int mpg123_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int mpg123_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mpg123_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mpg123_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int mpg123_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if MPG123 file
 */
static int mpg123_Check( void *xmm, char *filename )
{
  char *ptr;

  ptr = strrchr( filename, '.' );
  if( ptr == NULL )	return XMM_RET_ERROR;

  return ( !strcasecmp( ptr, ".mp3" ) || !strcasecmp( ptr, ".mp2" ));
}

/*
 * Get file info
 */
static int mpg123_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"MPG123",
				"Input: MPEG 1.0/2.0/2.5 Layer 1/2/3",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				mpg123_Init, mpg123_Open, mpg123_Close,
				mpg123_Control, mpg123_Seek, mpg123_GetInfo, 
				mpg123_AudioStreams, mpg123_AudioInfo,
				mpg123_AudioRead, mpg123_AudioSeek,
				mpg123_VideoStreams, mpg123_VideoInfo,
				mpg123_VideoRead, mpg123_VideoPTS,
				mpg123_VideoSeek,
				mpg123_Check, mpg123_FileInfo };

/*
 * Internal code
 */

/*
 * Build song title info
 */

static void create_song_title( struct priv_t *priv )
{
  char	ch, *format = cfg_display, *ptr = priv->mpg123_song_title;
  int	idx;

  if( priv->mpg123_has_id3v1tag == 0 )
  {
	strcpy( priv->mpg123_song_title, priv->filename );
	return;
  }

  for( idx = 0; format[idx]; idx++ )
  {
	if( format[idx] != '%' )	*ptr++ = format[idx];
	else
	{
	    ch = format[++idx];
	    switch( ch )
	    {
		    case 0:	idx--;
				*ptr++ = '%';
				break;

		    case '%':	*ptr++ = '%';
				break;		    

		    case '1':	strcpy( ptr, priv->id3v1tag.Artist );
				ptr += strlen( priv->id3v1tag.Artist );
				break;

		    case '2':	strcpy( ptr, priv->id3v1tag.Title );
				ptr += strlen( priv->id3v1tag.Title );
				break;

		    case '3':	strcpy( ptr, priv->id3v1tag.Album );
				ptr += strlen( priv->id3v1tag.Album );
				break;

		    case '4':	strcpy( ptr, priv->id3v1tag.Year );
				ptr += strlen( priv->id3v1tag.Year );
				break;

		    case '5':	strcpy( ptr, priv->id3v1tag.Comment );
				ptr += strlen( priv->id3v1tag.Comment );
				break;

		    case '6':	strcpy( ptr, tab_genre[ priv->id3v1tag.Genre ] );
				ptr += strlen( tab_genre[ priv->id3v1tag.Genre ] );
				break;

		    case '7':	strcpy( ptr, priv->filename );
				ptr += strlen( priv->filename );
				break;

		    case '8':	*ptr++ = ' ';		/* extension */
				break;	

		    case '9':	*ptr++ = ' ';		/* path */
				break;

		    default:	*ptr++ = ch;		/* not any defined format */
				break;
	    }
	}
  }
}

/* ========================================================================== *
 * * * * * * * * * * * * * * * * * * Gtk Stuff  * * * * * * * * * * * * * * * *
 * ========================================================================== */

#ifdef PROVIDE_GTK_WINDOWS

#include <gtk/gtk.h>
#include "gtk/interface.h"
#include "gtk/support.h"

/*
 * About
 */

static void mpg123_gtk_about( void )
{
  GtkWidget *about_window;

  about_window = create_mpg123_about_window();
  gtk_widget_show( about_window );
}

/*
 * Configure
 */

/*
 * Callback for settings OK button
 */

void on_mpg123_pref_ok_clicked( GtkWidget *pref_window, gpointer user_data )
{
  char *ptr;
  
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_44khz" ))->active )	cfg_rate = 44100;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_22khz" ))->active )	cfg_rate = 22050;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_11khz" ))->active )	cfg_rate = 11025;

  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_stereo" ))->active )	cfg_chan = 2;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_mono" ))->active )	cfg_chan = 1;

  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_16bit" ))->active )	cfg_size = 16;
  if( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
		    "pref_quality_8bit" ))->active )	cfg_size = 8;

  cfg_disable_id3tag = GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
					"pref_title_disable_check" ))->active;

  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( pref_window, "pref_title_disp_entry" )));
  strcpy( cfg_display, ptr );

  xmmCfg_BlockSave( "config", "input-mpg123", mpg123_config );

  gtk_widget_destroy( pref_window );
}

/*
 * Callback for toggling title description on/off
 */

void on_pref_title_disable_check_released( GtkWidget *pref_window, gpointer user_data )
{
  cfg_disable_id3tag = ( GTK_TOGGLE_BUTTON( lookup_widget( pref_window,
				"pref_title_disable_check" ))->active + 1 ) & 1;

  gtk_widget_set_sensitive( lookup_widget( pref_window, "pref_title_disp_label" ), cfg_disable_id3tag );
  gtk_widget_set_sensitive( lookup_widget( pref_window, "pref_title_disp_entry" ), cfg_disable_id3tag );
  gtk_widget_set_sensitive( lookup_widget( pref_window, "pref_title_desc1_label" ), cfg_disable_id3tag );
  gtk_widget_set_sensitive( lookup_widget( pref_window, "pref_title_desc2_label" ), cfg_disable_id3tag );
  gtk_widget_set_sensitive( lookup_widget( pref_window, "pref_title_desc3_label" ), cfg_disable_id3tag );
}

/*
 * Create window
 */

static int mpg123_gtk_configure( void )
{
  GtkWidget	*pref_window;

  /* Create window */ 
  pref_window = create_mpg123_pref_window();

  /* Load config */
  xmmCfg_BlockLoad( "config", "input-mpg123", mpg123_config );

  /*
   * Set widgets
   */

  if( cfg_rate == 44100 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_44khz" )), TRUE );

  if( cfg_rate == 22050 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_22khz" )), TRUE );

  if( cfg_rate == 11025 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_11khz" )), TRUE );


  if( cfg_chan == 2 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_stereo" )), TRUE );

  if( cfg_chan == 1 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_mono" )), TRUE );


  if( cfg_size == 8 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_8bit" )), TRUE );

  if( cfg_size == 16 )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_quality_16bit" )), TRUE );


  if( cfg_disable_id3tag )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( lookup_widget(
			    pref_window, "pref_title_disable_check" )), TRUE );


  gtk_entry_set_text( GTK_ENTRY( lookup_widget( pref_window,
				"pref_title_disp_entry" )), cfg_display );

  gtk_widget_show( pref_window );

  return 0;
}

/*
 * File information
 */

/*
 * The info window callbacks need the priv struct,
 * however there is no way to pass it directly to the functions.
 */

struct win_lookup_entry
{
    GtkWidget			*widg;
    struct priv_t		*priv;
    struct win_lookup_entry	*next;
};


static struct win_lookup_entry *win_lookup_list = NULL;

/*
 * Convert ASCII-Z string to ID3 Tag string
 */

static void text2tag( char *text, char *tag, int len )
{
  memset( tag, 32, len );
  memcpy( tag, text, strlen( text ) > len ? len : strlen( text ));
}

/*
 * Callback for ID3 tag save
 */

void on_info_save_clicked( GtkWidget *info_window, gpointer user_data )
{
  char				*ptr;
  ID3v1tag			id3v1tag;
  struct win_lookup_entry	*le, **prev;
  struct priv_t			*priv = NULL;

  /* Set genre */
  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window,
					"info_id3tag_genre_combo_entry" )));

  for( id3v1tag.Genre = 0; tab_genre[id3v1tag.Genre]; id3v1tag.Genre++ )
	if( !strcmp( ptr, tab_genre[id3v1tag.Genre] ))
	    break;

  if( tab_genre[id3v1tag.Genre] == NULL )	id3v1tag.Genre--;
  
  /* Other ID3 stuff */
  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window,
					"info_id3tag_title_entry" )));
  text2tag( ptr, id3v1tag.Title, 30 );

  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window,
					"info_id3tag_artist_entry" )));
  text2tag( ptr, id3v1tag.Artist, 30 );

  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window,
					"info_id3tag_album_entry" )));
  text2tag( ptr, id3v1tag.Album, 30 );

  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window,
					"info_id3tag_comment_entry" )));
  text2tag( ptr, id3v1tag.Comment, 30 );

  ptr = gtk_entry_get_text( GTK_ENTRY( lookup_widget( info_window,
					"info_id3tag_year_entry" )));
  text2tag( ptr, id3v1tag.Year, 4 );

  /* Find entry in lookup list */
  if( win_lookup_list )
  {
	for( le = win_lookup_list, prev = &win_lookup_list; le; le = le->next )
	{
		if( le->widg == info_window )
		{
			priv = le->priv;
			*prev = le->next;
			free( le );
			break;
		}

		prev = &le->next;
	}
  }

  /* Save ID3 tag */
  id3v1tag_save( priv->mpg123_path, &id3v1tag );

  /* Close window */
  gtk_widget_destroy( info_window );
}

/*
 * Callback for ID3 tag remove
 */

void on_info_remove_clicked( GtkWidget *info_window, gpointer user_data )
{
  struct win_lookup_entry	*le, **prev;
  struct priv_t			*priv = NULL;

  /* Find entry in lookup list */
  if( win_lookup_list )
  {
	for( le = win_lookup_list, prev = &win_lookup_list; le; le = le->next )
	{
		if( le->widg == info_window )
		{
			priv = le->priv;
			*prev = le->next;
			free( le );
			break;
		}

		prev = &le->next;
	}
  }

  /* Remove ID3 tag, if entry found in list */
  if( priv )	id3v1tag_remove( priv->mpg123_path );

  /* Close window */
  gtk_widget_destroy( info_window );
}

/*
 * Create window
 */

static void mpg123_gtk_fileinfo( struct priv_t *priv )
{
  GtkWidget	*info_window;
  MPG123_Info	mpg123_info;
  char		temp[256];
  char		*tab_mpeg[] = { "1", "2", "?", "2.5" };
  char		*tab_layer[] = { "?", "I", "II", "III" };
  char		*tab_flag[] = { "No", "Yes" };
  char		*tab_mode[] = { "Stereo", "Joint stereo", "Dual channel", "Mono" };
  GList		*glist = NULL;
  int		i;
  struct win_lookup_entry	*le, *last;

  /* Create window */
  info_window = create_mpg123_info_window();

  /*
   * Add widget to lookup list
   */

  if(( le = malloc( sizeof( struct win_lookup_entry ))) == NULL )
  {
	xmm_logging( 1, "MPG123! ERROR allocating memory\n" );
	return;
  }

  /* Initialize data */
  le->next = NULL;
  le->widg = info_window;
  le->priv = priv;


  /* Append to list */
  if( win_lookup_list == NULL )	win_lookup_list = le;
  else
  {
	for( last = win_lookup_list; last->next; last = last->next );
	last->next = le;
  }

  /* Get MPEG Info */
  MPG123_GetInfo( &priv->mpg123, &mpg123_info );

  sprintf( temp, "MPEG %s Layer: %s\nBitrate: %li Kbits/s\nSamplerate: %li\n( %s )\n\nError Protection: %s\nCopyright: %s\nOriginal: %s\n\nEmphasis: %li\nFrames: %i\nSize: %i\n",
					tab_mpeg[mpg123_info.mpeg],
					tab_layer[mpg123_info.layer],
					mpg123_info.bitRate,
					mpg123_info.sampRate,
					tab_mode[mpg123_info.mode],
					tab_flag[ mpg123_info.crc ],
					tab_flag[ mpg123_info.copyright ],
					tab_flag[ mpg123_info.original ],
					mpg123_info.emphasis,
					priv->tFrames,
					priv->tSize );

  gtk_label_set_text( GTK_LABEL( lookup_widget( info_window, "info_mpeg_label" )), temp );

  /* Filename */
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window,
				"info_filename_entry" )), priv->mpg123_path );

  /* Read ID3tag */
  for( priv->id3v1tag.Genre = 0; tab_genre[priv->id3v1tag.Genre][0] != '\0'; priv->id3v1tag.Genre++ );
  strcpy( priv->id3v1tag.Title, priv->filename );
  priv->id3v1tag.Artist[0] = '\0';
  priv->id3v1tag.Album[0] = '\0';
  priv->id3v1tag.Comment[0] = '\0';
  priv->id3v1tag.Year[0] = '\0';

  id3v1tag_read( priv->mpg123_path, &priv->id3v1tag );

  /* Set widgets */
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window,
				"info_id3tag_title_entry" )), priv->id3v1tag.Title );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window,
				"info_id3tag_artist_entry" )), priv->id3v1tag.Artist );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window,
				"info_id3tag_album_entry" )), priv->id3v1tag.Album );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window,
				"info_id3tag_comment_entry" )), priv->id3v1tag.Comment );
  gtk_entry_set_text( GTK_ENTRY( lookup_widget( info_window,
				"info_id3tag_year_entry" )), priv->id3v1tag.Year );

  for( i = 0; tab_genre[i]; i++ )	glist = g_list_append( glist, tab_genre[i] );
  glist = g_list_sort( glist, (gint (*)(gconstpointer,gconstpointer))strcasecmp );
  gtk_combo_set_popdown_strings( GTK_COMBO( lookup_widget( info_window,
				"info_id3tag_genre_combo" )), glist );

  gtk_list_select_item( GTK_LIST( GTK_COMBO( lookup_widget( info_window,
				"info_id3tag_genre_combo" ))->list ),
				g_list_index( glist, tab_genre[priv->id3v1tag.Genre] ));

  /* Show window */
  gtk_widget_show( info_window );
}

#endif
