/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * vorbis.c
 * Ogg/Vorbis input
 *
 * TODO:
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

#include <vorbis/vorbisfile.h>

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;

    int				seekable, bps;
    char 			*filename;

    OggVorbis_File		ovf;
    int				cBitStream;

    /* vorbisfile FIX : Needed to save some values before forking */
    vorbis_info			*ovfix_vi;
    vorbis_comment		*ovfix_comment;
    uint32_t			ovfix_pcm_tell;
    double			ovfix_time_total;
    double			ovfix_time_tell;

};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Prototypes
 */

static int StreamSeek_wrap( void *priv, ogg_int64_t off, int whence );
static size_t StreamRead_wrap( void *ptr, size_t size, size_t nmemb, void *priv );
static int StreamClose_wrap( void *priv );
static long StreamTell_wrap( void *priv );

/*
 * Initialize
 */
static XMM_PluginInput *vorbis_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(Ogg/Vorbis) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int vorbis_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  char			*buffer, *ptr;
  vorbis_info		*vi;
  ov_callbacks		callbacks =
  {
    (size_t (*)(void *, size_t, size_t, void *))StreamRead_wrap,
    (int (*)(void *, ogg_int64_t, int))		StreamSeek_wrap,
    (int (*)(void *)) 				StreamClose_wrap,
    (long (*)(void *))				StreamTell_wrap
  };

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(Ogg/Vorbis) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
  xmm_logging( 2, "Ogg/Vorbis! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Open file by Ogg/Vorbis library */
  if( ov_open_callbacks((void *)priv->pIO, &priv->ovf, NULL, 0, callbacks ) < 0 )
  {
	priv->pIO->Close( priv->pIO );
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(Ogg/Vorbis) ov_open_callback failed. No Ogg/Vorbis format." );
  }

  vi = ov_info( &priv->ovf, -1 );

  /* Save information */
  priv->ai.format.samprate = vi->rate;
  priv->ai.format.channels = vi->channels;
  priv->ai.format.format = XMM_SOUND_FMT_S16LE;

  /* vorbisfile FIX : Save some values for later */
  priv->ovfix_vi = ov_info( &priv->ovf, -1 );
  priv->ovfix_comment = ov_comment( &priv->ovf, -1 );
  priv->ovfix_time_total = ov_time_total( &priv->ovf, -1 );

  /* Initialize data */
  priv->bps = priv->ai.format.channels * ( priv->ai.format.format & XMM_SOUND_MASK_SIZE ) / 8;
  priv->ai.bitrate = ov_bitrate( &priv->ovf, -1 );
  priv->ai.tSamples = ov_pcm_total( &priv->ovf, -1 );

  return XMM_RET_OK;
}

/*
 * Close
 */
static int vorbis_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  free( priv->filename );
  ov_clear( &priv->ovf );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control function
 */
static int vorbis_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_SoundFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)data;
		if(( format->format == priv->ai.format.format ) &&
		    ( format->samprate = priv->ai.format.samprate ) &&
		    ( format->channels == priv->ai.format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)data;
		format->channels = priv->ai.format.channels;
		format->samprate = priv->ai.format.samprate;
		format->format = priv->ai.format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_CTLRET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int vorbis_Seek( XMM_PluginInput *input, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_NOTSUPPORTED, "(Ogg/Vorbis) Stream is not seekable." );

  seek *= ov_time_total( &priv->ovf, -1 );
  ov_time_seek( &priv->ovf, seek );

  return XMM_RET_OK;
}

/*
 * Get Information
 */
static double vorbis_GetInfo( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	char *artist, *title, buffer[256];

	artist = vorbis_comment_query( priv->ovfix_comment, "artist", 0 );
	title = vorbis_comment_query( priv->ovfix_comment, "title", 0 );

	if( artist && title )	sprintf( buffer, "%s - %s", artist, title );
	else if( artist )	sprintf( buffer, "%s ( Unkown Title )", artist );
	else if( title )	sprintf( buffer, "%s ( Unkown Artist )", title );
	else			sprintf( buffer, "%s ( No Info )", priv->filename );

	strncpy( ci->name, buffer, 64 );
	ci->name[63] = '\0';

	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = priv->pIO->Size( priv->pIO );
	ci->playtime = priv->ovfix_time_total;
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )		*seekval = priv->ovfix_time_tell / priv->ovfix_time_total;

  return priv->ovfix_time_tell;
}

/*
 * Get audio stream number
 */
static int vorbis_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int vorbis_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));
  if( cSample )	*cSample = priv->ovfix_pcm_tell;

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int vorbis_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  struct priv_t *priv = input->sys.priv;
  long proc;

  proc = ov_read( &priv->ovf, (char *)buffer, samples * priv->bps, 0, 2, 1, &priv->cBitStream );
  if( proc == 0 )	return XMM_RET_EOS;	/* End of stream */
  else if( proc < 0 )
  {
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(Ogg/Vorbis) Error in Stream." );
  }

  /* vorbisfile FIX : Save time for later */
  priv->ovfix_time_tell = ov_time_tell( &priv->ovf );
  priv->ovfix_pcm_tell = (uint32_t) ov_pcm_tell( &priv->ovf );

  return proc / priv->bps;
}

/*
 * Seek to position in audio stream
 */
static int vorbis_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_NOTSUPPORTED, "(Ogg/Vorbis) Stream is not seekable." );

  ov_pcm_seek( &priv->ovf, (ogg_int64_t)sample );

  return XMM_RET_OK;
}

/*
 * Ogg/Vorbis files do not contain video data
 */
static int vorbis_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int vorbis_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int vorbis_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  return XMM_RET_NOTSUPPORTED;
}

static int vorbis_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int vorbis_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if Ogg/Vorbis file
 */
static int vorbis_Check( void *xmm, char *filename )
{
  XMM_PluginIO		*pIO;
  OggVorbis_File	ovf;
  ov_callbacks		callbacks =
  {
    (size_t (*)(void *, size_t, size_t, void *))StreamRead_wrap,
    (int (*)(void *, ogg_int64_t, int))		StreamSeek_wrap,
    (int (*)(void *)) 				StreamClose_wrap,
    (long (*)(void *))				StreamTell_wrap
  };
  int ret;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  ret = ov_open_callbacks((void *)pIO, &ovf, NULL, 0, callbacks );
  if( ret < 0 )
  {
	pIO->Close( pIO );
	return 0;
  }

  ov_clear( &ovf );

  return 1;
}

/*
 * Get file info
 */
static int vorbis_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"Ogg/Vorbis",
				"Input: Ogg/Vorbis",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				vorbis_Init, vorbis_Open, vorbis_Close,
				vorbis_Control, vorbis_Seek, vorbis_GetInfo,
				vorbis_AudioStreams, vorbis_AudioInfo,
				vorbis_AudioRead, vorbis_AudioSeek,
				vorbis_VideoStreams, vorbis_VideoInfo,
				vorbis_VideoRead, vorbis_VideoPTS,
				vorbis_VideoSeek,
				vorbis_Check, vorbis_FileInfo };

/*
 * Internal stuff
 */

static int StreamSeek_wrap( void *priv, ogg_int64_t off, int whence )
{
  int ret;

  ret = ((XMM_PluginIO *)priv)->Seek((XMM_PluginIO *)priv, (long)off, whence );

  if( ret == XMM_RET_NOTSUPPORTED )	return -1;
  return ret;
}

static size_t StreamRead_wrap( void *ptr, size_t size, size_t nmemb, void *priv )
{
  return (size_t)((XMM_PluginIO *)priv)->Read((XMM_PluginIO *)priv, ptr, (int)size, (int)nmemb );
}

static int StreamClose_wrap( void *priv )
{
  return ((XMM_PluginIO *)priv)->Close( (XMM_PluginIO *)priv );
}

static long StreamTell_wrap( void *priv )
{
  return ((XMM_PluginIO *)priv)->Tell( (XMM_PluginIO *)priv );
}
