/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wave.c
 * WAVE input
 *
 * TODO:
 *  - AudioSeek() only seeks to sample positions multiple of the block size
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpinput.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include "wave.h"

/*
 * Definitions
 */
 
#define alignUP( a,b )	((((a) + (b) - 1 ) / (b)) * (b))

/*
 * Types
 */

struct priv_t
{
    XMM_PluginCodecAudio	*pCodec;
    XMM_PluginIO		*pIO;
    XMM_AudioInfo		ai;

    uint32_t			cSample, data_offset;
    int				seekable, bps, blockSize;
    char 			*filename;

    char			*rbuffer;
    int				rbuffersize;
};

/*
 * Global data
 */

extern XMM_PluginInput	plugin_info;

/*
 * Prototypes
 */

static uint32_t	READn( XMM_PluginIO *wave, void *data, uint32_t size );
static uint32_t	READ32( XMM_PluginIO *wave );
static uint16_t	READ16( XMM_PluginIO *wave );
static uint8_t	READ8( XMM_PluginIO *wave );
static int	SEEK( XMM_PluginIO *wave, long offset, int seekable );
static int	Read_fmt( XMM_PluginIO *wave, WAVEFORMATEX *wave_fmt_head, uint32_t size, unsigned char **wave_fmt_extra );

static int wave_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample );

/*
 * Initialize
 */
static XMM_PluginInput *wave_Init( void *xmm )
{
  XMM_PluginInput	*input;
  struct priv_t		*priv;

  if(( input = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginInput ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(WAVE) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &input[1];
  input->sys.priv = (void *) priv;
  input->sys.xmm = xmm;

  return input;
}

/*
 * Open
 */
static int wave_Open( XMM_PluginInput *input, char *filename, int flags )
{
  struct priv_t		*priv = input->sys.priv;
  uint32_t		dwSize, dwName, dwTemp, data_size = 0;
  char			*buffer, *ptr;
  int			loop = 1;
  XMM_AudioFormat	*af;
  WAVEFORMATEX		wave_fmt_head;
  unsigned char		*wave_fmt_extra;

  /* Only read mode supported */
  if( flags & XMM_INPUT_CF_MODE_DOUT )
	return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(WAVE) Only 'read' mode supported" );

  /* Open file */
  priv->pIO = xmm_IOOpen( input->sys.xmm, filename, XMM_IO_READ );
  if( priv->pIO == NULL )	return XMM_RET_ERROR;

  /* Seekable ? */
  if( priv->pIO->Seek( priv->pIO, 0, XMM_SEEK_CUR ) == XMM_RET_OK )
	priv->seekable = 1;
  else	priv->seekable = 0;
  xmm_logging( 2, "WAVE! Stream is %s seekable\n", priv->seekable ? "" : "NOT" );

  /* Parse file */
  while( loop )
  {
	dwName = READ32( priv->pIO );
	if( priv->pIO->Eof( priv->pIO ))	break;

	xmm_logging( 2, "WAVE! Chunk ID = '%s' ( %lx )\n", xmm_FOURCC_string( dwName ), dwName );

	dwSize = READ32( priv->pIO );
	xmm_logging( 2, "WAVE! Chunk Size = %li ( %lx )\n", dwSize, dwSize );
	
	switch( dwName )
	{
		case RIFF_ID_RIFF:
			dwTemp = READ32( priv->pIO );
			if( dwTemp != RIFF_ID_WAVE )
			{
				priv->pIO->Close( priv->pIO );
				return xmm_SetError( input->sys.xmm, XMM_ERR_UNKNOWN, "(WAVE) No WAVE RIFF form ( '%s' )", xmm_FOURCC_string( dwTemp ));
			}
			xmm_logging( 2, "WAVE! RIFF Type = '%s'\n", xmm_FOURCC_string( dwTemp ));
			break;

		case RIFF_ID_fmt:
			Read_fmt( priv->pIO, &wave_fmt_head, dwSize, &wave_fmt_extra );
			break;

		case RIFF_ID_data:
			priv->data_offset = priv->pIO->Tell( priv->pIO );
			data_size = dwSize;
			xmm_logging( 2, "WAVE! data_offset = %li ( %lx )\n", priv->data_offset, priv->data_offset );
			if( dwSize & 1 )	dwSize++;
			if( priv->seekable )	SEEK( priv->pIO, dwSize, priv->seekable );
			else	loop = 0;
			break;

		default:
			xmm_logging( 2, "WAVE! Ignoring chunk '%s' (%lx)\n", xmm_FOURCC_string(dwName), dwName );
			if( dwSize & 1 )	dwSize++;
			SEEK( priv->pIO, dwSize, priv->seekable );
			break;
	}
  }

  /* Save filename */
  buffer = strdup( filename );
  ptr = strrchr( buffer, '/' );
  if( ptr )	priv->filename = strdup( ptr + 1 );
  else	priv->filename = strdup( buffer );
  free( buffer );

  /* Set input data */
  if(( af = malloc( sizeof( XMM_AudioFormat ) + wave_fmt_head.cbSize )) == NULL )
  {
	return xmm_SetError( input->sys.xmm, XMM_ERR_ALLOC, "(WAVE) acm_format_source" );
  }

  af->formatID = wave_fmt_head.wFormatTag;
  af->channels = wave_fmt_head.nChannels;
  af->samprate = wave_fmt_head.nSamplesPerSec;
  af->bitrate = wave_fmt_head.nAvgBytesPerSec * 8;
  af->format = (( wave_fmt_head.wBitsPerSample == 8 ) ?
				XMM_SOUND_FMT_U8 : XMM_SOUND_FMT_S16LE );
  af->blockSize = wave_fmt_head.nBlockAlign;
  af->extraSize = wave_fmt_head.cbSize;
  af->desc[0] = '\0';

  memcpy( &af[1], wave_fmt_extra, af->extraSize );

  /* Initialize audio codec */
  priv->pCodec = xmm_CodecAudioOpen( input->sys.xmm, XMM_CODEC_MODE_DECODE, af );
  if( priv->pCodec == NULL )	return XMM_RET_ERROR;

  /* Query codec output format */
  priv->pCodec->Control( priv->pCodec, XMM_CTLGET_SFORMAT, &priv->ai.format );

  /* Seek to the beginning of data */
  if( priv->seekable )	priv->pIO->Seek( priv->pIO, priv->data_offset, XMM_SEEK_SET );

  /* Initialize data */
  priv->rbuffersize = -1;
  priv->cSample = 0;
  priv->bps = priv->ai.format.channels * ( priv->ai.format.format & XMM_SOUND_MASK_SIZE ) / 8;
  priv->blockSize = af->blockSize;
  priv->ai.bitrate = af->bitrate;
  priv->ai.tSamples = (uint32_t)(((double)data_size / ( af->bitrate / 8 )) *
						priv->ai.format.samprate );

  return XMM_RET_OK;
}

/*
 * Close
 */
static int wave_Close( XMM_PluginInput *input )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->filename )	free( priv->filename );
  if( priv->pIO )	priv->pIO->Close( priv->pIO );
  if( priv->rbuffer )	free( priv->rbuffer );
  if( priv->pCodec )	priv->pCodec->Close( priv->pCodec );

  free( input );
  return XMM_RET_OK;
}

/*
 * Control
 */
static int wave_Control( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = input->sys.priv;
  XMM_SoundFormat	*format;

  switch( cmd )
  {
	case XMM_CTLQUERY_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Query sound format */
	case XMM_CTLQUERY_SFORMAT:
		format = (XMM_SoundFormat *)data;
		if(( format->format == priv->ai.format.format ) &&
		    ( format->samprate = priv->ai.format.samprate ) &&
		    ( format->channels == priv->ai.format.channels ))	return XMM_CTLRET_TRUE;
		return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_YFLIP:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLGET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get sound format */
	case XMM_CTLGET_SFORMAT:
		format = (XMM_SoundFormat *)data;
		format->channels = priv->ai.format.channels;
		format->samprate = priv->ai.format.samprate;
		format->format = priv->ai.format.format;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Get capabilities */
	case XMM_CTLGET_CAPS:
		*((uint32_t *) data ) = XMM_INPUT_CF_MODE_READ | XMM_INPUT_CF_AUDIO;
		return XMM_CTLRET_ARG;

	case XMM_CTLSET_GFORMAT:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_SCALE:
		return XMM_CTLRET_NOTSUPPORTED;

	case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Direct out mode */
	case XMM_CTLINP_PLAY:
	case XMM_CTLINP_STOP:
	case XMM_CTLINP_PAUSE:
	case XMM_CTLINP_STATUS:
		return XMM_CTLRET_NOTSUPPORTED;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_CTLRET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_INPUT )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No INPUT command */
}

/*
 * Seek to position
 */
static int wave_Seek( XMM_PluginInput *input, double seek )
{
  struct priv_t	*priv = input->sys.priv;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_NOTSUPPORTED, "(WAVE) Stream is not seekable." );

  /* Clip seek value */
  if( seek < 0.0 )	seek = 0.0;
  if( seek > 100.0 )	seek = 100.0;

  /* Calculate sample from seek value */
  return wave_AudioSeek( input, 0, (uint32_t)( seek * priv->ai.tSamples ));
}

/*
 * Get Information
 */
static double wave_Info( XMM_PluginInput *input, XMM_ClipInfo *ci, double *avdiff, double *seekval )
{
  struct priv_t	*priv = input->sys.priv;

  if( ci )
  {
	strncpy( ci->name, priv->filename, 64 );
	ci->name[63] = '\0';
	ci->author[0] = '\0';
	ci->copyright[0] = '\0';
	ci->size = priv->pIO->Size( priv->pIO );
	ci->playtime = (double)priv->ai.tSamples / priv->ai.format.samprate;
  }

  if( avdiff )		*avdiff = (double)0.0;

  if( seekval )		*seekval = (double)priv->cSample / priv->ai.tSamples;

  return (double)priv->cSample / priv->ai.format.samprate;
}

/*
 * Get audio stream number
 */
static int wave_AudioStreams( XMM_PluginInput *input )
{
  return 1;
}

/*
 * Get audio stream information
 */
static int wave_AudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai, uint32_t *cSample )
{
  struct priv_t	*priv = input->sys.priv;

  if( ai )	memcpy( ai, &priv->ai, sizeof( XMM_AudioInfo ));
  if( cSample )	*cSample = priv->cSample;

  return XMM_RET_OK;
}

/*
 * Read audio data
 */
static int wave_AudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t samples )
{
  struct priv_t *priv = input->sys.priv;
  uint32_t	read, decoded;

  decoded = read = samples * priv->bps;
  priv->pCodec->Control( priv->pCodec, XMM_CTLGET_DATA_SSIZE, &read );

  if(( priv->rbuffersize < read ) && priv->rbuffer )
  {
	free( priv->rbuffer );
	priv->rbuffer = NULL;
  }

  if( priv->rbuffer == NULL )
  {
	priv->rbuffersize = read;
	if(( priv->rbuffer = malloc( priv->rbuffersize )) == NULL )
	{
		return xmm_SetError( input->sys.xmm, XMM_ERR_ALLOC, "(WAVE) audio buffer." );
	}
  }

  read = priv->pIO->Read( priv->pIO, priv->rbuffer, 1, read );
  if(( read == 0 ) && priv->pIO->Tell( priv->pIO ))	return XMM_RET_EOS;

  decoded = samples;
  priv->pCodec->Decode( priv->pCodec, priv->rbuffer, read, buffer, &decoded, NULL );
  priv->cSample += decoded;

  return decoded;
}

/*
 * Seek to position in audio stream
 */
static int wave_AudioSeek( XMM_PluginInput *input, int stream, uint32_t sample )
{
  struct priv_t	*priv = input->sys.priv;
  uint32_t	bytes;

  if( priv->seekable == 0 )
	return xmm_SetError( input->sys.xmm, XMM_ERR_NOTSUPPORTED, "(WAVE) Stream is not seekable." );

  /* Calculate compressed data position */
  bytes = ( sample / priv->ai.format.samprate ) * priv->ai.bitrate / 8;

  /* Align bytes to a multiple of the blocksize */
  bytes = alignUP( bytes, priv->blockSize );

  /* Update cSample */
  priv->cSample = bytes / ( priv->ai.bitrate / 8 ) * priv->ai.format.samprate;

  /* I/O Seek */
  return priv->pIO->Seek( priv->pIO, priv->data_offset + bytes, XMM_SEEK_SET );
}

/*
 * WAVE files do not contain video data
 */
static int wave_VideoStreams( XMM_PluginInput *input )
{
  return 0;
}

static int wave_VideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame )
{
  return XMM_RET_NOTSUPPORTED;
}

static int wave_VideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[] )
{
  return XMM_RET_NOTSUPPORTED;
}

static int wave_VideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS )
{
  return XMM_RET_NOTSUPPORTED;
}

static int wave_VideoSeek( XMM_PluginInput *input, int stream, uint32_t frame )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Check if WAV file
 */
static int wave_Check( void *xmm, char *filename )
{
  uint32_t		dwSize, dwName, dwTemp;
  XMM_PluginIO		*pIO;

  pIO = xmm_IOOpen( xmm, filename, XMM_IO_READ );
  if( pIO == NULL )	return XMM_RET_ERROR;

  dwName = READ32( pIO );
  dwSize = READ32( pIO );
  dwTemp = READ32( pIO );

  pIO->Close( pIO );

  if(( dwName == RIFF_ID_RIFF ) && ( dwTemp == RIFF_ID_WAVE ))	return 1;
  else	return 0;
}

/*
 * Get file info
 */
static int wave_FileInfo( void *xmm, char *filename, XMM_ClipInfo *ci )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */

XMM_PluginInput	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_INPUT,
				0,
				XMM_VERSION_NUM,
				"",
				"WAVE",
				"Input: WAVE",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				wave_Init, wave_Open, wave_Close,
				wave_Control, wave_Seek, wave_Info, 
				wave_AudioStreams, wave_AudioInfo,
				wave_AudioRead, wave_AudioSeek,
				wave_VideoStreams, wave_VideoInfo,
				wave_VideoRead, wave_VideoPTS, wave_VideoSeek,
				wave_Check, wave_FileInfo };

/*
 * Internal code
 */

/*
 * Misc functions for file i/o
 */

static uint32_t READn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Read( pIO, data, size, 1 );
}

static uint32_t READ32( XMM_PluginIO *pIO )
{
  uint32_t tmp;

  pIO->Read( pIO, &tmp, 4, 1 );

  return tmp;
}

static uint16_t READ16( XMM_PluginIO *pIO )
{
  uint16_t tmp;

  pIO->Read( pIO, &tmp, 2, 1 );

  return tmp;
}

static uint8_t READ8( XMM_PluginIO *pIO )
{
  uint8_t tmp;

  pIO->Read( pIO, &tmp, 1, 1 );

  return tmp;
}

static int SEEK( XMM_PluginIO *pIO, long offset, int seekable )
{
  if( seekable )	return pIO->Seek( pIO, offset, XMM_SEEK_CUR );
  else
  {
	uint8_t tmp;
	int i;

	for( i = 0; i < offset; i++ )	pIO->Read( pIO, &tmp, 1, 1 );
	return i;
  }
}

/*
 * Read WAVE format header
 */

static int Read_fmt( XMM_PluginIO *pIO, WAVEFORMATEX *wave_fmt_head, uint32_t size, unsigned char **wave_fmt_extra )
{
  if( size < 0x10 )
  {
	xmm_logging( 1, "WAVE! fmt: FILE ERROR: Size < 16 bytes ( is %li )\n", size );
	return 0;
  }

  wave_fmt_head->wFormatTag		= READ16( pIO );
  wave_fmt_head->nChannels		= READ16( pIO );
  wave_fmt_head->nSamplesPerSec		= READ32( pIO );
  wave_fmt_head->nAvgBytesPerSec	= READ32( pIO );
  wave_fmt_head->nBlockAlign		= READ16( pIO );
  wave_fmt_head->wBitsPerSample		= READ16( pIO );
  wave_fmt_head->cbSize			= 0;

  if( wave_fmt_head->wFormatTag != 0x01 )
  {
	if( size < 0x12 )
	{
		xmm_logging( 1, "WAVE! fmt: FILE ERROR: Format not 0x01 and no extradata found\n" );
		return 0;
	}

	wave_fmt_head->cbSize		= READ16( pIO );

	if( size > ( 0x12 + wave_fmt_head->cbSize ))
		xmm_logging( 1, "WAVE! fmt: WARNING: Extra size < chunk size - 18 ( fixing, diff = %li bytes )\n", size - ( 0x12 + wave_fmt_head->cbSize ));

	size -= 2;

	*wave_fmt_extra = malloc( size - 16 );
	if( *wave_fmt_extra == NULL )
	{
	    xmm_logging( 1, "WAVE! fmt: ERROR allocating memory for extra data ( %li bytes )\n", size - 16 );
	    return 0;
	}
  
	READn( pIO, *wave_fmt_extra, size - 16 );
  }
  else
  {
	if( size != 16 )
	{
		xmm_logging( 1, "WAVE! fmt: ERROR: Header size for PCM format not 16 ( is %li )\n", size );
		return 0;
	}
  }

  xmm_logging( 2, "WAVE! fmt: FormatTag = %x\n", wave_fmt_head->wFormatTag );
  xmm_logging( 2, "WAVE! fmt: Channels = %i\n", wave_fmt_head->nChannels );
  xmm_logging( 2, "WAVE! fmt: Samples/Sec = %li\n",  wave_fmt_head->nSamplesPerSec );
  xmm_logging( 2, "WAVE! fmt: Avg Bytes/Sec = %li\n", wave_fmt_head->nAvgBytesPerSec );
  xmm_logging( 2, "WAVE! fmt: BlockAlign = %i\n", wave_fmt_head->nBlockAlign );
  xmm_logging( 2, "WAVE! fmt: Bits / Sample = %i\n", wave_fmt_head->wBitsPerSample );
  xmm_logging( 2, "WAVE! fmt: Extra data: %i bytes\n", wave_fmt_head->cbSize );

  return 1;
}
