/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * file.c
 * File I/O
 *
 * TODO:
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <libxmm/lpio.h>
#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

/*
 * Types
 */

struct priv_t
{
    FILE			*stream;
};

/*
 * Global data
 */

extern XMM_PluginIO	plugin_info;

/*
 * Open file
 */

static XMM_PluginIO *file_Open( void *xmm, const char *url, int mode )
{
  XMM_PluginIO	*io;
  struct priv_t *priv;
  char		smode[2] = " ";

  /* Check filename */
  if( strncmp( url, "file://", 7 ) && strstr( url, "://" ))
	return NULL;

  if( !strncmp( url, "file://", 7 ))	url += 7;

  /* Check mode */
  if( mode == XMM_IO_READ )	smode[0] = 'r';
  else if( mode == XMM_IO_WRITE )	smode[0] = 'w';
  else if( mode == XMM_IO_APPEND )	smode[0] = 'a';
  else
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(FILE) Wrong mode ( %i )", mode );
	return NULL;
  }

  /* Initialize plugin */
  if(( io = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginIO ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(FILE) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &io[1];
  io->sys.priv = (void *) priv;
  io->sys.xmm = xmm;

  /* Open file */
  if(( priv->stream = fopen( url, smode )) == NULL )
  {
	free( io );
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(FILE) Unable to open file" );
	return NULL;
  }

  return io;
}

/*
 * Close
 */

static int file_Close( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;
  int ret;

  ret = ( fclose( priv->stream ) == 0 ) ? XMM_RET_OK : XMM_RET_ERROR;
  free( io );

  return ret;
}

/*
 * Read
 */

static int file_Read( XMM_PluginIO *io, void *ptr, int size, int nmemb )
{
  struct priv_t	*priv = io->sys.priv;
  int ret;

  if(( size == 0 ) || ( nmemb == 0 ))	return 0;

  ret = fread( ptr, size, nmemb, priv->stream );
  if(( ret == 0 ) && !feof( priv->stream ))	return XMM_RET_ERROR;

  return ret;
}

/*
 * Write
 */

static int file_Write( XMM_PluginIO *io, void *ptr, int size, int nmemb )
{
  struct priv_t	*priv = io->sys.priv;
  int ret;

  ret = fwrite( ptr, size, nmemb, priv->stream );
  if( ret == 0 )	return XMM_RET_ERROR;

  return ret;
}

/*
 * Tell
 */

static long file_Tell( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;
  long ret;

  ret = ftell( priv->stream );
  if( ret == -1 )	return XMM_RET_ERROR;

  return ret;
}

/*
 * Seek
 */ 

static int file_Seek( XMM_PluginIO *io, long offset, int whence )
{
  struct priv_t	*priv = io->sys.priv;

  if( fseek( priv->stream, offset, whence ) == -1 )	return XMM_RET_NOTSUPPORTED;

  return XMM_RET_OK;
}

/*
 * End of file check
 */

static int file_Eof( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;

  return feof( priv->stream );
}

/*
 * File size
 */

static long file_Size( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;
  long pos, size;

  pos = ftell( priv->stream );

  fseek( priv->stream, 0, SEEK_END );
  size = ftell( priv->stream );

  fseek( priv->stream, pos, SEEK_SET );

  return size;
}

/*
 * Plugin data
 */

XMM_PluginIO	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_IO,
				0,
				XMM_VERSION_NUM,
				"",
				"FILE",
				"I/O: FILE",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				file_Open, file_Close,
				file_Read, file_Write,
				file_Tell, file_Seek,
				file_Eof, file_Size };
