/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * http.c
 * HTTP I/O
 *
 * TODO:
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <libxmm/lpio.h>
#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/error.h>
#include <libxmm/event.h>
#include <libxmm/util/utils.h>

/*
 * Types
 */

struct priv_t
{
    FILE			*stream;
    int				filesize;
    long			filepos;
};

/*
 * Global data
 */

extern XMM_PluginIO	plugin_info;

/* Should these be global or private to streams ? */
static char		*proxy_url = "www-cache.rz.uni-karlsruhe.de:3128";
static int		proxy_enable = 0;

static int		proxy_auth = 0;
static char		*proxy_user = "";
static char		*proxy_passwd = "";

/*
 * Prototypes
 */

static FILE *http_open( void *xmm, const char *url, int *filesize );

/*
 * Open HTTP stream
 */

static XMM_PluginIO *http_Open( void *xmm, const char *url, int mode )
{
  XMM_PluginIO	*io;
  struct priv_t *priv;

  /* Check filename */
  if( strncmp( url, "http://", 7 ))	return NULL;
  else	url += 7;

  /* Check mode */
  if( mode != XMM_IO_READ )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Wrong mode ( %i )", mode );
	return NULL;
  }

  /* Initialize plugin */
  if(( io = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginIO ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(HTTP) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &io[1];
  io->sys.priv = (void *) priv;
  io->sys.xmm = xmm;

  if(( priv->stream = http_open( xmm, url, &priv->filesize )) == NULL )
  {
	free( io );
	return NULL;
  }

  return io;
}

/*
 * Close
 */

static int http_Close( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;
  int ret;

  ret = ( fclose( priv->stream ) == 0 ) ? XMM_RET_OK : XMM_RET_ERROR;
  free( io );

  return ret;
}

/*
 * Read
 */

static int http_Read( XMM_PluginIO *io, void *ptr, int size, int nmemb )
{
  struct priv_t	*priv = io->sys.priv;
  int ret;

  if(( size == 0 ) || ( nmemb == 0 ))	return 0;

  ret = fread( ptr, size, nmemb, priv->stream );
  if( ret == 0 )	return XMM_RET_ERROR;

  priv->filepos += ret;

  return ret;
}

/*
 * Write
 */

static int http_Write( XMM_PluginIO *io, void *ptr, int size, int nmemb )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Tell
 */

static long http_Tell( XMM_PluginIO *io )
{
  return ((struct priv_t *)io->sys.priv)->filepos;
}

/*
 * Seek
 */

static int http_Seek( XMM_PluginIO *io, long offset, int whence )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * End of file check
 */

static int http_Eof( XMM_PluginIO *io )
{
  return feof(((struct priv_t *)io->sys.priv)->stream );
}

/*
 * File size
 */

static long http_Size( XMM_PluginIO *io )
{
  struct priv_t	*priv = io->sys.priv;

  return priv->filesize;
}

/*
 * Plugin data
 */

XMM_PluginIO	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_IO,
				0,
				XMM_VERSION_NUM,
				"",
				"HTTP",
				"IO: HTTP",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				http_Open, http_Close,
				http_Read, http_Write,
				http_Tell, http_Seek,
				http_Eof, http_Size };

/*
 * Private ( HTTP ) stuff
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>

/*
 * Definitions
 */

#define	PREFIX_HTTP	"http://"
#define	SIZE_URL	1000
#define	SIZE_CMD	1100
#define	MAX_RELOCS	5

/*
 * Split url into into hostname, ip and port
 */

static char *SplitURL( const char *url, char *name, unsigned long *ip, unsigned int *port )
{
  struct hostent *host;
  struct in_addr addr;
  int iip = 1;
  char *ptr, *hname;

  /* Skip `http://` */
  if( !strncmp( url, PREFIX_HTTP, 7 ))	url += strlen( PREFIX_HTTP );

  /* Check if ip or name */
  for( ptr = (char *)url; *ptr && ( *ptr != ':' ) && ( *ptr != '/'); ptr++ )
	if(( *ptr != '.' ) && (( *ptr < '0' ) || ( *ptr > '9' )))	iip = 0;

  /* Get host name / IP */
  if(( hname = malloc( (int)ptr - (int)url + 1 )) == NULL )
  {
	hname = NULL;
	return NULL;
  }
  hname[ptr - url] = '\0';
  strncpy( hname, url, ptr - url );

  if( name )	strcpy( name, hname );

  /* Get IP */
  if( !iip )
  {
	if( !( host = gethostbyname( hname )))	return NULL;
	*ip = ((struct in_addr *)host->h_addr)->s_addr;
  }
  else
  {
	if( !inet_aton( hname, &addr ))	return NULL;
	*ip = addr.s_addr;
  }

  free( hname );

  /* Get port */
  if(( *ptr == '\0' ) || ( *ptr == '/' ))
  {
	*port = 80;
	return ptr;
  }

  *port = atoi( ++ptr );
  while( *ptr && ( *ptr != '/' ))	ptr++;

  return ptr;		/* return path */
}

/*
 * Write string
 */

static int WriteString( int fd, char *string )
{
  int written, todo = strlen( string );

  while( todo )
  {
	written = write( fd, string, todo );

	if((( written < 0 ) && ( errno != EINTR )) || ( written == 0 ))
		return 0;

	string += written;
	todo -= written;
  }

  return 1;
}

/*
 * Read string
 */

static int ReadString( char *string, int size, FILE *stream )
{
  char *ptr;

  do
  {
    ptr = fgets( string, size, stream );
  } while(( ptr == NULL ) && ( errno == EINTR ));

  if( ptr == NULL )	return 0;

  return 1;
}

#define BASE64_LENGTH(len) (4 * (((len) + 2) / 3))

/*
 * Encode base64
 */

static void base64_encode( char *string, char *dest )
{
  int size = strlen( string ), i;
  char basis[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

  for( i = 0; i < size; i++, string += 3 )
  {
	*dest++ = basis[ string[0] >> 2 ];
	*dest++ = basis[(( string[0] & 0x3 ) << 4 ) | (( string[1] & 0xF0 ) >> 4 )];
	if( !string[1] )	break;
	*dest++ = basis[(( string[1] & 0xF ) << 2 ) | (( string[2] & 0xC0 ) >> 6 )];
	if( !string[2] )	break;
	*dest++ = basis[string[2] & 0x3F];
  }

  for( i = 0; i < ( size % 3 ) - 1; i++ )	*dest++ = '=';
  *dest++ = '\0';
}

/*
 * Create authorization string ( basic )
 */

static char *create_auth_basic( char *user, char *passwd )
{
  char *t1, *t2;
  int len = strlen( user ) + strlen( passwd ) + 1;

  t1 = (char *)malloc( len + 1 );
  sprintf( t1, "%s:%s", user, passwd );

  t2 = (char *)malloc( BASE64_LENGTH( len ) + 1 );
  base64_encode( t1, t2 );

  return t2;
}

/*
 * Open http
 */

FILE *http_open( void *xmm, const char *url, int *filesize )
{
  char		turl[SIZE_URL], data[SIZE_CMD], *ptr, temp[256], auth[256] = "";
  char		proxy_name[256], name[256], rem_name[256];
  char		*type_req, *type_proxy = "Proxy", *type_http = "HTTP";
  unsigned long	proxy_ip, ip, rem_ip, tmp_ip;
  unsigned int	proxy_port, port, rem_port;
  int		proxy_use, relocate, nrelocs = 0, sockfd, ret;
  struct	sockaddr_in server;
  FILE		*stream = NULL;
  XMM_Event_Auth	eauth;
  char		http_user[XMM_MAXPATHLEN], http_passwd[XMM_MAXPATHLEN];

  /* Check url */
  if( url == NULL )
  {
  	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Wrong url format: (NULL)" );
	return NULL;
  }

  if( url[0] == '\0' )

  /* Check for proxy */
  if( proxy_enable && ( proxy_url[0] == '\0' ))
  {
	if(( proxy_url = getenv( "http_proxy" )) == NULL )
		if(( proxy_url = getenv( "HTTP_PROXY" )) == NULL )
		{
			xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Unable to get 'http_proxy' or 'HTTP_PROXY' environment variable" );
			return NULL;
		}
  }
  
  if( proxy_enable && proxy_url && proxy_url[0] )
  {
	if( !( SplitURL( proxy_url, proxy_name, &proxy_ip, &proxy_port )))
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Wrong url format: '%s'", proxy_url );
		return NULL;
	}

	proxy_use = 1;
  }
  else	proxy_use = 0;

  /* Get remote host name, ip and port */
  if(!( ptr = SplitURL( url, rem_name, &rem_ip, &rem_port )))
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Wrong url format: '%s'", url );
	return NULL;
  }

  tmp_ip = ntohl( rem_ip );
  xmm_logging( 2, "HTTP! ==> Getting: %s%s ( %i.%i.%i.%i:%i )\n\n", rem_name, ptr, ( tmp_ip >> 24 ) & 0xFF, ( tmp_ip >> 16 ) & 0xFF, ( tmp_ip >> 8 ) & 0xFF, tmp_ip & 0xFF, rem_port );

  /*
   * Main relocate loop
   */

  strncpy( turl, url, SIZE_URL - 1 );
  turl[SIZE_URL - 1] = '\0';

  for( relocate = 2; ( relocate == 2 ) && ( nrelocs < MAX_RELOCS ); nrelocs++ )
  {
	/*
	 * Get url and create request string
	 */

	strcpy( data, "GET ");

	if( proxy_use )
	{
	    if( strncmp( url, PREFIX_HTTP, 7 ))	strcat( data, PREFIX_HTTP );
	    port = proxy_port;
	    ip = proxy_ip;
	    strcat( data, turl );
	    strcpy( name, proxy_name );
	    type_req = type_proxy;
	}
	else
	{
	    if(!( ptr = SplitURL( turl, name, &ip, &port )))
	    {
	    	    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Wrong url format: '%s'", turl );
		    return NULL;
	    }

	    strcat( data, ptr );
	    type_req = type_http;
	}

	sprintf( temp, " HTTP/1.0\r\nUser-Agent: XMMP/%s\r\n", XMM_VERSION_STRING );
	strcat( data, temp );

	sprintf( temp, "Host: %s:%d\r\n", rem_name, rem_port );
	strcat( data, temp );

	if( proxy_auth && proxy_user && proxy_passwd )
	{
		sprintf( temp, "Proxy-Authorization: Basic %s\r\n", create_auth_basic( proxy_user, proxy_passwd ));
		strcat( data, temp );
	}

	if( auth[0] > ' ' )
	{
		xmm_logging( 2, "HTTP! authorization required.\n" );

		eauth.type = XMM_EVENT_AUTH;
		eauth.url = NULL;
		eauth.user = http_user;
		eauth.passwd = http_passwd;
		eauth.maxlen = XMM_MAXPATHLEN;

		ret = xmm_PushEvent( xmm, (XMM_Event *)&eauth, 0 );
		if( ret < 0 )
		{
			xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Unable to get authorization from user" );
			return NULL;
		}

		if( !strncasecmp( auth, "Basic", 5 ))
		{
			sprintf( temp, "Authorization: Basic %s\r\n", create_auth_basic( http_user, http_passwd ));
			strcat( data, temp );
		}
		else
		{
			xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Unsupported Authorization type: %s", auth );
			return NULL;
		}

		auth[0] = ' ';
	}

	/* Create socket */
	if(( sockfd = socket( PF_INET, SOCK_STREAM, 6 )) < 0 )	/* 6 = TCP */
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Cannot create socket" );
		free( name );
		return NULL;
	}

	/* Connect */
	server.sin_family = AF_INET;
	server.sin_port = htons( port );
	server.sin_addr.s_addr = ip;

	tmp_ip = ntohl( ip );
	xmm_logging( 2, "HTTP! Connecting to %s [ %li.%li.%li.%li:%i ]... ", name, ( tmp_ip >> 24 ) & 0xFF, ( tmp_ip >> 16 ) & 0xFF, ( tmp_ip >> 8 ) & 0xFF, tmp_ip & 0xFF, port );

	if( connect( sockfd, (struct sockaddr *)&server, sizeof( struct sockaddr )))
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Cannot connect to socket: %s", strerror( errno ));
		return NULL;
	}

	xmm_logging( 2, "connected\n" );

	/* Finish request string */
	strcat( data, "\r\n" );
	xmm_logging( 4, "HTTP! ---request begin---\n%sHTTP! ---request end---\n", data );

	/* send request */
	if( !WriteString( sockfd, data ))
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Write failed" );
		return NULL;
	}

	xmm_logging( 2, "HTTP! %s request sent, waiting for response... ", type_req );

	/* Open socket file */
	if(( stream = fdopen( sockfd, "rb" )) == NULL )
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Failed to fdopen socket: %s", strerror( errno ));
		return NULL;
	}

	relocate = 0;

	/* Get response to HTTP request */
	if( !ReadString( data, SIZE_CMD - 1, stream ))
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Read failed" );
		return NULL;
	}

	if(( ptr = strchr( data, ' ' )))
	{
	    if( !strncmp( &ptr[1], "200", 3 ))		/* OK */
	    {
		xmm_logging( 2, "200 OK\n" );
		if( *auth > ' ' )	*auth = ' ';	/* No more relocation *1 */
	    }
	    else if( !strncmp( &ptr[1], "3", 1 ))
	    {
		relocate = 1;
		xmm_logging( 2, "%c%c%c Redirection", ptr[1], ptr[2], ptr[3] ); 
		if( !strncmp( &ptr[1], "300", 3 ))
		    xmm_logging( 2, "( MULTIPLE CHOICES )\n" );
		if( !strncmp( &ptr[1], "301", 3 ))
		    xmm_logging( 2, "( MOVED_PERMANENTLY )\n" );
		if( !strncmp( &ptr[1], "302", 3 ))
		    xmm_logging( 2, "( MOVED_TEMPORARILY )\n" );
	    }
	    else if( !strncmp( &ptr[1], "4", 1 ))
	    {
		if( !strncmp( &ptr[1], "401", 3 ))
		{
		    xmm_logging( 2, "401 UNAUTHORIZED\n" );
		    if( auth[0] == ' ' )
		    {
		    	    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Authorization failed: Check if User and Password correct" );
			    return NULL;
		    }

		    auth[0] = ' ';
		}
		else
		{
		    if( !strncmp( &ptr[1], "400", 3 ))
			    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) 400 BAD REQUEST" );
		    else if( !strncmp( &ptr[1], "403", 3 ))
			    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) 403 FORBIDDEN" );
		    else if( !strncmp( &ptr[1], "404", 3 ))
			    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) 404 NOT FOUND" );
		    else    xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) %c%c%c UKNOWN Client Error", ptr[1], ptr[2], ptr[3] );

		    return NULL;
		}
	    }
	    else if( !strncmp( &ptr[1], "5", 1 ))	/* ERROR */
	    {
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Server error %s", &ptr[1] );
		return NULL;
	    }
	    else					/* Unkown */
	    {
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Unknown http answer: %s", data );
		return NULL;
	    }
	}
	else
	{
		xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) What happened ? The author did not think of this." );
		return NULL;
	}

	/* Get file information */
	do
	{
	    if( !ReadString( data, SIZE_CMD - 1, stream ))
	    {
	    	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Read failed" );
		return NULL;
	    }

	    if( data[strlen(data) - 1 ] == 10 )	data[strlen(data) - 1 ] = '\0';
	    
	    if( !strncmp( data, "Location", 8 ))
	    {
		xmm_logging( 4, "HTTP! Location: %s\n", &data[10] );
		strcpy( turl, &data[10] );
		relocate++;
	    }
	    else if( !strncmp( data, "Content-Length", 14 ))
	    {
		*filesize = atoi( &data[16] );
		xmm_logging( 4, "HTTP! Length: %s\n", &data[16] );
	    }
	    else if( !strncmp( data, "Content-Type", 12 ))
	    {
		xmm_logging( 4, "HTTP! Type: %s\n", &data[14] );
	    }
	    else if( !strncmp( data, "Last-Modified", 13 ))
	    {
		xmm_logging( 4, "HTTP! Last modified: %s\n", &data[15] );
	    }
	    else if( !strncmp( data, "Server", 6 ))
	    {
		xmm_logging( 4, "HTTP! Server: %s\n", &data[8] );
	    }
	    else if( !strncmp( data, "Accept-Ranges", 13 ))
	    {
		xmm_logging( 4, "HTTP! Accept-Ranges: %s\n", &data[15] );
	    }
	    else if( !strncmp( data, "Date", 4 ))
	    {
		xmm_logging( 4, "HTTP! Date: %s\n", &data[6] );
	    }
	    else if( !strncmp( data, "WWW-Authenticate", 16 ))
	    {
		xmm_logging( 2, "HTTP! WWW-Authenticate: %s\n", &data[18] );
		if( auth[0] == ' ' )	strcpy( auth, &data[18] );
	    }
	    else if(( data[0] != '\r' ) && ( data[0] != 'n' ))
	    {
		xmm_logging( 2, "HTTP! Unkown keyword: %s\n", data );
	    }
	} while(( data[0] != '\r' ) && ( data[0] != 'n' ));

	if( relocate == 2 )	xmm_logging( 2, "HTTP! New Location: %s\n", turl );

	if( auth[0] > ' ' )	/* relocate to add Authorization ( *1 needed ) */
	{
		relocate = 2;
		continue;
	}

	xmm_logging( 1, "\n" );
  }

  if( relocate )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(HTTP) Too many relocs ( %i ). Giving up...", nrelocs );
	return NULL;
  }

  return stream;
}
