/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * alsa.c
 * ALSA sound output
 *
 * TODO:
 *  - Flush() is not implemented, as it did not work. Should be fixed soon.
 */

#include <stdlib.h>
#include <sys/asoundlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpsound.h>
#include <libxmm/lpfiltera.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Definitions
 */

/* Used to set buffer num and size */
#define	FRAG_SIZE	8192
#define	FRAG_COUNT	16


/* Mixer group */
#define	GROUP_NAME	"PCM"
#define	GROUP_INDEX	0

/*
 * Global data
 */

			/* global sound driver data */
static snd_pcm_t	*sound_handle = NULL;

			/* Sound device */
static char		SoundDev[] = "0:0";
static int		card = 0, device = 0;

			/* Buffer: Size, Bytes / Second and Delay */
static long		bufferBPS = 1;
static long		bufferSize = 0;
static long		bufferSSize = 0;
static double		bufferDelay = 0;

extern XMM_PluginSound	plugin_info;

/*
 * Prototypes
 */

static int alsa_GetVolume( XMM_PluginSound *sound, uint32_t *volume );
static int alsa_SetVolume( XMM_PluginSound *sound, uint32_t volume );
static int alsa_Check( XMM_PluginSound *sound, XMM_SoundFormat *sformat );

/*
 * Initialize
 */
static XMM_PluginSound *alsa_Init( void *xmm )
{
  XMM_PluginSound *sound;

  if(( sound = xmm_memdup( &plugin_info, sizeof( XMM_PluginSound ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(ALSA) Unable to duplicate plugin_info" );
	return NULL;
  }

  sound->sys.xmm = xmm;

  return sound;
}

/*
 * Close
 */
static void alsa_Close( XMM_PluginSound *sound )
{
  sound->Flush( sound );

  /* Stop audio chain */
  xmm_FilterAChainStop( sound->sys.xmm );

  snd_pcm_close( sound_handle );
  free( sound );
}

/*
 * Control function
 */
static int alsa_Control( XMM_PluginSound *sound, uint32_t cmd, uint32_t arg, void *data )
{
  snd_pcm_channel_status_t status;

  memset( &status, 0, sizeof( status ));
  if( snd_pcm_channel_status( sound_handle, &status ) < 0 )
  {
	xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Could not get channel status" );
	return XMM_CTLRET_ERROR;
  }       

  switch( cmd )
  {
    case XMM_CTLQUERY_SFORMAT:
		return alsa_Check( sound, (XMM_SoundFormat *) data );

    case XMM_CTLGET_DELAY:
		*((double *)data) = (double) status.count / bufferBPS;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = (uint32_t) status.count + status.free;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = (uint32_t) status.free;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_VOLUME:
		alsa_GetVolume( sound, (uint32_t *) data );
		return XMM_CTLRET_ARG;		/* Result in arg */
  
    case XMM_CTLSET_VOLUME:
		alsa_SetVolume( sound, arg );
		return XMM_CTLRET_TRUE;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_SOUND )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No SOUND command */
}

/*
 * Start output
 */
static int alsa_Start( XMM_PluginSound *sound, XMM_SoundFormat *sformat, int fsize, int fcount )
{
  snd_pcm_channel_params_t	params;
  snd_pcm_channel_setup_t	setup;
  snd_pcm_channel_info_t	ci;
  int				samprate = sformat->samprate;
  int				channels = sformat->channels;
  int				format = sformat->format;
  int				err, i, xmm_fmt_idx;
  char				*cardname;
  int				afmts[6] =
    {
	SND_PCM_SFMT_U8, SND_PCM_SFMT_S16_LE, SND_PCM_SFMT_S16_BE,
	SND_PCM_SFMT_S8, SND_PCM_SFMT_U16_LE, SND_PCM_SFMT_U16_BE
    };
  struct format_conv
	{
	int	xmm;
	int	alt[6];
  } format_table[] =
    {
	{ XMM_SOUND_FMT_U8	,{ 0, 3, 1, 2, 4, 5 }},
	{ XMM_SOUND_FMT_S16LE	,{ 1, 2, 0, 3, 4, 5 }},
	{ XMM_SOUND_FMT_S16BE	,{ 2, 1, 0, 3, 5, 4 }},
	{ XMM_SOUND_FMT_S8	,{ 3, 0, 1, 2, 4, 5 }},
	{ XMM_SOUND_FMT_U16LE	,{ 4, 5, 0, 3, 1, 2 }},
	{ XMM_SOUND_FMT_U16BE	,{ 5, 4, 0, 3, 2, 1 }}
    };

  /* Check Sound Device string */
  if(( SoundDev[1] != ':' ) && SoundDev[3] )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Wrong sound device format. Check config file ( 'card:device' )" );

  card = SoundDev[0] - 48;
  device = SoundDev[2] - 48;

  if((err = snd_card_get_longname( card, &cardname )))
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Error optainig card name %i,%i: %s", card, device, snd_strerror(err));

  xmm_logging( 2, "ALSA! Using soundcard '%s'\n", cardname );
  free( cardname );

  /* Open PCM output */
  if((err = snd_pcm_open( &sound_handle, card, device, SND_PCM_OPEN_PLAYBACK )))
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Error opening audio device %i,%i: %s", card, device, snd_strerror(err));

  /*
   * Check format
   */

  /* Get channel information */
  ci.channel = SND_PCM_CHANNEL_PLAYBACK;
  snd_pcm_channel_info( sound_handle, &ci );

  /* Check sample rate */
  if( samprate > ci.max_rate )	samprate = ci.max_rate;
  if( samprate < ci.min_rate )	samprate = ci.min_rate;

  /* Check channels */
  if( channels > ci.max_voices )	channels = ci.max_voices;
  if( channels < ci.min_voices )	channels = ci.min_voices;

  /* Check format */
  for( xmm_fmt_idx = 0; xmm_fmt_idx < 6; xmm_fmt_idx++ )
	if( format == format_table[xmm_fmt_idx].xmm )
		break;

  if( xmm_fmt_idx == 6 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Unkown XMM format" );

  /* Search matching format */
  for( i = 0; i < 6; i++ )
	if(( 1 << afmts[format_table[xmm_fmt_idx].alt[i]] ) & ci.formats )
		break;

  if( i == 6 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) No matching format found" );

  /* Set new ( maybe old ) format */
  xmm_fmt_idx = format_table[xmm_fmt_idx].alt[i];
  format = format_table[xmm_fmt_idx].xmm;

  /* Initialize ALSA */
  memset(&params, 0, sizeof(params));
  params.channel = SND_PCM_CHANNEL_PLAYBACK;
  params.mode = SND_PCM_MODE_BLOCK;
  memset( &params.format, 0, sizeof(params.format));
  params.format.interleave = 1;
  params.format.format = afmts[ xmm_fmt_idx ];
  params.format.rate = samprate;
  params.format.voices = channels;
  params.start_mode = SND_PCM_START_FULL;
  params.stop_mode = SND_PCM_STOP_STOP;
  params.buf.block.frag_size = FRAG_SIZE;
  if( fsize > 0 )	params.buf.block.frag_size = fsize;
  params.buf.block.frags_max = FRAG_COUNT;
  if( fcount > 0 )	params.buf.block.frags_max = fcount;
  params.buf.block.frags_min = 1;

  snd_pcm_channel_flush( sound_handle, SND_PCM_CHANNEL_PLAYBACK );

  if((err = snd_pcm_channel_params( sound_handle, &params )))
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Error setting parameters: %s", snd_strerror(err));

  if((err = snd_pcm_channel_prepare( sound_handle, SND_PCM_CHANNEL_PLAYBACK )))
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Error preparing playback: %s", snd_strerror(err));

  memset(&setup, 0, sizeof(setup));
  setup.mode = SND_PCM_MODE_BLOCK;
  setup.channel = SND_PCM_CHANNEL_PLAYBACK;

  if((err = snd_pcm_channel_setup( sound_handle, &setup )))
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Setup error: %s", snd_strerror(err));

  xmm_logging( 2, "ALSA! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
			samprate, channels, format, setup.buf.block.frag_size );

  /* Set source format */
  xmm_FilterAChainInput( sound->sys.xmm, sformat->samprate, sformat->channels, sformat->format );

  /* Set dest format */
  bufferSize = xmm_FilterAChainOutput( sound->sys.xmm, samprate, channels, format, setup.buf.block.frag_size );

  /* Initialize data */
  bufferBPS = (long)( samprate * channels * ( format & XMM_SOUND_MASK_SIZE ) / 8 );
  bufferDelay = (double) setup.buf.block.frag_size / bufferBPS;
  bufferSSize = channels * ( format & XMM_SOUND_MASK_SIZE ) / 8;

  /* return buffer size ( = number of samples ) */
  return bufferSize;
}

/*
 * Write data
 */
static int alsa_Write( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  snd_pcm_channel_status_t	status;
  int				proc, written = 0, conv, done, ret;
  uint8_t			*ptr;

  /*
   * samples > bufferSize: We have to split data for xmm_FilterAChain()
   */

  while( samples > 0 )
  {
	proc = samples;
	if( proc > bufferSize )	proc = bufferSize;

	conv = proc;
	ptr = data;

	if( xmm_FilterAChainActive( sound->sys.xmm ))
	{
	    if(( ret = xmm_FilterAChain( sound->sys.xmm, data, proc, &ptr, &conv )) != XMM_RET_OK )
		return ret;
	}

	done = snd_pcm_write( sound_handle, ptr, conv * bufferSSize );
	if( done < 0 )	xmm_logging( 1, "ALSA! Error '%s'\n", snd_strerror(done));

	memset( &status, 0, sizeof( status ));
	if( snd_pcm_channel_status( sound_handle, &status ) < 0 )
	{
	    xmm_logging( 1, "ALSA! Could not get channel status\n" );
	    written += done;
	    break;
	}       

	if( status.underrun )
	{
	    xmm_logging( 1, "ALSA! Underrun. Ressetting channel...\n" );
	    snd_pcm_channel_flush( sound_handle, SND_PCM_CHANNEL_PLAYBACK );
	    snd_pcm_playback_prepare( sound_handle );
/* * *
	    done = snd_pcm_write( sound_handle, ptr, conv );

	    if( snd_pcm_channel_status( sound_handle, &status ) < 0 )
	    {
		xmm_logging( 1, "ALSA! Could not get channel status\n" );
		written += done;
		break;
	    }       

	    if( status.underrun )
	    {
		xmm_logging( 1, "ALSA! Still underrun. Giving up...\n" );
		written += done;
		break;
	    }               
* * */
	}

	written += done;
	data += ( proc * bufferSSize );
	samples -= proc;
  }

  if( written == 0 )	return XMM_RET_ERROR;

  return written / bufferSSize;
}

/*
 * Flush data
 */
static void alsa_Flush( XMM_PluginSound *sound )
{
/*
 * This causes a 'bad file descriptor' error in snd_pcm_write()
 */

//  snd_pcm_playback_drain( sound_handle );
}

/*
 * Plugin info
 */

XMM_PluginSound	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_SOUND,
				0,
				XMM_VERSION_NUM,
				"",
				"ALSA",
				"Sound: ALSA",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				alsa_Init, alsa_Close, alsa_Control,
				alsa_Start, alsa_Write, alsa_Flush };

/*
 * Internal code
 */

/*
 * Check output parameters
 */
static int alsa_Check( XMM_PluginSound *sound, XMM_SoundFormat *sformat )
{
  snd_pcm_channel_info_t	ci;
  int				err, ret = XMM_CTLRET_TRUE, i;
  struct format_conv
  {
	int	xmm;
	int	alsa;
  } format_table[] =
    {
	{ XMM_SOUND_FMT_U8	, SND_PCM_FMT_U8	},
	{ XMM_SOUND_FMT_S16LE	, SND_PCM_FMT_S16_LE	},
	{ XMM_SOUND_FMT_S16BE	, SND_PCM_FMT_S16_BE	},
	{ XMM_SOUND_FMT_S8	, SND_PCM_FMT_S8	},
	{ XMM_SOUND_FMT_U16LE	, SND_PCM_FMT_U16_LE	},
	{ XMM_SOUND_FMT_U16BE	, SND_PCM_FMT_U16_BE	}
    };

  if(( SoundDev[1] != ':' ) && SoundDev[3] )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Wrong sound device format. Check config file ( 'card:device' )" );

  card = SoundDev[0] - 48;
  device = SoundDev[2] - 48;

  /* Open */
  if((err = snd_pcm_open( &sound_handle, card, device, SND_PCM_OPEN_PLAYBACK )))
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Error opening audio device %i,%i: %s", card, device, snd_strerror(err));

  /* Get channel information */
  ci.channel = SND_PCM_CHANNEL_PLAYBACK;
  snd_pcm_channel_info( sound_handle, &ci );

  /* Check sample rate */
  if(( sformat->samprate > ci.max_rate ) || ( sformat->samprate < ci.min_rate ))
	ret = XMM_CTLRET_FALSE;

  /* Check channels */
  if(( sformat->channels > ci.max_voices ) || ( sformat->channels < ci.min_voices ))
	ret = XMM_CTLRET_FALSE;

  /* Check format */
  for( i = 0; i < 6; i++ )
	if( sformat->format == format_table[i].xmm )
		break;

  if( i == 6 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Unkown XMM format" );

  if(( ci.formats & format_table[i].alsa ) == 0 )	ret = XMM_CTLRET_FALSE;

  /* Close */
  snd_pcm_close( sound_handle );

  return ret;
}

/*
 * Mixer
 */

static int get_percent( int val, int min, int max )
{
  if( max == min )		return 0;
  return (( val - min ) * 100 ) / ( max - min );
}

static int get_range( int val, int min, int max )
{
  if( max == min )	return 0;
  return ( val * ( max - min ) / 100 ) + min;
}

/*
 * Get volume
 */
static int alsa_GetVolume( XMM_PluginSound *sound, uint32_t *volume )
{
  int			err, left, right;
  snd_mixer_t		*handle;
  snd_mixer_gid_t	gid;
  snd_mixer_group_t	group;

  /* Be verbose */
  xmm_logging( 2, "ALSA! Mixer: Group '%s' (%i)\n", GROUP_NAME, GROUP_INDEX );

  /* Open Mixer */
  if(( err = snd_mixer_open( &handle, card, device )) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Mixer %i:%i open error: %s", card, device, snd_strerror(err));

  /* Fill group id struct */
  memset( &gid, 0, sizeof( snd_mixer_gid_t ));
  strncpy( gid.name, GROUP_NAME, sizeof( gid.name ) - 1 );
  gid.index = GROUP_INDEX;

  /* Fill group struct */
  memset( &group, 0, sizeof( snd_mixer_group_t ));
  memcpy( &group.gid, &gid, sizeof( snd_mixer_gid_t ));	/* Set Group ID */

  /* Read Group info */
  if(( err = snd_mixer_group_read( handle, &group )) < 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Mixer %i:%i group read error: %s", card, device, snd_strerror(err));
  }

  /* Output informations */
  if( group.channels == 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Group '%s(%i)' has no channels", GROUP_NAME, GROUP_INDEX );
  }

  if( group.channels == SND_MIXER_CHN_MASK_MONO )
  {
	left = get_percent( group.volume.names.front_left, group.min, group.max );
	right = left;
  }
  else
  {
	left = get_percent( group.volume.names.front_left, group.min, group.max );
	right = get_percent( group.volume.names.front_right, group.min, group.max );
  }

  snd_mixer_close( handle );

  *volume = left | ( right << 16 );
  return 0;
}

/*
 * Set Volume
 */
static int alsa_SetVolume( XMM_PluginSound *sound, uint32_t volume )
{
  int			err, ch;
  snd_mixer_t		*handle;
  snd_mixer_gid_t	gid;
  snd_mixer_group_t	group;

  /* Be verbose */
  xmm_logging( 2, "ALSA! Mixer: Group '%s' (%i)\n", GROUP_NAME, GROUP_INDEX );

  /* Open mixer */
  if(( err = snd_mixer_open( &handle, card, device )) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Mixer %i:%i open error: %s", card, device, snd_strerror( err ));

  /* Fill group id struct */
  memset( &gid, 0, sizeof( snd_mixer_gid_t ));
  strncpy( gid.name, GROUP_NAME, sizeof( gid.name ) - 1 );
  gid.index = GROUP_INDEX;

  /* Fill group struct */
  memset( &group, 0, sizeof( snd_mixer_group_t ));
  memcpy( &group.gid, &gid, sizeof( snd_mixer_gid_t ));	/* Set Group ID */

  /* Read Group info */
  if(( err = snd_mixer_group_read( handle, &group )) < 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Mixer %i:%i group read error: %s", card, device, snd_strerror( err ));
  }

  /* Fill group struct with new settings */
  if( group.caps & SND_MIXER_GRPCAP_JOINTLY_VOLUME )
  {
	for( ch = 0; ch < SND_MIXER_CHN_LAST; ch++ )
	    if( group.channels & ( 1 << ch ))	group.volume.values[ch] = get_range( volume & 0x0000FFFF, group.min, group.max );
  }
  else
  {
	group.volume.values[SND_MIXER_CHN_FRONT_LEFT] = get_range( volume & 0x0000FFFF, group.min, group.max );
	group.volume.values[SND_MIXER_CHN_FRONT_RIGHT] = get_range(( volume & 0xFFFF0000 ) >> 16, group.min, group.max );
  }

  /* Write group */
  if(( err = snd_mixer_group_write( handle, &group )) < 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(ALSA) Mixer %i:%i group write error: %s", card, device, snd_strerror( err ));
  }

  /* Close mixer */
  snd_mixer_close( handle );

  return 0;
}


#if 0

static int mixer_info( void )
{
  int			err;
  snd_mixer_t		*handle;
  snd_mixer_info_t	info;
	
  if(( err = snd_mixer_open( &handle, card, device )) < 0 )
  {
	xmm_logging( 3, "ALSA! Mixer %i:%i open error: %s\n", card, device, snd_strerror( err ));
	return -1;
  }

  if(( err = snd_mixer_info( handle, &info )) < 0 )
  {
	xmm_logging( 3, "ALSA! Mixer %i:%i info error: %s\n", card, device, snd_strerror( err ));
	return -1;
  }

  xmm_logging( 3, "ALSA! ALSA Mixer ( ID: %s ) %s\n", info.id, info.name );
  xmm_logging( 3, "ALSA! \tType (Soundcard) : 0x%x\n", info.type );
  xmm_logging( 3, "ALSA! \tAttribute        : 0x%x\n", info.attrib );
  xmm_logging( 3, "ALSA! \tElements         : %i\n", info.elements );
  xmm_logging( 3, "ALSA! \tGroups           : %i\n", info.groups );

  snd_mixer_close(handle);

  return 1;
}

static void mixer_group_info( snd_mixer_group_t *group )
{
  int ch;

  xmm_logging( 3, "ALSA! Mixer Group Capabilities:" );

  if( group->caps & SND_MIXER_GRPCAP_VOLUME )		xmm_logging( 3, " volume" );
  if( group->caps & SND_MIXER_GRPCAP_MUTE )		xmm_logging( 3, " mute" );
  if( group->caps & SND_MIXER_GRPCAP_JOINTLY_MUTE )	xmm_logging( 3, " jointly-mute" );
  if( group->caps & SND_MIXER_GRPCAP_CAPTURE )		xmm_logging( 3, " capture" );
  if( group->caps & SND_MIXER_GRPCAP_JOINTLY_CAPTURE )	xmm_logging( 3, " jointly-capture" );
  if( group->caps & SND_MIXER_GRPCAP_EXCL_CAPTURE )	xmm_logging( 3, " exclusive-capture" );
  xmm_logging( 3, "\n" );

  if(( group->caps & SND_MIXER_GRPCAP_CAPTURE ) && ( group->caps & SND_MIXER_GRPCAP_EXCL_CAPTURE ))
	    xmm_logging( 3, "ALSA! Mixer Group: Capture exclusive group: %i\n", group->capture_group );

  xmm_logging( 3, "ALSA! Mixer Group: Limits: min = %i, max = %i\n", group->min, group->max );

  xmm_logging( 3, "ALSA! Mixer Group: Channels:\n" );
  if( group->channels == SND_MIXER_CHN_MASK_MONO )
  {
	xmm_logging( 3, "ALSA! Mixer Group: Mono: %i [%i%%] [%s]\n",  group->volume.names.front_left,
					    get_percent( group->volume.names.front_left, group->min, group->max ),
					    group->mute & SND_MIXER_CHN_MASK_MONO ? "mute" : "on" );
  }
  else
  {
	for( ch = 0; ch <= SND_MIXER_CHN_LAST; ch++ )
	{
	    if( group->channels & ( 1 << ch ))
		xmm_logging( 3, "ALSA! Mixer Group: %s: %i [%i%%] [%s] [%s]\n",	snd_mixer_channel_name( ch ),
						group->volume.values[ch],
						get_percent( group->volume.values[ch], group->min, group->max ),
						group->mute & ( 1 << ch ) ? "mute" : "on",
						group->capture & ( 1 << ch ) ? "capture" : "---" );
	}
  }
}

#endif
