/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lame.c
 * MPEG Layer 3 sound ( writer )
 *
 * TODO:
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpsound.h>
#include <libxmm/lpfiltera.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include "lame/lame.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		8192

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;

    lame_global_flags		gf;
    char			*mp3data;
    int				mp3data_size;

    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Global data
 */

extern XMM_PluginSound	plugin_info;

/*
 * Initialize
 */
static XMM_PluginSound *lame_Init( void *xmm )
{
  XMM_PluginSound *sound;
  struct priv_t *priv;

  if(( sound = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginSound ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(LAME) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &sound[1];
  sound->sys.priv = (void *) priv;
  sound->sys.xmm = xmm;

  return sound;
}

/*
 * Close
 */
static void lame_Close( XMM_PluginSound *sound )
{
  struct priv_t *priv = sound->sys.priv;
  int		done;

  /* Stop audio chain */
  xmm_FilterAChainStop( sound->sys.xmm );

  done = lame_encode_finish( &priv->gf, priv->mp3data, priv->mp3data_size );
  if( done > 0 )	priv->pIO->Write( priv->pIO, priv->mp3data, done, 1 );

  priv->pIO->Close( priv->pIO );

  free( sound );
}

/*
 * Control function
 */
static int lame_Control( XMM_PluginSound *sound, uint32_t cmd, uint32_t arg, void *data )
{
  struct priv_t *priv = sound->sys.priv;

  switch( cmd )
  {
    case XMM_CTLQUERY_SFORMAT:
		if(((( XMM_SoundFormat *)data)->format & XMM_SOUND_MASK_SIZE ) != 16 )
		    return XMM_CTLRET_FALSE;

		return XMM_CTLRET_TRUE;

    case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;
  
    case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_SOUND )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No SOUND command */
}

/*
 * Start output
 */
static int lame_Start( XMM_PluginSound *sound, XMM_SoundFormat *sformat, int fsize, int fcount )
{
  struct priv_t	*priv = sound->sys.priv;
  const char	*ofn;
  int		samprate = sformat->samprate, channels = sformat->channels;
  int		format = sformat->format;

  if(( ofn = xmm_AcOutputFilename( sound->sys.xmm, NULL, ".mp3" )) == NULL )
	return XMM_RET_ERROR;

  if(( priv->pIO = xmm_IOOpen( sound->sys.xmm, ofn, XMM_IO_WRITE )) == NULL )
	return XMM_RET_ERROR;

  /* Init LAME stuff */
  lame_init( &priv->gf );

  priv->gf.num_channels = channels;
  priv->gf.in_samplerate = samprate;
  priv->gf.brate = 128;
  priv->gf.mode = 1;
  priv->gf.quality = 2;
  priv->gf.silent = 1;

  lame_init_params( &priv->gf );

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  /* Calculate and allocate mp3 buffer */
  priv->mp3data_size = fsize * 5 / 4 + 7200;
  if(( priv->mp3data = malloc( priv->mp3data_size )) == NULL )
  {
	xmm_SetError( sound->sys.xmm, XMM_ERR_ALLOC, "(LAME) mp3 buffer." );
	return XMM_RET_ERROR;
  }

  if(( format & XMM_SOUND_MASK_SIZE ) != 16 )
  {
	xmm_logging( 2, "LAME! 8 bit not supported. Converting to 16 bit\n" );
	format = XMM_SOUND_FMT_S16LE;
  }

  xmm_logging( 2, "LAME! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
					samprate, channels, format, fsize );

  /* Set source format */
  xmm_FilterAChainInput( sound->sys.xmm, sformat->samprate, sformat->channels, sformat->format );

  /* Set dest format */
  priv->bufferSize = xmm_FilterAChainOutput( sound->sys.xmm, samprate, channels, format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)( samprate * channels * ( format & XMM_SOUND_MASK_SIZE ));
  priv->bufferSSize = ( channels * ( format & XMM_SOUND_MASK_SIZE ) / 8 );

  /* return buffer size ( = number of samples ) */
  return priv->bufferSize;
}

/*
 * Write data
 */
static int lame_Write( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  struct priv_t	*priv = sound->sys.priv;
  int		proc, written = 0, conv, done, ret;
  uint8_t	*ptr;

  /*
   * samples > bufferSize: We have to split data for xmm_FilterAChain()
   */

  while( samples > 0 )
  {
	proc = samples;
	if( proc > priv->bufferSize )	proc = priv->bufferSize;

	conv = proc;
	ptr = data;

	if( xmm_FilterAChainActive( sound->sys.xmm ))
	{
	    if(( ret = xmm_FilterAChain( sound->sys.xmm, data, proc, &ptr, &conv )) != XMM_RET_OK )
		return ret;
	}

	if( priv->gf.num_channels == 1 )
	    done = lame_encode_buffer( &priv->gf, (short *)ptr, (short *)ptr, conv, priv->mp3data, priv->mp3data_size );
	else
  	    done = lame_encode_buffer_interleaved( &priv->gf, (short *)ptr, conv, priv->mp3data, priv->mp3data_size );

	if( done > 0 )	priv->pIO->Write( priv->pIO, priv->mp3data, done, 1 );
	if( done < 0 )	done = 0;

	written += done;
	data += ( proc * priv->bufferSSize );
	samples -= proc;
  }

  if( written == 0 )	return XMM_RET_ERROR;

  return written;
}

/*
 * Flush data
 */
static void lame_Flush( XMM_PluginSound *sound )
{
}

/*
 * Plugin info
 */

XMM_PluginSound	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_SOUND,
				0,
				XMM_VERSION_NUM,
				"",
				"LAME",
				"Sound: MPEG Layer 3",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				lame_Init, lame_Close, lame_Control,
				lame_Start, lame_Write, lame_Flush };
