/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * null.c
 * Dummy sound
 *
 * TODO:
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpsound.h>
#include <libxmm/lpfiltera.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		8192

/*
 * Types
 */

struct priv_t
{
    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Global data
 */

extern XMM_PluginSound	plugin_info;

/*
 * Init
 */

static XMM_PluginSound *null_Init( void *xmm )
{
  XMM_PluginSound *sound;
  struct priv_t *priv;

  if(( sound = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginSound ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(NULL) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &sound[1];
  sound->sys.priv = (void *) priv;
  sound->sys.xmm = xmm;

  return sound;
}

/*
 * Start output
 */

static int null_Start( XMM_PluginSound *sound, XMM_SoundFormat *sformat, int fsize, int fcount )
{
  struct priv_t	*priv = sound->sys.priv;
  int		samprate = sformat->samprate, channels = sformat->channels;
  int		format = sformat->format;

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  /* Initialize data */
  priv->bufferDelay = fsize / (double)( samprate * channels * ( format & XMM_SOUND_MASK_SIZE ));
  priv->bufferSSize = ( channels * ( format & XMM_SOUND_MASK_SIZE ) / 8 );

  xmm_logging( 2, "NULL! Initialized ( Rate = %i Channels = %i format = %x bs = %i bytes )\n", samprate, channels, format, fsize );

  /* Set conversion if needed */
  xmm_FilterAChainInput( sound->sys.xmm, samprate, channels, format );
  priv->bufferSize = fsize / priv->bufferSSize;

  /* return buffer size ( = number of samples ) */
  return priv->bufferSize;
}

/*
 * Close
 */
static void null_Close( XMM_PluginSound *sound )
{
  free( sound );
}

/*
 * Write data
 */
static int null_Write( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  return samples;
}

/*
 * Control function
 */
static int null_Control( XMM_PluginSound *sound, uint32_t cmd, uint32_t arg, void *data )
{
  struct priv_t *priv = sound->sys.priv;

  switch( cmd )
  {
    case XMM_CTLQUERY_SFORMAT:
		return XMM_CTLRET_TRUE;

    case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;
  
    case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_SOUND )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No SOUND command */
}

/*
 * Flush data
 */
static void null_Flush( XMM_PluginSound *sound )
{
}

/*
 * Plugin info
 */
XMM_PluginSound	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_SOUND,
				0,
				XMM_VERSION_NUM,
				"",
				"NULL",
				"Sound: NULL",
				"Copyright (c) 200, 2001 by Arthur Kleer",
				NULL, NULL },
				null_Init, null_Close, null_Control,
				null_Start, null_Write,	null_Flush };
