/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * oss.c
 * OSS sound output
 *
 * TODO:
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/soundcard.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpsound.h>
#include <libxmm/lpfiltera.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Definitions
 */

/* Used to set buffer num and size */
#define FRAG_SPEC	0x0010000D

/*
 * Global data
 */

			/* sound device file descriptor */
static int		fd_dsp = -2;

			/* Sound device */
static char		SoundDev[] = "/dev/dsp";
static char		MixerDev[] = "/dev/mixer";

			/* Buffer: Size and Bytes / Second */
static long		bufferBPS = 1;
static long		bufferSize = 0;
static long		bufferSSize = 0;

extern XMM_PluginSound	plugin_info;

/*
 * Prototypes
 */

static int oss_GetVolume( XMM_PluginSound *sound, uint32_t *volume );
static int oss_SetVolume( XMM_PluginSound *sound, uint32_t volume );
static int oss_Check( XMM_PluginSound *sound, XMM_SoundFormat *sformat );

/*
 * Initialize
 */
static XMM_PluginSound *oss_Init( void *xmm )
{
  XMM_PluginSound *sound;

  /* Check for multiple use */
  if( fd_dsp != -2 )
  {
	xmm_SetError( xmm, XMM_ERR_UNKNOWN, "(OSS) OSS plugin already in use." );
	return NULL;
  }

  if(( sound = xmm_memdup( &plugin_info, sizeof( XMM_PluginSound ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(OSS) Unable to duplicate plugin_info" );
	return NULL;
  }

  fd_dsp = -1;		/* plugin in use */
  sound->sys.xmm = xmm;

  return sound;
}

/*
 * Close
 */
static void oss_Close( XMM_PluginSound *sound )
{
  sound->Flush( sound );

  /* Stop audio chain */
  xmm_FilterAChainStop( sound->sys.xmm );

  if( fd_dsp >= 0 )
  {
	close( fd_dsp );
	fd_dsp = -1;
  }
  free( sound );
}

/*
 * Control function
 */
static int oss_Control( XMM_PluginSound *sound, uint32_t cmd, uint32_t arg, void *data )
{
  audio_buf_info abi;

  switch( cmd )
  {
    case XMM_CTLQUERY_SFORMAT:
		return oss_Check( sound, (XMM_SoundFormat *)data );

    case XMM_CTLGET_DELAY:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((double *)data) = ((double)abi.fragstotal * abi.fragsize - abi.bytes ) / bufferBPS;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_MAX_DELAY:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((double *)data) = (double) abi.fragstotal * abi.fragsize / bufferBPS;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_TOTAL:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((uint32_t *)data) = (uint32_t) abi.fragstotal * abi.fragsize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_FREE:
		ioctl( fd_dsp, SNDCTL_DSP_GETOSPACE, &abi );
		*((uint32_t *)data) = (uint32_t) abi.bytes;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_VOLUME:
		oss_GetVolume( sound, (uint32_t *) data );
		return XMM_CTLRET_ARG;		/* Result in arg */
  
    case XMM_CTLSET_VOLUME:
		oss_SetVolume( sound, arg );
		return XMM_CTLRET_TRUE;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_SOUND )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No SOUND command */
}

/*
 * Start output
 */
static int oss_Start( XMM_PluginSound *sound, XMM_SoundFormat *sformat, int fsize, int fcount )
{
  int tmp, bits, ssize;
  int samprate = sformat->samprate, channels = sformat->channels;
  int format = sformat->format;

  if( fd_dsp >= 0 )
  {
	close( fd_dsp );
	fd_dsp = -1;
  }

  /* Check device access */
  if( access( SoundDev, W_OK ) != 0)
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Accessing audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  /* Open device */
  if(( fd_dsp = open( SoundDev, O_WRONLY )) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Opening audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  /* get buffer size */
  tmp = FRAG_SPEC;

  if( fsize > 0 )
  {
	for( bits = -1; fsize; fsize >>= 1, bits++ );
	tmp = ( tmp & 0xFFFF0000 ) | bits;
  }

  if( fcount != -1 )
  {
	tmp = ( fcount << 16 ) | ( tmp & 0x0000FFFF );
  }

  /* set fragment size and count */
  if( ioctl( fd_dsp, SNDCTL_DSP_SETFRAGMENT, &tmp ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Setting FRAGMENT to %i: %s", FRAG_SPEC, sys_errlist[errno] );

  /* set sample size */
  ssize = format & XMM_SOUND_MASK_SIZE;
  if( ioctl( fd_dsp, SNDCTL_DSP_SAMPLESIZE, &ssize ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Setting DSP to %u bits: %s", (unsigned) tmp, sys_errlist[errno] );

  format = ( format & ~XMM_SOUND_MASK_SIZE ) | ssize;

  /* set channels */
  channels--;
  if( ioctl( fd_dsp, SNDCTL_DSP_STEREO, &channels ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Unable to set DSP to %s mode: %s", channels ? "Stereo":"Mono", sys_errlist[errno] );

  /* set sample rate */
  if( ioctl( fd_dsp, SNDCTL_DSP_SPEED, &samprate ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Unable to set audio sampling rate to %i: %s", samprate, sys_errlist[errno] );

  /* get block size */
  if( ioctl( fd_dsp, SNDCTL_DSP_GETBLKSIZE, &tmp ) == -1 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Optaining DSP's block size: %s", sys_errlist[errno] );

  xmm_logging( 2, "OSS! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
					samprate, channels + 1, format, tmp );

  /* Set source format */
  xmm_FilterAChainInput( sound->sys.xmm, sformat->samprate, sformat->channels, sformat->format );

  /* Set dest format */
  bufferSize = xmm_FilterAChainOutput( sound->sys.xmm, samprate, channels + 1, format, tmp );

  /* Initialize data */
  bufferBPS = (long)( samprate * ( channels + 1 ) * ( format & XMM_SOUND_MASK_SIZE ) / 8 );
  bufferSSize = (( channels + 1 ) * ( format & XMM_SOUND_MASK_SIZE ) / 8 );

  /* return buffer size ( = number of samples ) */
  return bufferSize;
}

/*
 * Write data
 */
static int oss_Write( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  int		proc, written = 0, conv, ret;
  uint8_t	*ptr;

  /*
   * samples > bufferSize: We have to split data for xmm_FilterAChain()
   */

  while( samples > 0 )
  {
	proc = samples;
	if( proc > bufferSize )	proc = bufferSize;

	conv = proc;
	ptr = data;

	if( xmm_FilterAChainActive( sound->sys.xmm ))
	{
	    if(( ret = xmm_FilterAChain( sound->sys.xmm, data, proc, &ptr, &conv )) != XMM_RET_OK )
		return ret;
	}

	written += write( fd_dsp, ptr, conv * bufferSSize );
	data += ( proc * bufferSSize );
	samples -= proc;
  }

  if( written == 0 )	return XMM_RET_ERROR;

  return written / bufferSSize;
}

/*
 * Flush data
 */
static void oss_Flush( XMM_PluginSound *sound )
{
  ioctl( fd_dsp, SNDCTL_DSP_RESET, NULL );
}

/*
 * Plugin info
 */

XMM_PluginSound	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_SOUND,
				0,
				XMM_VERSION_NUM,
				"",
				"OSS",
				"Sound: OSS",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				oss_Init, oss_Close, oss_Control,
				oss_Start, oss_Write, oss_Flush };

/*
 * Internal code
 */

/*
 * Check output parameters
 */
static int oss_Check( XMM_PluginSound *sound, XMM_SoundFormat *sformat )
{
  int tmp, ret = XMM_CTLRET_TRUE, fd;

  if( access( SoundDev, W_OK ) != 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Error accessing audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  if(( fd = open( SoundDev, O_WRONLY )) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Opening audio device <%s>: %s", SoundDev, sys_errlist[errno] );

  /* Check sample rate */
  tmp = sformat->samprate;
  if( ioctl( fd, SNDCTL_DSP_SPEED, &tmp ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Unable to set audio sampling rate to %i: %s", sformat->samprate, sys_errlist[errno] );

  if( tmp != sformat->samprate )	ret = XMM_CTLRET_FALSE;

  /* Check sample size */
  tmp = sformat->format & XMM_SOUND_MASK_SIZE;
  if( ioctl( fd, SNDCTL_DSP_SAMPLESIZE, &tmp ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Setting DSP to %u bits: %s", (unsigned) sformat->format & XMM_SOUND_MASK_SIZE, sys_errlist[errno] );

  if( tmp != ( sformat->format & XMM_SOUND_MASK_SIZE ))	ret = XMM_CTLRET_FALSE;

  /* Check channels */
  tmp = sformat->channels - 1;
  if( ioctl( fd, SNDCTL_DSP_STEREO, &tmp ) < 0 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Unable to set DSP to %s mode: %s", sformat->channels == 1 ? "Mono" : "Stereo", sys_errlist[errno] );

  if( tmp != ( sformat->channels - 1 ))	ret = XMM_CTLRET_FALSE;

  /* close device */
  close( fd );

  /* return */
  return ret;
}

/*
 * Mixer
 */

static int oss_GetVolume( XMM_PluginSound *sound, uint32_t *volume )
{
  int fd, value, supported;

  if(( fd = open( MixerDev , O_RDWR )) == -1 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Mixer: Opening mixer for read: %s", sys_errlist[errno] );

  if( ioctl( fd, SOUND_MIXER_READ_DEVMASK, &supported ) < 0 )
	supported = 0xffff;	/* All supported ??? */

  if(( supported & SOUND_MASK_PCM ) == 0 )
  {
	close( fd );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Mixer: Volume read not supported" );
  }

  if( ioctl( fd, SOUND_MIXER_READ_PCM, &value ) < 0 )
  {
	close( fd );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Mixer: Unable to read volume" );
  }

  *volume = ( value & 0x00FF ) | ((( value & 0xFF00 ) << 8 ));

  close( fd );
  return 0;
}

static int oss_SetVolume( XMM_PluginSound *sound, uint32_t volume )
{
  int fd, value, supported;

  if(( fd = open( MixerDev , O_RDWR )) == -1 )
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Mixer: Opening mixer for write: %s", sys_errlist[errno] );

  if( ioctl( fd, SOUND_MIXER_READ_DEVMASK, &supported ) < 0 )
	supported = 0xffff;	/* All supported ??? */

  if(( supported & SOUND_MASK_PCM ) == 0 )
  {
	close( fd );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Mixer: Volume write not supported" );
  }

  value = (( volume & 0xFFFF0000 ) >> 8 ) | ( volume & 0x0000FFFF );
  if( ioctl( fd, SOUND_MIXER_WRITE_PCM, &value ) < 0 )
  {
	close( fd );
	return xmm_SetError( sound->sys.xmm, XMM_ERR_UNKNOWN, "(OSS) Mixer: Unable to write volume" );
  }

  close( fd );
  return 0;
}
