/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * vorbis.c
 * Ogg/Vorbis sound ( writer )
 *
 * TODO:
 */

#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpsound.h>
#include <libxmm/lpfiltera.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include <vorbis/vorbisenc.h>

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE	16384
#define	DEFAULT_BITRATE	128000

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;

    /* Ogg/Vorbis encoder stuff */
    ogg_stream_state		os;
    ogg_page			og;
    ogg_packet			op;
    vorbis_info			vi;
    vorbis_comment		vc;
    vorbis_dsp_state		vd;
    vorbis_block		vb;

    /* Buffer stuff */
    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Global data
 */

extern XMM_PluginSound	plugin_info;

/*
 * Initialize
 */
static XMM_PluginSound *vorbis_Init( void *xmm )
{
  XMM_PluginSound *sound;
  struct priv_t *priv;

  if(( sound = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginSound ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(Ogg/Vorbis) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &sound[1];
  sound->sys.priv = (void *) priv;
  sound->sys.xmm = xmm;

  return sound;
}

/*
 * Close
 */
static void vorbis_Close( XMM_PluginSound *sound )
{
  struct priv_t *priv = sound->sys.priv;
  int		ret;

  /* Stop audio chain */
  xmm_FilterAChainStop( sound->sys.xmm );

  /* Signalize end of stream */
  vorbis_analysis_wrote( &priv->vd, 0 );

  /* Write blocks */
  while( vorbis_analysis_blockout( &priv->vd, &priv->vb ) == 1 )
  {
	vorbis_analysis( &priv->vb, &priv->op );
	/* packet into bitstream */
	ogg_stream_packetin( &priv->os, &priv->op );

	do 
	{
	    ret = ogg_stream_pageout( &priv->os, &priv->og );
	    if( ret == 0 )	break;

	    priv->pIO->Write( priv->pIO, priv->og.header, 1, priv->og.header_len );
	    priv->pIO->Write( priv->pIO, priv->og.body, 1, priv->og.body_len );
	} while( !ogg_page_eos( &priv->og ));
  }

  /* Close encoder stuff */
  ogg_stream_clear( &priv->os );
  vorbis_block_clear( &priv->vb );
  vorbis_dsp_clear( &priv->vd );
  vorbis_info_clear( &priv->vi );

  priv->pIO->Close( priv->pIO );

  free( sound );
}

/*
 * Control function
 */
static int vorbis_Control( XMM_PluginSound *sound, uint32_t cmd, uint32_t arg, void *data )
{
  struct priv_t *priv = sound->sys.priv;

  switch( cmd )
  {
    case XMM_CTLQUERY_SFORMAT:
		if(((( XMM_SoundFormat *)data)->format & XMM_SOUND_MASK_SIZE ) != 16 )
		    return XMM_CTLRET_FALSE;

		return XMM_CTLRET_TRUE;

    case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;
  
    case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_SOUND )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No SOUND command */
}

/*
 * Start output
 */
static int vorbis_Start( XMM_PluginSound *sound, XMM_SoundFormat *sformat, int fsize, int fcount )
{
  struct priv_t *priv = sound->sys.priv;
  const char	*ofn;
  int		samprate = sformat->samprate, channels = sformat->channels;
  int		format = sformat->format;
  ogg_packet	header, header_comm, header_code;

  if(( ofn = xmm_AcOutputFilename( sound->sys.xmm, NULL, ".ogg" )) == NULL )
	return XMM_RET_ERROR;

  if(( priv->pIO = xmm_IOOpen( sound->sys.xmm, ofn, XMM_IO_WRITE )) == NULL )
	return XMM_RET_ERROR;

  /*
   * Init Ogg/Vorbis
   */

  /* Initialize encoding */
  vorbis_info_init( &priv->vi );
  vorbis_encode_init( &priv->vi, channels, samprate, -1, DEFAULT_BITRATE, -1 );

  /* Add encoder as comment line */
  vorbis_comment_init( &priv->vc );
  vorbis_comment_add( &priv->vc, "Encoded by "XMM_PROJECT_NAME" Ogg/Vorbis Plugin" );

  /* Init analyser */
  vorbis_analysis_init( &priv->vd, &priv->vi );
  vorbis_block_init( &priv->vd, &priv->vb );
  
  /* Init packet->stream encoder ( serial number is random ) */
  srand( time( NULL ));
  ogg_stream_init( &priv->os, rand());

  /* Write the three Ogg/Vorbis headers */
  vorbis_analysis_headerout( &priv->vd, &priv->vc, &header, &header_comm, &header_code );
  ogg_stream_packetin( &priv->os, &header);
  ogg_stream_packetin( &priv->os, &header_comm);
  ogg_stream_packetin( &priv->os, &header_code);

  while( 1 )
  {
	int result = ogg_stream_flush( &priv->os, &priv->og );
	if( result == 0 )	break;
	priv->pIO->Write( priv->pIO, priv->og.header, 1, priv->og.header_len );
	priv->pIO->Write( priv->pIO, priv->og.body, 1, priv->og.body_len );
  }

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  if(( format & XMM_SOUND_MASK_SIZE ) != 16 )
  {
	xmm_logging( 2, "Ogg/Vorbis! 8 bit not supported. Converting to 16 bit\n" );
	format = XMM_SOUND_FMT_S16LE;
  }

  xmm_logging( 2, "Ogg/Vorbis! Started ( %i Hz, %i channel(s), format = %x, bs = %i )\n",
					samprate, channels, format, fsize );

  /* Set source format */
  xmm_FilterAChainInput( sound->sys.xmm, sformat->samprate, sformat->channels, sformat->format );

  /* Set dest format */
  priv->bufferSize = xmm_FilterAChainOutput( sound->sys.xmm, samprate, channels, format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)( samprate * channels * ( format & XMM_SOUND_MASK_SIZE ));
  priv->bufferSSize = ( channels * ( format & XMM_SOUND_MASK_SIZE ) / 8 );

  /* return buffer size ( = number of samples ) */
  return priv->bufferSize;
}

/*
 * Write data
 */
static int vorbis_Write( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  struct priv_t *priv = sound->sys.priv;
  float		**buffer;
  int		proc, written = 0, conv, done, i, ret, ss, ch;
  signed char	*ptr;

  ss = priv->vi.channels * 2;	/* We need the sample size in the inner loop */

  /*
   * samples > bufferSize: We have to split data for xmm_FilterAChain()
   */

  while( samples > 0 )
  {
	proc = samples;
	if( proc > priv->bufferSize )	proc = priv->bufferSize;

	conv = proc;
	ptr = data;

	if( xmm_FilterAChainActive( sound->sys.xmm ))
	{
	    if(( ret = xmm_FilterAChain( sound->sys.xmm, data, proc, (uint8_t **)&ptr, &conv )) != XMM_RET_OK )
		return ret;
	}

	/* get buffer */
	buffer = vorbis_analysis_buffer( &priv->vd, conv * priv->bufferSSize );

	/* Convert to uninterleaved float */
	for( i = 0; i < conv; i++ )
	{
	    for( ch = 0; ch < priv->vi.channels; ch++ )
		buffer[ch][i] = (( ptr[i * ss + ch * 2 + 1] << 8 ) | ( 0x00ff & (int)ptr[i * ss + ch * 2] )) / 32768.00;
	}

	/* Write data */
	vorbis_analysis_wrote( &priv->vd, i );

	/* Write blocks */
	while( vorbis_analysis_blockout( &priv->vd, &priv->vb ) == 1 )
	{
	    vorbis_analysis( &priv->vb, &priv->op );
	    /* packet into bitstream */
	    ogg_stream_packetin( &priv->os, &priv->op );

	    do 
	    {
		ret = ogg_stream_pageout( &priv->os, &priv->og );
		if( ret == 0 )	break;

		done = priv->pIO->Write( priv->pIO, priv->og.header, 1, priv->og.header_len );
		done += priv->pIO->Write( priv->pIO, priv->og.body, 1, priv->og.body_len );
		written += done;
	    } while( !ogg_page_eos( &priv->og ));
	}

	data += ( proc * priv->bufferSSize );
	samples -= proc;
  }

  if( written == 0 )	return XMM_RET_ERROR;

  return written;
}

/*
 * Flush data
 */
static void vorbis_Flush( XMM_PluginSound *sound )
{
}

/*
 * Plugin info
 */

XMM_PluginSound	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_SOUND,
				0,
				XMM_VERSION_NUM,
				"",
				"Ogg/Vorbis",
				"Sound: Ogg/Vorbis",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				vorbis_Init, vorbis_Close, vorbis_Control,
				vorbis_Start, vorbis_Write, vorbis_Flush };
